/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.test.mdb.unit;

import java.util.Map;

import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.TextMessage;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.naming.Context;

import junit.framework.Test;
import junit.framework.TestCase;

import org.jboss.ejb3.statistics.InvocationStatistics;
import org.jboss.ejb3.test.mdb.QueueTestMDB;
import org.jboss.logging.Logger;
import org.jboss.test.JBossTestCase;

/**
 * @author <a href="mailto:bdecoste@jboss.com">William DeCoste</a>
 */
public class MetricsUnitTestCase extends JBossTestCase
{
   private static final Logger log = Logger.getLogger(MetricsUnitTestCase.class);

   /**
    * Name of the Queue Connection Factory in JNDI
    */
   private static final String JNDI_NAME_CONNECTION_FACTORY = "ConnectionFactory";

   public MetricsUnitTestCase(String name)
   {
      super(name);
   }

   public void testJmxMetrics() throws Exception
   {
      MBeanServerConnection server = getServer();

      int size = 0;

      ObjectName testerName = new ObjectName("jboss.j2ee:jar=mdb-test.jar,name=QueueMDB,service=EJB3");

      size = (Integer) server.getAttribute(testerName, "MinPoolSize");
      assertEquals(1, size);

      size = (Integer) server.getAttribute(testerName, "MaxPoolSize");
      assertEquals(1, size);

      size = (Integer) server.getAttribute(testerName, "MaxMessages");
      assertEquals(1, size);

      size = (Integer) server.getAttribute(testerName, "KeepAliveMillis");
      assertEquals(60000, size);

      // Get the invocation stats
      Map invokeStats = ((InvocationStatistics) server.getAttribute(testerName, "InvokeStats")).getStats();
      log.info("Stats before invocation: " + invokeStats);
      // Ensure stats are first empty
      TestCase.assertTrue("Stats should be empty before invoked", invokeStats.isEmpty());
      // Invoke by sending a message to the Queue
      this.sendTextMessageToQueue("test", QueueTestMDB.NAME_QUEUE);
      // Wait
      Thread.sleep(2000L);
      // Get stats again
      invokeStats = ((InvocationStatistics) server.getAttribute(testerName, "InvokeStats")).getStats();
      log.info("Stats after invocation: " + invokeStats);
      // Check content
      TestCase.assertEquals("Stats should have been invoked once", 1, invokeStats.size());

      testerName = new ObjectName("jboss.j2ee:jar=mdb-test.jar,name=TransactionQueueTestMDB,service=EJB3");

      size = (Integer) server.getAttribute(testerName, "MaxPoolSize");
      assertEquals(1, size);

      testerName = new ObjectName("jboss.j2ee:jar=mdb-test.jar,name=DefaultedQueueTestMDB,service=EJB3");

      size = (Integer) server.getAttribute(testerName, "MaxPoolSize");
      assertEquals(30, size);

   }

   public static Test suite() throws Exception
   {
      return getDeploySetup(MetricsUnitTestCase.class, "mdbtest-${jboss.jms.provider}-service.xml, mdb-test.jar");
   }

   /**
    * Sends a JMS {@link TextMessage} containing the specified contents to the 
    * queue of the specified name  
    * 
    * @param contents
    * @param queueName
    * @throws Exception
    * @throws IllegalArgumentException If either argument is not provided
    */
   private void sendTextMessageToQueue(final String contents, final String queueName) throws Exception,
         IllegalArgumentException
   {
      // Precondition check
      if (contents == null || contents.length() == 0)
      {
         throw new IllegalArgumentException("contents must be provided");
      }
      if (queueName == null || queueName.length() == 0)
      {
         throw new IllegalArgumentException("queueName must be provided");
      }

      // Get the queue from JNDI
      final Context context = this.getInitialContext();
      final Queue queue = (Queue) context.lookup(queueName);

      // Get the ConnectionFactory from JNDI
      final QueueConnectionFactory factory = (QueueConnectionFactory) context.lookup(JNDI_NAME_CONNECTION_FACTORY);

      // Make a Connection
      final QueueConnection connection = factory.createQueueConnection();
      final QueueSession sendSession = connection.createQueueSession(false, QueueSession.AUTO_ACKNOWLEDGE);

      // Make the message
      final TextMessage message = sendSession.createTextMessage(contents);

      // Send the message
      final QueueSender sender = sendSession.createSender(queue);
      sender.send(message);
      log.info("Sent message " + message + " with contents: " + contents);

      // Clean up
      sendSession.close();
      connection.close();
   }

}
