/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2015, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.as.test.manualmode.weld.extension;

import junit.framework.AssertionFailedError;
import org.jboss.arquillian.container.test.api.ContainerController;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.container.test.api.TargetsContainer;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.EmptyAsset;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.EnterpriseArchive;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.AfterClass;
import org.junit.Test;
import org.junit.runner.RunWith;

import javax.enterprise.inject.spi.Extension;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.ArrayList;
import java.util.List;

import static org.junit.Assert.assertEquals;

/**
 * A test to ensure that the UserTransaction and TransactionSynchronizationRegistry can be retrieved via JNDI when
 * an extensions BeforeShutdown method is invoked.
 * <p/>
 * See WFLY-5232
 *
 * @author Ryan Emerson
 */
@RunWith(Arquillian.class)
@RunAsClient
public class BeforeShutdownJNDILookupTestCase {

    public static final String TEST_URL = "target" + File.separator + "results.txt";
    private static final String CONTAINER = "default-jbossas";
    private static final String DEPLOYMENT = "test.ear";

    @Deployment(name = DEPLOYMENT, managed = true)
    @TargetsContainer(CONTAINER)
    public static Archive<?> deploy() {
        EnterpriseArchive ear = ShrinkWrap.create(EnterpriseArchive.class, DEPLOYMENT);
        WebArchive war = ShrinkWrap.create(WebArchive.class, "testWar.war");
        JavaArchive warLib = ShrinkWrap.create(JavaArchive.class, "testLib.jar");
        warLib.addClasses(BeforeShutdownExtension.class);
        warLib.add(new StringAsset(BeforeShutdownExtension.class.getName()), "META-INF/services/" + Extension.class.getName());
        war.addAsLibrary(warLib);
        war.add(EmptyAsset.INSTANCE, "WEB-INF/beans.xml");
        JavaArchive lib = ShrinkWrap.create(JavaArchive.class, "lib.jar");
        lib.addClasses(BeforeShutdownJNDILookupTestCase.class);
        ear.addAsLibrary(lib);
        ear.addAsModule(war);
        return ear;
    }

    @ArquillianResource
    ContainerController controller;

    @Test
    public void testTransactionJNDILookupDuringShutdownEvent() throws Exception {
        controller.start(CONTAINER);
        controller.kill(CONTAINER);

        List<String> output = readAllLinesInFile(TEST_URL);
        if (output.get(0).equals("Exception")) {
            String stacktrace = output.get(1).replaceAll(",", System.getProperty("line.separator"));
            String msg = "An exception was thrown by the deployment %s during shutdown.  The server stacktrace is shown below: %n%s";
            throw new AssertionFailedError(String.format(msg, DEPLOYMENT, stacktrace));
        }
        assertEquals("Contents of result.txt is not valid!", "UserTransaction", output.get(0));
    }

    @AfterClass
    public static void cleanup() throws Exception {
        new File(TEST_URL).delete();
    }

    private List<String> readAllLinesInFile(String filePath) throws Exception {
        List<String> output = new ArrayList<String>();
        BufferedReader br = new BufferedReader(new FileReader(filePath));
        try {
            String line;
            while ((line = br.readLine()) != null) {
                output.add(line);
            }
            return output;
        } finally {
            br.close();
        }
    }
}
