/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2015, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.wildfly.clustering.web.hotrod.session;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

import org.infinispan.client.hotrod.Flag;
import org.infinispan.client.hotrod.MetadataValue;
import org.infinispan.client.hotrod.RemoteCache;
import org.wildfly.clustering.ee.Mutator;
import org.wildfly.clustering.ee.MutatorFactory;
import org.wildfly.clustering.ee.cache.CacheProperties;
import org.wildfly.clustering.ee.hotrod.RemoteCacheEntryMutator;
import org.wildfly.clustering.ee.hotrod.RemoteCacheMutatorFactory;
import org.wildfly.clustering.infinispan.client.Key;
import org.wildfly.clustering.web.cache.session.CompositeSessionMetaData;
import org.wildfly.clustering.web.cache.session.CompositeSessionMetaDataEntry;
import org.wildfly.clustering.web.cache.session.InvalidatableSessionMetaData;
import org.wildfly.clustering.web.cache.session.MutableSessionAccessMetaData;
import org.wildfly.clustering.web.cache.session.MutableSessionCreationMetaData;
import org.wildfly.clustering.web.cache.session.SessionAccessMetaData;
import org.wildfly.clustering.web.cache.session.SessionCreationMetaData;
import org.wildfly.clustering.web.cache.session.SessionCreationMetaDataEntry;
import org.wildfly.clustering.web.cache.session.SessionMetaDataFactory;
import org.wildfly.clustering.web.cache.session.SimpleSessionAccessMetaData;
import org.wildfly.clustering.web.cache.session.SimpleSessionCreationMetaData;
import org.wildfly.clustering.web.session.ImmutableSessionMetaData;

/**
 * @author Paul Ferraro
 */
public class HotRodSessionMetaDataFactory<L> implements SessionMetaDataFactory<CompositeSessionMetaDataEntry<L>> {

    private final RemoteCache<Key<String>, Object> cache;
    private final RemoteCache<SessionCreationMetaDataKey, SessionCreationMetaDataEntry<L>> creationMetaDataCache;
    private final MutatorFactory<SessionCreationMetaDataKey, SessionCreationMetaDataEntry<L>> creationMetaDataMutatorFactory;
    private final RemoteCache<SessionAccessMetaDataKey, SessionAccessMetaData> accessMetaDataCache;
    private final CacheProperties properties;
    private final Flag[] ignoreReturnFlags;
    private final Flag[] forceReturnFlags;

    public HotRodSessionMetaDataFactory(HotRodSessionMetaDataFactoryConfiguration configuration) {
        this.cache = configuration.getCache();
        // N.B. HotRod client flags are thread local
        this.ignoreReturnFlags = configuration.getIgnoreReturnFlags();
        this.forceReturnFlags = configuration.getForceReturnFlags();
        this.creationMetaDataCache = configuration.getCache();
        this.creationMetaDataMutatorFactory = new RemoteCacheMutatorFactory<>(this.creationMetaDataCache, this.ignoreReturnFlags);
        this.accessMetaDataCache = configuration.getCache();
        this.properties = configuration.getCacheProperties();
    }

    @Override
    public CompositeSessionMetaDataEntry<L> createValue(String id, Void context) {
        SessionCreationMetaDataEntry<L> creationMetaDataEntry = new SessionCreationMetaDataEntry<>(new SimpleSessionCreationMetaData());
        SessionAccessMetaData accessMetaData = new SimpleSessionAccessMetaData();
        Map<Key<String>, Object> entries = new HashMap<>(3);
        entries.put(new SessionCreationMetaDataKey(id), creationMetaDataEntry);
        entries.put(new SessionAccessMetaDataKey(id), accessMetaData);
        this.cache.withFlags(this.ignoreReturnFlags).putAll(entries);
        return new CompositeSessionMetaDataEntry<>(creationMetaDataEntry, accessMetaData);
    }

    @Override
    public CompositeSessionMetaDataEntry<L> findValue(String id) {
        SessionCreationMetaDataKey key = new SessionCreationMetaDataKey(id);
        MetadataValue<SessionCreationMetaDataEntry<L>> versionedCreationMetaDataEntry = this.creationMetaDataCache.getWithMetadata(key);
        if (versionedCreationMetaDataEntry != null) {
            SessionCreationMetaDataEntry<L> creationMetaDataEntry = versionedCreationMetaDataEntry.getValue();
            SessionAccessMetaData accessMetaData = this.accessMetaDataCache.get(new SessionAccessMetaDataKey(id));
            if (accessMetaData != null) {
                return new CompositeSessionMetaDataEntry<>(creationMetaDataEntry, accessMetaData);
            }
            this.creationMetaDataCache.withFlags(this.ignoreReturnFlags).removeWithVersion(key, versionedCreationMetaDataEntry.getVersion());
        }
        return null;
    }

    @Override
    public InvalidatableSessionMetaData createSessionMetaData(String id, CompositeSessionMetaDataEntry<L> entry) {
        boolean newSession = entry.getCreationMetaData().isNew();
        boolean requireMutator = !this.properties.isTransactional() || !newSession;

        SessionCreationMetaData creationMetaData = entry.getCreationMetaData();
        if (requireMutator) {
            SessionCreationMetaDataKey creationMetaDataKey = new SessionCreationMetaDataKey(id);
            SessionCreationMetaDataEntry<L> creationMetaDataEntry = new SessionCreationMetaDataEntry<>(creationMetaData, entry.getLocalContext());
            Mutator mutator = this.creationMetaDataMutatorFactory.createMutator(creationMetaDataKey, creationMetaDataEntry);
            creationMetaData = new MutableSessionCreationMetaData(creationMetaData, mutator);
        }

        SessionAccessMetaData accessMetaData = entry.getAccessMetaData();
        if (requireMutator) {
            SessionAccessMetaDataKey accessMetaDataKey = new SessionAccessMetaDataKey(id);
            Mutator accessMutator = this.createSessionAccessMetaDataMutator(accessMetaDataKey, accessMetaData, creationMetaData);
            accessMetaData = new MutableSessionAccessMetaData(entry.getAccessMetaData(), accessMutator);
        }

        return new CompositeSessionMetaData(creationMetaData, accessMetaData);
    }

    @Override
    public ImmutableSessionMetaData createImmutableSessionMetaData(String id, CompositeSessionMetaDataEntry<L> entry) {
        return new CompositeSessionMetaData(entry.getCreationMetaData(), entry.getAccessMetaData());
    }

    @Override
    public boolean remove(String id) {
        SessionCreationMetaDataKey key = new SessionCreationMetaDataKey(id);
        SessionCreationMetaDataEntry<L> creationMetaData = this.creationMetaDataCache.withFlags(this.forceReturnFlags).remove(key);
        if (creationMetaData == null) return false;
        this.accessMetaDataCache.withFlags(this.ignoreReturnFlags).remove(new SessionAccessMetaDataKey(id));
        return true;
    }

    private Mutator createSessionAccessMetaDataMutator(SessionAccessMetaDataKey key, SessionAccessMetaData value, SessionCreationMetaData creationMetaData) {
        return new RemoteCacheEntryMutator<>(this.accessMetaDataCache, this.ignoreReturnFlags, key, value, new Function<SessionAccessMetaData, Duration>() {
            @Override
            public Duration apply(SessionAccessMetaData accessMetaData) {
                return creationMetaData.getMaxInactiveInterval();
            }
        });
    }
}
