/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2015, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.wildfly.clustering.web.hotrod.session.fine;

import java.io.IOException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import org.infinispan.client.hotrod.Flag;
import org.infinispan.client.hotrod.RemoteCache;
import org.wildfly.clustering.ee.Immutability;
import org.wildfly.clustering.ee.Mutator;
import org.wildfly.clustering.ee.MutatorFactory;
import org.wildfly.clustering.ee.cache.CacheProperties;
import org.wildfly.clustering.ee.cache.function.MapComputeFunction;
import org.wildfly.clustering.ee.hotrod.RemoteCacheComputeMutatorFactory;
import org.wildfly.clustering.marshalling.spi.Marshaller;
import org.wildfly.clustering.web.cache.session.CompositeImmutableSession;
import org.wildfly.clustering.web.cache.session.ImmutableSessionAttributeActivationNotifier;
import org.wildfly.clustering.web.cache.session.SessionAttributeActivationNotifier;
import org.wildfly.clustering.web.cache.session.SessionAttributes;
import org.wildfly.clustering.web.cache.session.SessionAttributesFactory;
import org.wildfly.clustering.web.cache.session.fine.FineImmutableSessionAttributes;
import org.wildfly.clustering.web.cache.session.fine.FineSessionAttributes;
import org.wildfly.clustering.web.hotrod.logging.Logger;
import org.wildfly.clustering.web.hotrod.session.HotRodSessionAttributesFactoryConfiguration;
import org.wildfly.clustering.web.session.HttpSessionActivationListenerProvider;
import org.wildfly.clustering.web.session.ImmutableSessionAttributes;
import org.wildfly.clustering.web.session.ImmutableSessionMetaData;

/**
 * {@link SessionAttributesFactory} for fine granularity sessions.
 * A given session's attributes are mapped to N+1 co-located cache entries, where N is the number of session attributes.
 * A separate cache entry stores the activate attribute names for the session.
 * @author Paul Ferraro
 */
public class FineSessionAttributesFactory<S, C, L, V> implements SessionAttributesFactory<C, Map.Entry<Map<String, UUID>, Map<UUID, Object>>> {

    private final RemoteCache<SessionAttributeNamesKey, Map<String, UUID>> namesCache;
    private final RemoteCache<SessionAttributeKey, V> attributeCache;
    private final MutatorFactory<SessionAttributeNamesKey, Map.Entry<Map<String, UUID>, Map<UUID, Object>>> mutatorFactory;
    private final Marshaller<Object, V> marshaller;
    private final Immutability immutability;
    private final CacheProperties properties;
    private final HttpSessionActivationListenerProvider<S, C, L> provider;
    private final Flag[] ignoreReturnFlags;
    private final Flag[] forceReturnFlags;

    public FineSessionAttributesFactory(HotRodSessionAttributesFactoryConfiguration<S, C, L, Object, V> configuration) {
        this.namesCache = configuration.getCache();
        // N.B. HotRod client flags are thread local
        this.ignoreReturnFlags = configuration.getIgnoreReturnFlags();
        this.forceReturnFlags = configuration.getForceReturnFlags();
        this.attributeCache = configuration.getCache();
        this.marshaller = configuration.getMarshaller();
        this.immutability = configuration.getImmutability();
        this.properties = configuration.getCacheProperties();
        this.provider = configuration.getHttpSessionActivationListenerProvider();
        MutatorFactory<SessionAttributeNamesKey, Map<String, UUID>> namesMutatorFactory = new RemoteCacheComputeMutatorFactory<>(this.namesCache, this.ignoreReturnFlags, MapComputeFunction::new);
        this.mutatorFactory = new MutatorFactory<SessionAttributeNamesKey, Map.Entry<Map<String, UUID>, Map<UUID, Object>>>() {
            @Override
            public Mutator createMutator(SessionAttributeNamesKey namesKey, Map.Entry<Map<String, UUID>, Map<UUID, Object>> entry) {
                String id = namesKey.getId();
                Map<String, UUID> names = entry.getKey();
                Map<UUID, Object> attributes = entry.getValue();
                Map<SessionAttributeKey, V> updates = new HashMap<>(attributes.size());
                List<SessionAttributeKey> removes = new ArrayList<>(attributes.size());
                RemoteCache<SessionAttributeKey, V> cache = configuration.getCache();
                Flag[] ignoreReturnFlags = configuration.getIgnoreReturnFlags();
                Marshaller<Object, V> marshaller = configuration.getMarshaller();
                try {
                    for (Map.Entry<UUID, Object> attribute : attributes.entrySet()) {
                        SessionAttributeKey key = new SessionAttributeKey(id, attribute.getKey());
                        Object value = attribute.getValue();
                        if (value != null) {
                            updates.put(key, marshaller.write(value));
                        } else {
                            removes.add(key);
                        }
                    }
                } catch (IOException e) {
                    throw new IllegalStateException(e);
                }
                return new Mutator() {
                    @Override
                    public void mutate() {
                        if (!updates.isEmpty()) {
                            cache.withFlags(ignoreReturnFlags).putAll(updates);
                        }
                        if (!names.isEmpty()) {
                            namesMutatorFactory.createMutator(namesKey, names).mutate();
                        }
                        for (SessionAttributeKey key : removes) {
                            cache.withFlags(ignoreReturnFlags).remove(key);
                        }
                    }
                };
            }
        };
    }

    @Override
    public Map.Entry<Map<String, UUID>, Map<UUID, Object>> createValue(String id, Void context) {
        return new AbstractMap.SimpleImmutableEntry<>(new ConcurrentHashMap<>(), new ConcurrentHashMap<>());
    }

    @Override
    public Map.Entry<Map<String, UUID>, Map<UUID, Object>> findValue(String id) {
        return this.getValue(id, true);
    }

    @Override
    public Map.Entry<Map<String, UUID>, Map<UUID, Object>> tryValue(String id) {
        return this.getValue(id, false);
    }

    private Map.Entry<Map<String, UUID>, Map<UUID, Object>> getValue(String id, boolean purgeIfInvalid) {
        Map<String, UUID> names = this.namesCache.get(new SessionAttributeNamesKey(id));
        if (names == null) {
            return this.createValue(id, null);
        }
        // Read attribute entries via bulk read
        Map<SessionAttributeKey, String> keys = new HashMap<>();
        for (Map.Entry<String, UUID> entry : names.entrySet()) {
            keys.put(new SessionAttributeKey(id, entry.getValue()), entry.getKey());
        }
        Map<SessionAttributeKey, V> values = this.attributeCache.getAll(keys.keySet());
        // Validate attributes
        Map<UUID, Object> attributes = new ConcurrentHashMap<>();
        for (Map.Entry<SessionAttributeKey, String> entry : keys.entrySet()) {
            SessionAttributeKey key = entry.getKey();
            V value = values.get(entry.getKey());
            if (value != null) {
                try {
                    attributes.put(key.getAttributeId(), this.marshaller.read(value));
                    continue;
                } catch (IOException e) {
                    Logger.ROOT_LOGGER.failedToActivateSessionAttribute(e, id, entry.getValue());
                }
            } else {
                Logger.ROOT_LOGGER.missingSessionAttributeCacheEntry(id, entry.getValue());
            }
            if (purgeIfInvalid) {
                this.purge(id);
            }
            return null;
        }
        return new AbstractMap.SimpleImmutableEntry<>(new ConcurrentHashMap<>(names), attributes);
    }

    @Override
    public boolean remove(String id) {
        Map<String, UUID> names = this.namesCache.withFlags(this.forceReturnFlags).remove(new SessionAttributeNamesKey(id));
        if (names != null) {
            for (UUID attributeId : names.values()) {
                this.attributeCache.withFlags(this.ignoreReturnFlags).remove(new SessionAttributeKey(id, attributeId));
            }
        }
        return true;
    }

    @Override
    public SessionAttributes createSessionAttributes(String id, Map.Entry<Map<String, UUID>, Map<UUID, Object>> entry, ImmutableSessionMetaData metaData, C context) {
        SessionAttributeActivationNotifier notifier = new ImmutableSessionAttributeActivationNotifier<>(this.provider, new CompositeImmutableSession(id, metaData, this.createImmutableSessionAttributes(id, entry)), context);
        return new FineSessionAttributes<>(new SessionAttributeNamesKey(id), entry.getKey(), entry.getValue(), this.mutatorFactory, this.marshaller, this.immutability, this.properties, notifier);
    }

    @Override
    public ImmutableSessionAttributes createImmutableSessionAttributes(String id, Map.Entry<Map<String, UUID>, Map<UUID, Object>> entry) {
        return new FineImmutableSessionAttributes(entry.getKey(), entry.getValue());
    }
}
