/*
 * Copyright 2023 JBoss by Red Hat.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.eap.insights.report;

import static com.redhat.insights.InsightsErrorCode.ERROR_CLIENT_FAILED;
import com.redhat.insights.InsightsException;
import com.redhat.insights.InsightsReport;
import static com.redhat.insights.http.InsightsHttpClient.GENERAL_MIME_TYPE;

import com.redhat.insights.config.InsightsConfiguration;
import com.redhat.insights.http.InsightsHttpClient;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.function.Supplier;
import javax.net.ssl.SSLContext;
import org.apache.http.HttpHost;
import org.apache.http.ParseException;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.DefaultProxyRoutePlanner;
import org.apache.http.util.EntityUtils;
import org.jboss.eap.insights.report.logging.InsightsReportLogger;

/**
 * @author Emmanuel Hugonnet (c) 2023 Red Hat, Inc.
 */
class InsightsApacheHttpClient implements InsightsHttpClient {

    private static final ContentType GENERAL_CONTENT_TYPE = ContentType.create(GENERAL_MIME_TYPE);
    private final Supplier<SSLContext> sslContextSupplier;
    private final InsightsConfiguration configuration;

    InsightsApacheHttpClient(InsightsConfiguration configuration, Supplier<SSLContext> sslContextSupplier) {
        this.configuration = configuration;
        this.sslContextSupplier = sslContextSupplier;
    }

    @Override
    public void decorate(InsightsReport report) {
        if (configuration.useMTLS()) {
            report.decorate("app.transport.type.https", "mtls");
            // We can't send anything more useful (e.g. SHA hash of cert file) as until
            // we try to send, we don't know if we can read the file at this path
            report.decorate("app.transport.cert.https", configuration.getCertFilePath());
        } else {
            final String authToken = configuration.getMaybeAuthToken().get();
            report.decorate("app.transport.type.https", "token");
            report.decorate("app.auth.token", authToken);
        }
    }

    @Override
    public void sendInsightsReport(String reportName, InsightsReport report) {
        decorate(report);
        String json = report.serialize();
        InsightsReportLogger.ROOT_LOGGER.debug("Red Hat Insights Report:\n" + json);
        sendCompressedInsightsReport(reportName + ".gz", InsightsHttpClient.gzipReport(json));
    }

    void sendCompressedInsightsReport(String filename, byte[] bytes) {
        HttpClientBuilder clientBuilder = HttpClientBuilder.create();
        if (configuration.getProxyConfiguration().isPresent()) {
            InsightsConfiguration.ProxyConfiguration conf = configuration.getProxyConfiguration().get();
            clientBuilder.setRoutePlanner(new DefaultProxyRoutePlanner(new HttpHost(conf.getHost(), conf.getPort(), "http")));
        }
        clientBuilder.setRetryHandler(new DefaultHttpRequestRetryHandler(configuration.getHttpClientRetryMaxAttempts(), true));
        if (configuration.useMTLS()) {
            if (sslContextSupplier.get() == null) {
                return;
            }
            clientBuilder.setSSLContext(sslContextSupplier.get());
        } else {
            clientBuilder.setSSLHostnameVerifier(NoopHostnameVerifier.INSTANCE);
        }
        CloseableHttpClient client = clientBuilder.build();
        try {
            HttpPost post;
            if (configuration.useMTLS()) {
                post = createPost();
            } else {
                post = createAuthTokenPost(configuration.getMaybeAuthToken().get());
            }
            post.setHeader("Cache-Control", "no-store");
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
            builder.addBinaryBody("file", bytes, GENERAL_CONTENT_TYPE, filename);
            builder.addTextBody("type", GENERAL_MIME_TYPE);
            post.setEntity(builder.build());
            try (CloseableHttpResponse response = client.execute(post)) {
                InsightsReportLogger.ROOT_LOGGER.debug(
                        "Red Hat Insights HTTP Client: status="
                        + response.getStatusLine()
                        + ", body="
                        + EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8));
                switch (response.getStatusLine().getStatusCode()) {
                    case 200:
                        InsightsReportLogger.ROOT_LOGGER.debug(
                                "Red Hat Insights - Payload was processed - "
                                + "This is unexpected as the request should "
                                + "take some time to be processed and a 201 / 202 is expected instead.");
                        break;
                    case 201:
                        InsightsReportLogger.ROOT_LOGGER.debug(
                                "Red Hat Insights - Advisor content type with no metadata accepted for"
                                + " processing");
                        break;
                    case 202:
                        InsightsReportLogger.ROOT_LOGGER.debug("Red Hat Insights - Payload was accepted for processing");
                        break;
                    case 401:
                        throw InsightsReportLogger.ROOT_LOGGER.missingAuthentication(response.getStatusLine().getReasonPhrase());
                    case 413:
                        throw InsightsReportLogger.ROOT_LOGGER.payloadTooLarge(response.getStatusLine().getReasonPhrase());
                    case 415:
                        throw InsightsReportLogger.ROOT_LOGGER.unsupportedContentType(response.getStatusLine().getReasonPhrase());
                    case 500:
                    case 503:
                    default:
                        throw InsightsReportLogger.ROOT_LOGGER.serversideError(response.getStatusLine().toString());
                }
            }
        } catch (IOException | ParseException ioex) {
            InsightsReportLogger.ROOT_LOGGER.debug("Error", ioex);
            throw new InsightsException(ERROR_CLIENT_FAILED, ioex);
        } finally {
            try {
                client.close();
            } catch (IOException ex) {
                InsightsReportLogger.ROOT_LOGGER.debug("Error", ex);
            }
        }
    }

    private HttpPost createAuthTokenPost(String token) {
        HttpPost post = new HttpPost(assembleURI(configuration.getUploadBaseURL(), configuration.getUploadUri()));
        post.setHeader("Authorization", "Bearer " + token);
        return post;
    }

    private HttpPost createPost() {
        return new HttpPost(assembleURI(configuration.getUploadBaseURL(), configuration.getUploadUri()));
    }

    URI assembleURI(String url, String path) {
        String fullURL;
        if (!url.endsWith("/") && !path.startsWith("/")) {
            fullURL = url + "/" + path;
        } else if (url.endsWith("/") && path.startsWith("/")) {
            fullURL = url + path.substring(1);
        } else {
            fullURL = url + path;
        }
        return URI.create(fullURL);
    }

    @Override
    public boolean isReadyToSend() {
        return !configuration.useMTLS()
                || (new File(configuration.getCertFilePath()).exists()
                && new File(configuration.getKeyFilePath()).exists()
                && new File(configuration.getMachineIdFilePath()).exists());
    }

    @Override
    public String toString() {
        if (configuration.useMTLS()) {
            return "InsightsApacheHttpClient{"
                    + "keyFile= " + configuration.getKeyFilePath()
                    + ", certFile= " + configuration.getCertFilePath()
                    + ", url= " + assembleURI(configuration.getUploadBaseURL(), configuration.getUploadUri())
                    + '}';
        }
        return "InsightsApacheHttpClient{"
                + "token= " + configuration.getMaybeAuthToken().get()
                + ", url= " + assembleURI(configuration.getUploadBaseURL(), configuration.getUploadUri())
                + '}';
    }
}
