/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.mavenlocalrepository;

import org.apache.commons.text.StringSubstitutor;
import org.jboss.installer.actions.impl.MavenLocalRepositoryHelper;
import org.junit.Assert;
import org.junit.Test;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;

@SuppressWarnings("OptionalGetWithoutIsPresent")
public class MavenLocalRepositoryTest {

    public static final String USER_SPECIFIC_LOCAL_REPOSITORY = "${user.home}" + File.separator + ".m2" + File.separator + "repository2";
    public static final String EXPECTED_LOCAL_REPOSITORY = StringSubstitutor.replaceSystemProperties(USER_SPECIFIC_LOCAL_REPOSITORY);

    @Test
    public void loadMavenLocalRepositoryTest() {
        Optional<Path> localMavenRepository = MavenLocalRepositoryHelper.loadLocalRepositoryFromSettings(Paths.get("src", "test", "resources", "user-specific-settings.xml"));
        Assert.assertEquals("local repositories don't match", EXPECTED_LOCAL_REPOSITORY, localMavenRepository.get().toString());
    }

    @Test
    public void getMavenHomeTest() {
        // since this is a maven project, it should get a non-empty maven home value.
        Optional<String> mavenHome = MavenLocalRepositoryHelper.getMavenHome();
        Assert.assertTrue("can't get maven home ", mavenHome.isPresent());
    }

    @Test
    public void determineLocalRepositoryFromSettingsTest() throws IOException {
        Optional<Path> localRepository = MavenLocalRepositoryHelper.loadLocalRepositoryFromSettings(Paths.get("src", "test", "resources", "user-specific-settings.xml"));
        Assert.assertTrue("local repository can't be found in user-specific-settings.xml", localRepository.isPresent());
        Assert.assertEquals("local repositories don't match", EXPECTED_LOCAL_REPOSITORY, localRepository.get().toString());
    }

    @Test
    public void determineLocalRepositoryFromSettingsWithEmptyLocalRepoTest() throws IOException {
        Optional<Path> localRepository = MavenLocalRepositoryHelper.loadLocalRepositoryFromSettings(Paths.get("src", "test", "resources", "empty-local-repo-settings.xml"));
        Assert.assertTrue("local repository is not empty in empty-user-specific-settings.xml", localRepository.isEmpty());
    }

    @Test
    public void determineLocalRepositoryFromNonexistentSettingsTest() throws IOException {
        Optional<Path> localRepository = MavenLocalRepositoryHelper.loadLocalRepositoryFromSettings(Paths.get("src", "test", "resources", "nonexistent-user-specific-settings.xml"));
        Assert.assertFalse("local repository shouldn't present in nonexistent-user-specific-settings.xml", localRepository.isPresent());
    }
}
