/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.panels.frame;

import org.jboss.installer.core.Screen;
import org.jboss.installer.screens.DefaultScreen;
import org.junit.Test;

import javax.swing.JPanel;

import static org.junit.Assert.assertEquals;

public class IndexGeneratorTest {

    private IndexGenerator indexGenerator = new IndexGenerator();

    @Test
    public void firstScreenGetsIndex1() {
        Screen firstScreen = testScreen();
        assertEquals("1", nextScreen(firstScreen));
    }

    @Test
    public void secondScreenGetsIndex2() {
        Screen firstScreen = testScreen();
        Screen secondScreen = testScreen();

        nextScreen(firstScreen);

        assertEquals("2", nextScreen(secondScreen));
    }

    @Test
    public void screenWithParentGetsSubIndex() {
        Screen firstScreen = testScreen();
        Screen firstChild = testScreen(firstScreen);

        nextScreen(firstScreen);
        String index = nextScreen(firstChild);

        assertEquals("1.1", index);
    }

    @Test
    public void secondScreenWithParentGetsIncrementedSubIndex() {
        Screen firstScreen = testScreen();
        Screen firstChild = testScreen(firstScreen);
        Screen secondChild = testScreen(firstScreen);

        nextScreen(firstScreen);
        nextScreen(firstChild);
        String index = nextScreen(secondChild);

        assertEquals("1.2", index);
    }

    @Test
    public void screenAfterChildGoesBackToSingleIndex() {
        Screen firstScreen = testScreen();
        Screen firstChild = testScreen(firstScreen);
        Screen secondScreen = testScreen();

        nextScreen(firstScreen);
        nextScreen(firstChild);
        String index = nextScreen(secondScreen);

        assertEquals("2", index);
    }

    @Test
    public void nestedChildGoesIntoSubSubIndex() {
        Screen firstScreen = testScreen();
        Screen firstChild = testScreen(firstScreen);
        Screen firstSubChild = testScreen(firstChild);

        nextScreen(firstScreen);
        nextScreen(firstChild);
        String index = nextScreen(firstSubChild);

        assertEquals("1.1.1", index);
    }

    @Test
    public void childAfterNestedChildGoesBackOneLevel() {
        Screen firstScreen = testScreen();
        Screen firstChild = testScreen(firstScreen);
        Screen firstSubChild = testScreen(firstChild);
        Screen secondChild = testScreen(firstScreen);

        nextScreen(firstScreen);
        nextScreen(firstChild);
        nextScreen(firstSubChild);
        String index = nextScreen(secondChild);

        assertEquals("1.2", index);
    }

    @Test
    public void nextScreenAfterNestedChildGoesBackToTopLevel() {
        Screen firstScreen = testScreen();
        Screen firstChild = testScreen(firstScreen);
        Screen firstSubChild = testScreen(firstChild);
        Screen secondScreen = testScreen();

        nextScreen(firstScreen);
        nextScreen(firstChild);
        nextScreen(firstSubChild);
        String index = nextScreen(secondScreen);

        assertEquals("2", index);
    }

    @Test
    public void twoNestedChildren() {
        Screen firstScreen = testScreen();
        Screen firstChild = testScreen(firstScreen);
        Screen firstSubChild = testScreen(firstChild);
        Screen secondChild = testScreen(firstScreen);
        Screen secondSubChild = testScreen(secondChild);

        nextScreen(firstScreen);
        nextScreen(firstChild);
        nextScreen(firstSubChild);
        nextScreen(secondChild);
        String index = nextScreen(secondSubChild);

        assertEquals("1.2.1", index);
    }

    private String nextScreen(Screen screen) {
        return indexGenerator.nextScreen(screen);
    }

    private DefaultScreen testScreen() {
        return testScreen(null);
    }

    private DefaultScreen testScreen(Screen parent) {
        return new DefaultScreen(parent,null, true) {

            @Override
            public String getTitle() {
                return null;
            }

            @Override
            public String getName() {
                return null;
            }

            @Override
            public JPanel getContent() {
                return null;
            }
        };
    }

}
