/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.server;

import org.jboss.as.controller.PathAddress;
import org.jboss.dmr.ModelNode;
import org.junit.Test;

import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;

import static org.jboss.as.controller.operations.common.Util.createEmptyOperation;
import static org.jboss.installer.test.utils.TestServer.TARGET_PATH;
import static org.junit.Assert.assertEquals;

public class DomainServerTest extends AbstractEmbeddedServerTest {

    @Override
    protected String getConfig() {
        return "host.xml";
    }

    @Override
    protected void assertServerMode(ModelNode node) {
        assertEquals("EMBEDDED_HOST_CONTROLLER", node.get(0).get("result").get(0).get("summary").get("standalone-or-domain-identifier").asString());
    }

    @Override
    protected EmbeddedServer getServer(Path targetPath) {
        return new DomainServer(targetPath);
    }

    @Override
    protected List<String> getExpectedConfigurations() {
        return Arrays.asList("host.xml", DomainServer.HOST_SECONDARY_XML, DomainServer.HOST_PRIMARY_XML);
    }

    @Test
    public void executeOnProfilesAppliesOperationToAllProfiles() throws Exception {
        DomainServer server = (DomainServer) getServer(TARGET_PATH);

        try {
            server.start(getConfig());

            final ModelNode setLoggerOp = createEmptyOperation("write-attribute", PathAddress.pathAddress("subsystem", "logging").append("root-logger", "ROOT"));
            setLoggerOp.get("name").set("level");
            setLoggerOp.get("value").set("WARN");

            server.executeOnProfiles(setLoggerOp, "");

            assertLogLevelChanged(server, "default");
            assertLogLevelChanged(server, "ha");
            assertLogLevelChanged(server, "full");
            assertLogLevelChanged(server, "full-ha");
        } finally {
            server.shutdown();
            server.close();
        }
    }

    private void assertLogLevelChanged(DomainServer server, String profileName) throws Exception {
        final ModelNode query = createEmptyOperation("read-attribute", PathAddress.pathAddress("profile", profileName)
                .append("subsystem", "logging").append("root-logger", "ROOT"));
        query.get("name").set("level");
        assertEquals("WARN", server.execute(query, "").asString());
    }
}
