/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.postinstall.task.secdom.DatabaseSecurity;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Before;
import org.junit.Test;

import java.io.IOException;

import static org.jboss.installer.screens.domain.DatabaseSecurityDomainPanel.SQL_QUERY_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.GROUP_SELECTOR_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.HASH_ENC_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.PASSWORD_SELECTOR_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.SALT_ENCODING_LABEL;
import static org.jboss.installer.screens.domain.SelectorPanel.SALT_SELECTOR_LABEL;
import static org.jboss.installer.validators.CertificateSecurityDomainValidator.GENERIC_VALIDATION_EMPTY_FIELD;
import static org.jboss.installer.validators.IntRangeValidator.INVALID_INT_VALUE;
import static org.jboss.installer.validators.IntRangeValidator.INVALID_MIN_VALUE;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class DataBaseSecurityDomainValidatorTest {

    private final MockLanguageUtils langUtils = new MockLanguageUtils();
    private final DataBaseSecurityDomainValidator validator = new DataBaseSecurityDomainValidator(langUtils);
    private DatabaseSecurity.JdbcConfig plainConfig;
    private DatabaseSecurity.JdbcConfig simpleDigestConfig;
    private DatabaseSecurity.JdbcConfig saltedDigestConfig;

    @Before
    public void setUp() throws IOException {
        plainConfig = defaultPlainConfig();
        simpleDigestConfig = defaultSimpleDigestConfig();
        saltedDigestConfig = defaultSaltedDigestConfig();
    }

    @Test
    public void testDefaultConfigOK() {
        assertEquals(ValidationResult.ok(), validator.validate(plainConfig));
        assertEquals(ValidationResult.ok(), validator.validate(simpleDigestConfig));
        assertEquals(ValidationResult.ok(), validator.validate(saltedDigestConfig));
    }

    @Test
    public void testSqlQueryPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(SQL_QUERY_LABEL));
        plainConfig.setSqlQuery(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(plainConfig));

    }

    @Test
    public void testPasswordIndexPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(PASSWORD_SELECTOR_LABEL));
        plainConfig.setPasswordIndex(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(plainConfig));
        plainConfig.setPasswordIndex("text");
        assertTrue(validator.validate(plainConfig).getMessage().contains(INVALID_INT_VALUE));
        plainConfig.setPasswordIndex("0");
        assertTrue(validator.validate(plainConfig).getMessage().contains(INVALID_MIN_VALUE));
    }

    @Test
    public void testGroupIndexPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(GROUP_SELECTOR_LABEL));
        plainConfig.setGroupIndex(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(plainConfig));
        plainConfig.setGroupIndex("text");
        assertTrue(validator.validate(plainConfig).getMessage().contains(INVALID_INT_VALUE));
        plainConfig.setGroupIndex("0");
        assertTrue(validator.validate(plainConfig).getMessage().contains(INVALID_MIN_VALUE));
    }

    @Test
    public void testHashEncodingPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(HASH_ENC_LABEL));
        simpleDigestConfig.setHashEncoding(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(simpleDigestConfig));
    }

    @Test
    public void testSaltIndexPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(SALT_SELECTOR_LABEL));
        saltedDigestConfig.setSaltIndex(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(saltedDigestConfig));
        saltedDigestConfig.setSaltIndex("text");
        assertTrue(validator.validate(saltedDigestConfig).getMessage().contains(INVALID_INT_VALUE));
        saltedDigestConfig.setSaltIndex("0");
        assertTrue(validator.validate(saltedDigestConfig).getMessage().contains(INVALID_MIN_VALUE));
    }

    @Test
    public void testEncodingAlgorithmPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(HASH_ENC_LABEL));
        saltedDigestConfig.setEncodingAlgorithm(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(saltedDigestConfig));
    }

    @Test
    public void testSaltEncodingPresentExistsAndNonEmpty() {
        final String expectedError = langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(SALT_ENCODING_LABEL));
        saltedDigestConfig.setSaltEncoding(null);
        assertEquals(ValidationResult.error(expectedError), validator.validate(saltedDigestConfig));
    }

    private DatabaseSecurity.JdbcConfig defaultPlainConfig() {
        final DatabaseSecurity.JdbcConfig config = new DatabaseSecurity.JdbcConfig();
        config.setDatasourceName("ExampleDS");
        config.setSqlQuery("sql query");
        config.setMapperType(DatabaseSecurity.JdbcConfig.MapperType.Plain);
        config.setPasswordIndex("1");
        config.setGroupIndex("2");
        return config;
    }

    private DatabaseSecurity.JdbcConfig defaultSimpleDigestConfig() {
        final DatabaseSecurity.JdbcConfig config = new DatabaseSecurity.JdbcConfig();
        config.setDatasourceName("ExampleDS");
        config.setSqlQuery("sql query");
        config.setMapperType(DatabaseSecurity.JdbcConfig.MapperType.Digest);
        config.setPasswordIndex("1");
        config.setGroupIndex("2");
        config.setEncodingAlgorithm("base64");
        config.setHashEncoding("simple-digest-md2");
        return config;
    }

    private DatabaseSecurity.JdbcConfig defaultSaltedDigestConfig() {
        final DatabaseSecurity.JdbcConfig config = new DatabaseSecurity.JdbcConfig();
        config.setDatasourceName("ExampleDS");
        config.setSqlQuery("sql query");
        config.setMapperType(DatabaseSecurity.JdbcConfig.MapperType.SaltedDigest);
        config.setPasswordIndex("1");
        config.setGroupIndex("2");
        config.setSaltIndex("3");
        config.setEncodingAlgorithm("base64");
        config.setSaltEncoding("base64");
        config.setHashEncoding("simple-digest-md2");
        return config;
    }
}
