/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.actions;

import org.codehaus.plexus.util.FileUtils;
import org.jboss.installer.common.InstallerDialogs;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.Navigator;
import org.jboss.installer.core.ScreenManager;

import javax.swing.JFrame;
import java.awt.Frame;
import java.io.IOException;
import java.nio.file.Path;

public class QuitAction {

    private static final String QUIT_ACTION_DESCRIPTION = "quit_action.description";
    private static final String QUIT_ACTION_DELETE_ERROR = "quit_action.delete_error";
    private final LanguageUtils langUtils;
    private final Navigator navigator;
    private final ScreenManager screenManager;
    private final JFrame rootFrame;

    public QuitAction(LanguageUtils langUtils, Navigator navigator, ScreenManager screenManager, JFrame rootFrame) {
        this.langUtils = langUtils;
        this.navigator = navigator;
        this.screenManager = screenManager;
        this.rootFrame = rootFrame;
    }


    public void actionPerformed() {
        if (!screenManager.hasPrevious() || !screenManager.hasNext()) {
            closeWindows();
            return;
        }
        // create the options strings so that we can choose which one is default
        Object[] options = new Object[]{langUtils.getString("generic.yes"), langUtils.getString("generic.no")};
        if (InstallerDialogs.showQuestionMessage(langUtils.getString(QUIT_ACTION_DESCRIPTION), options, null)) {
            navigator.stopCurrentAction();
            Path installationPath = navigator.getInstallationData().getTargetFolder();
            if (installationPath != null && screenManager.hasNext()) {
                try {
                    FileUtils.deleteDirectory(installationPath.toFile());
                } catch (IOException e) {
                    InstallerDialogs.showErrorMessage(langUtils.getString(QUIT_ACTION_DELETE_ERROR), langUtils, null);
                    closeWindows();
                }
            }
            closeWindows();
        }
    }

    private void closeWindows() {
        // dispose of all the frames to release the resources
        for (Frame frame : rootFrame.getFrames()) {
            frame.dispose();
        }
        rootFrame.dispose();

        // still need to call System.exit because the testing framework (jemmy)
        // the TS requires the installer to be run in the same JVM as the tests, making 'dispose' never exit
        System.exit(0);
    }
}
