/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2023 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.installer.dialogs.downloader;

import org.jboss.installer.common.FontResources;
import org.jboss.installer.common.InstallerDialogs;
import org.jboss.installer.common.UiResources;
import org.jboss.installer.common.WidthWrappingTextMessagePane;
import org.jboss.installer.core.LanguageUtils;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DownloadDisplayPanel extends JPanel implements DownloadHandler {
    private final JButton okButton;
    private final JButton cancelButton;
    private final LanguageUtils langUtils;
    private Map<Download, JProgressBar> progressBars = new HashMap<>();

    public DownloadDisplayPanel(List<Download> downloads, DownloadManager downloadManager, LanguageUtils langUtils) {
        this.langUtils = langUtils;
        setLayout(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        c.anchor = GridBagConstraints.FIRST_LINE_START;
        c.fill = GridBagConstraints.BOTH;
        c.gridwidth = 2;
        c.insets = new Insets(10, 0, 10, 0);
        c.gridx = 0;
        c.gridy = 0;

        for (Download download : downloads) {
            this.add(new WidthWrappingTextMessagePane(langUtils.getString("generic.downloading.label")+ " " + download.getUrl(),
                    InstallerDialogs.DIALOG_TEXT_MAX_WIDTH, FontResources.getOpenSansRegular(), langUtils.getLocale()), c);

            c.gridy++;
            final JProgressBar progressBar = new JProgressBar();
            progressBars.put(download, progressBar);
            this.add(progressBar, c);
            c.gridy++;
        }

        c.gridwidth = 1;
        okButton = UiResources.createButton(langUtils.getString("generic.ok"), (event) -> {
        });
        okButton.setEnabled(false);
        this.add(okButton, c);
        c.gridx++;
        cancelButton = UiResources.createButton(langUtils.getString("generic.cancel"), (event) -> {
            downloadManager.cancel();
        });
        this.add(cancelButton, c);

        this.revalidate();
    }

    public Object getOKButton() {
        return okButton;
    }

    public Object[] getButtons() {
        return new Object[]{okButton, cancelButton};
    }

    @Override
    public void updateProgress(Download item, int progress) {
        if (progressBars.containsKey(item)) {
            progressBars.get(item).setValue(progress);
        }
    }

    @Override
    public void downloadsComplete() {
        okButton.setEnabled(true);
    }

    @Override
    public void downloadError(Download download, IOException e) {
        InstallerDialogs.showErrorMessage(langUtils.getString("generic.download.failure", download.getUrl().toString()),
                langUtils, this);
        cancelButton.doClick();
    }
}
