/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2021 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.server;

import org.jboss.as.controller.OperationFailedException;
import org.jboss.as.controller.PathAddress;
import org.jboss.dmr.ModelNode;
import org.junit.Test;

import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;

import static org.jboss.as.controller.operations.common.Util.createEmptyOperation;
import static org.jboss.installer.test.utils.TestServer.TARGET_PATH;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

public class StandaloneServerTest extends AbstractEmbeddedServerTest {

    @Override
    protected EmbeddedServer getServer(Path targetPath) {
        return new StandaloneServer(TARGET_PATH);
    }

    @Override
    protected void assertServerMode(ModelNode node) {
        assertEquals("EMBEDDED_SERVER", node.get(0).get("summary").get("standalone-or-domain-identifier").asString());
    }

    @Override
    protected String getConfig() {
        return "standalone.xml";
    }

    @Override
    protected List<String> getExpectedConfigurations() {
        return Arrays.asList("standalone.xml", "standalone-ha.xml", "standalone-full-ha.xml", "standalone-full.xml");
    }

    @Test
    public void executeReturnsOperationResult() throws Exception {
        EmbeddedServer srv = getServer(TARGET_PATH);

        try {
            srv.start(getConfig());
            final ModelNode op = createEmptyOperation("read-resource", PathAddress.pathAddress("subsystem", "logging"));
            final ModelNode result = srv.execute(op, "");
            assertNotNull(result);
            assertEquals(true, result.get("add-logging-api-dependencies").asBoolean());
        } finally {
            srv.shutdown();
            srv.close();
        }
    }

    @Test
    public void executeThrowsExceptionOnFailure() throws Exception {
        EmbeddedServer srv = getServer(TARGET_PATH);

        try {
            srv.start(getConfig());
            final ModelNode op = createEmptyOperation("read-resource", PathAddress.pathAddress("subsystem", "idontexist"));
            try {
                srv.execute(op, "");
                fail("operation should fail");
            } catch (OperationFailedException e) {
                // OK, do nothing
            }
        } finally {
            srv.shutdown();
            srv.close();
        }
    }
}
