/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2023 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Test;

import static org.assertj.core.api.Assertions.assertThat;
import static org.jboss.installer.validators.PortOffsetValidator.EMPTY_OFFSET_ERROR_KEY;
import static org.jboss.installer.validators.PortOffsetValidator.HIGH_OFFSET_ERROR_KEY;
import static org.jboss.installer.validators.PortOffsetValidator.MAX_PORT_VALUE;
import static org.jboss.installer.validators.PortOffsetValidator.NEGATIVE_OFFSET_ERROR_KEY;
import static org.jboss.installer.validators.PortOffsetValidator.NON_INT_OFFSET_ERROR_KEY;
import static org.junit.Assert.assertEquals;

public class PortOffsetValidatorTest {

    private final PortOffsetValidator portOffsetValidator = new PortOffsetValidator(new MockLanguageUtils());

    @Test
    public void emptyString_CausesError() {
        final ValidationResult result = portOffsetValidator.validate("");
        assertEquals(ValidationResult.Result.ERROR, result.getResult());

        assertThat(result.getMessage())
                .contains(EMPTY_OFFSET_ERROR_KEY);
    }

    @Test
    public void nullString_CausesError() {
        final ValidationResult result = portOffsetValidator.validate(null);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());

        assertThat(result.getMessage())
                .contains(EMPTY_OFFSET_ERROR_KEY);
    }

    @Test
    public void integerString_IsOK() {
        final ValidationResult result = portOffsetValidator.validate("100");
        assertEquals(ValidationResult.Result.OK, result.getResult());
    }

    @Test
    public void nonIntegerString_CausesError() {
        final ValidationResult result = portOffsetValidator.validate("abc");
        assertEquals(ValidationResult.Result.ERROR, result.getResult());

        assertThat(result.getMessage())
                .contains(NON_INT_OFFSET_ERROR_KEY);
    }

    @Test
    public void negativeIntegerString_CausesError() {
        final ValidationResult result = portOffsetValidator.validate("-1");
        assertEquals(ValidationResult.Result.ERROR, result.getResult());

        assertThat(result.getMessage())
                .contains(NEGATIVE_OFFSET_ERROR_KEY);
    }

    @Test
    public void offsetOverMaxPortValue_CausesError() {
        final ValidationResult result = portOffsetValidator.validate("" + (MAX_PORT_VALUE - portOffsetValidator.getMaxPortValue() + 1));
        assertEquals(ValidationResult.Result.ERROR, result.getResult());

        assertThat(result.getMessage())
                .contains(HIGH_OFFSET_ERROR_KEY);
    }

    @Test
    public void offsetOnMaxPortValue_IsOk() {
        final ValidationResult result = portOffsetValidator.validate("" + (MAX_PORT_VALUE - portOffsetValidator.getMaxPortValue()));
        assertEquals(ValidationResult.Result.OK, result.getResult());
    }

    @Test
    public void offsetOverIntegerMaxValue_CausesError() {
        final ValidationResult result = portOffsetValidator.validate("" + Long.MAX_VALUE);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());

        assertThat(result.getMessage())
                .contains(HIGH_OFFSET_ERROR_KEY);
    }
}