/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2023 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.core;

import org.junit.Test;

import java.util.Optional;
import java.util.Set;

import static org.junit.Assert.*;

public class MnemonicUtilsTest {

    @Test
    public void testEmptyString_ReturnsEmpty() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();

        assertEquals(Optional.empty(), mnemonicUtils.findMnemonic(""));
    }

    @Test
    public void testOneLetterWord_ReturnTheLetter() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();

        assertCharEquals('o', mnemonicUtils.findMnemonic("O"));
    }

    @Test
    public void testMultiLetterWord_ReturnFirstLetter() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();

        assertCharEquals('h', mnemonicUtils.findMnemonic("Hello"));
    }

    @Test
    public void testFirstLetterUsed_ReturnSecondLetter() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();
        mnemonicUtils.findMnemonic("Hello");

        assertCharEquals('i', mnemonicUtils.findMnemonic("Hi"));
    }

    @Test
    public void testAllLettersUsed_ReturnsEmpty() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();
        mnemonicUtils.findMnemonic("Hello");
        mnemonicUtils.findMnemonic("Hello");
        mnemonicUtils.findMnemonic("Hello");

        assertEquals(Optional.empty(), mnemonicUtils.findMnemonic("Hel"));
    }

    @Test
    public void testFirstLetterReserved_ReturnsSecondLetter() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();
        try {
            MnemonicUtils.reserveGlobal(Set.of('h'));
            assertCharEquals('e', mnemonicUtils.findMnemonic("Hello"));
        } finally {
            MnemonicUtils.resetGlobal();
        }
    }

    @Test
    public void testForceIgnoreGlobalReserved() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();
        try {
            MnemonicUtils.reserveGlobal(Set.of('h'));
            assertCharEquals('h', mnemonicUtils.findMnemonic("Hello", true));
        } finally {
            MnemonicUtils.resetGlobal();
        }
    }

    @Test
    public void testFirstLetterReservedSecondUsed_ReturnsThirdLetter() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();
        try {
            MnemonicUtils.reserveGlobal(Set.of('h'));
            mnemonicUtils.findMnemonic("Hello");
            assertCharEquals('l', mnemonicUtils.findMnemonic("Hello"));
        } finally {
            MnemonicUtils.resetGlobal();
        }
    }

    @Test
    public void testCapitalAndLowerLetterAreTreatedAsTheSame() {
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();
        mnemonicUtils.findMnemonic("Hello");

        assertCharEquals('e', mnemonicUtils.findMnemonic("hello"));
    }

    private void assertCharEquals(Character expected, Optional<Character> actual) {
        if (actual.isEmpty()) {
            fail("Expecting " + expected + " + but got empty Optional");
        }
        assertEquals(expected, actual.get());
    }
}