/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2024 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.postinstall.task;

import static org.jboss.installer.test.utils.TestServer.TARGET_PATH;
import static org.junit.Assert.assertEquals;

import java.io.File;
import java.nio.file.Path;
import java.util.Optional;

import org.apache.commons.io.FileUtils;
import org.jboss.installer.auto.InstallationDataSerializer;
import org.jboss.installer.core.InstallationData;
import org.jboss.installer.test.utils.MockInputConsole;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.jboss.installer.validators.LDAPValidatorTest;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class LDAPSetupDescriptorTest {
    protected static final String LDAP_REALM_NAME = "my-ldap-realm";

    @Rule
    public TemporaryFolder tempFolder = new TemporaryFolder();
    private InstallationData idata;

    @Before
    public void setUp() throws Exception {
        idata = new InstallationData();
        idata.setTargetFolder(TARGET_PATH);

        LDAPManagementAuthConfig config = new LDAPManagementAuthConfig("test.connection", LDAPValidatorTest.ldapTestUrl, LDAPValidatorTest.ldapAdminDn, LDAPValidatorTest.ldapAdminPassword);
        config.setRealmName(LDAP_REALM_NAME);
        config.setFilterType(LDAPManagementAuthConfig.FilterType.USERNAME);
        config.setFilter("uid");
        config.setRecursive(false);
        config.setBaseDN("o=TestOrganization");
        idata.putConfig(config);
    }

    @Test
    public void testNotSerializingPassword() throws Exception {
        final InstallationDataSerializer serializer = new InstallationDataSerializer(new MockLanguageUtils(), new MockInputConsole());
        final Path outputFile = tempFolder.newFile().toPath();
        serializer.serialize(idata, outputFile);

        // we expect a password provided by the input console, due to missing .variables file and a missing value
        final LDAPManagementAuthConfig config = serializer.deserialize(outputFile, Optional.empty()).getConfig(LDAPManagementAuthConfig.class);
        assertEquals("test", config.getPassword());
    }

    @Test
    public void testSerializeConfig() throws Exception {
        final InstallationDataSerializer serializer = new InstallationDataSerializer(new MockLanguageUtils());
        final Path outputFile = tempFolder.newFile().toPath();
        serializer.serialize(idata, outputFile);

        final File variableFile = outputFile.getParent().resolve(outputFile.getFileName().toString() + ".variables").toFile();
        FileUtils.writeStringToFile(variableFile, LDAPManagementAuthConfig.PASSWORD_PROPERTY_NAME + "=" + LDAPValidatorTest.ldapAdminPassword, "UTF-8");
        final LDAPManagementAuthConfig deserializedConfig = serializer.deserialize(outputFile, Optional.empty()).getConfig(LDAPManagementAuthConfig.class);

        assertEquals(idata.getConfig(LDAPManagementAuthConfig.class), deserializedConfig);
    }
}
