/*
 * Decompiled with CFR 0.152.
 */
package org.hornetq.core.protocol.stomp;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import org.hornetq.api.core.HornetQBuffer;
import org.hornetq.api.core.HornetQException;
import org.hornetq.api.core.Interceptor;
import org.hornetq.api.core.SimpleString;
import org.hornetq.core.journal.IOAsyncTask;
import org.hornetq.core.logging.Logger;
import org.hornetq.core.postoffice.Bindings;
import org.hornetq.core.protocol.stomp.StompConnection;
import org.hornetq.core.protocol.stomp.StompDecoder;
import org.hornetq.core.protocol.stomp.StompException;
import org.hornetq.core.protocol.stomp.StompFrame;
import org.hornetq.core.protocol.stomp.StompSession;
import org.hornetq.core.protocol.stomp.StompUtils;
import org.hornetq.core.server.HornetQServer;
import org.hornetq.core.server.ServerSession;
import org.hornetq.core.server.impl.ServerMessageImpl;
import org.hornetq.spi.core.protocol.ConnectionEntry;
import org.hornetq.spi.core.protocol.ProtocolManager;
import org.hornetq.spi.core.protocol.RemotingConnection;
import org.hornetq.spi.core.remoting.Acceptor;
import org.hornetq.spi.core.remoting.Connection;
import org.hornetq.spi.core.security.HornetQSecurityManager;
import org.hornetq.utils.UUIDGenerator;

class StompProtocolManager
implements ProtocolManager {
    private static final SimpleString JMS_TOPIC_PREFIX = new SimpleString("jms.topic");
    private static final SimpleString JMS_QUEUE_PREFIX = new SimpleString("jms.queue");
    private static final Logger log = Logger.getLogger(StompProtocolManager.class);
    private static final String CONNECTION_ID_PROP = "__HQ_CID";
    private final HornetQServer server;
    private final SimpleString managementAddress;
    private final Executor executor;
    private final Map<String, StompSession> transactedSessions = new HashMap<String, StompSession>();
    private final Map<Object, StompSession> sessions = new HashMap<Object, StompSession>();

    private static StompFrame createError(Exception e, StompFrame request) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            PrintWriter stream = new PrintWriter(new OutputStreamWriter((OutputStream)baos, "UTF-8"));
            e.printStackTrace(stream);
            stream.close();
            HashMap<String, Object> headers = new HashMap<String, Object>();
            headers.put("message", e.getMessage());
            String receiptId = (String)request.getHeaders().get("receipt");
            if (receiptId != null) {
                headers.put("receipt-id", receiptId);
            }
            byte[] payload = baos.toByteArray();
            headers.put("content-length", payload.length);
            return new StompFrame("ERROR", headers, payload);
        }
        catch (UnsupportedEncodingException ex) {
            log.warn("Unable to create ERROR frame from the exception", ex);
            return null;
        }
    }

    public StompProtocolManager(HornetQServer server, List<Interceptor> interceptors) {
        this.server = server;
        this.managementAddress = server.getConfiguration().getManagementAddress();
        this.executor = server.getExecutorFactory().getExecutor();
    }

    @Override
    public ConnectionEntry createConnectionEntry(Acceptor acceptorUsed, Connection connection) {
        Long ttl;
        StompConnection conn = new StompConnection(acceptorUsed, connection, this, this.server.getExecutorFactory().getExecutor());
        String ttlStr = (String)acceptorUsed.getConfiguration().get("connection-ttl");
        Long l = ttl = ttlStr == null ? null : Long.valueOf(ttlStr);
        if (ttl != null) {
            if (ttl > 0L) {
                return new ConnectionEntry(conn, null, System.currentTimeMillis(), ttl);
            }
            throw new IllegalStateException("Stomp Connection TTL cannot be negative : " + ttl);
        }
        ttl = this.server.getConfiguration().getConnectionTTLOverride();
        if (ttl != -1L) {
            return new ConnectionEntry(conn, null, System.currentTimeMillis(), ttl);
        }
        return new ConnectionEntry(conn, null, System.currentTimeMillis(), 60000L);
    }

    @Override
    public void removeHandler(String name) {
    }

    @Override
    public int isReadyToHandle(HornetQBuffer buffer) {
        return -1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void handleBuffer(RemotingConnection connection, HornetQBuffer buffer) {
        long start = System.nanoTime();
        StompConnection conn = (StompConnection)connection;
        conn.setDataReceived();
        StompDecoder decoder = conn.getDecoder();
        do {
            StompFrame request;
            try {
                request = decoder.decode(buffer);
            }
            catch (Exception e) {
                log.error("Failed to decode", e);
                return;
            }
            if (request == null) break;
            try {
                String command = request.getCommand();
                StompFrame response = null;
                if ("CONNECT".equals(command)) {
                    response = this.onConnect(request, conn);
                } else if ("DISCONNECT".equals(command)) {
                    response = this.onDisconnect(request, conn);
                } else if ("SEND".equals(command)) {
                    response = this.onSend(request, conn);
                } else if ("SUBSCRIBE".equals(command)) {
                    response = this.onSubscribe(request, conn);
                } else if ("UNSUBSCRIBE".equals(command)) {
                    response = this.onUnsubscribe(request, conn);
                } else if ("ACK".equals(command)) {
                    response = this.onAck(request, conn);
                } else if ("BEGIN".equals(command)) {
                    response = this.onBegin(request, this.server, conn);
                } else if ("COMMIT".equals(command)) {
                    response = this.onCommit(request, conn);
                } else if ("ABORT".equals(command)) {
                    response = this.onAbort(request, conn);
                } else {
                    log.error("Unsupported Stomp frame: " + request);
                    response = new StompFrame("ERROR", new HashMap<String, Object>(), ("Unsupported frame: " + command).getBytes());
                }
                if (request.getHeaders().containsKey("receipt")) {
                    if (response == null) {
                        HashMap<String, Object> h = new HashMap<String, Object>();
                        response = new StompFrame("RECEIPT", h);
                    }
                    response.getHeaders().put("receipt-id", request.getHeaders().get("receipt"));
                }
                if (response != null) {
                    this.sendReply(conn, response);
                }
                if (!"DISCONNECT".equals(command)) continue;
                conn.destroy();
            }
            catch (Exception e) {
                e.printStackTrace();
                StompFrame error = StompProtocolManager.createError(e, request);
                if (error == null) continue;
                this.sendReply(conn, error);
            }
            finally {
                this.server.getStorageManager().clearContext();
            }
        } while (decoder.hasBytes());
        long end = System.nanoTime();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void send(StompConnection connection, StompFrame frame) {
        if (log.isTraceEnabled()) {
            log.trace("sent " + frame);
        }
        StompConnection stompConnection = connection;
        synchronized (stompConnection) {
            if (connection.isDestroyed() || !connection.isValid()) {
                log.warn("Connection closed " + connection);
                return;
            }
            try {
                HornetQBuffer buffer = frame.toHornetQBuffer();
                connection.getTransportConnection().write(buffer, false, false);
            }
            catch (Exception e) {
                log.error("Unable to send frame " + frame, e);
            }
        }
    }

    private StompFrame onSubscribe(StompFrame frame, StompConnection connection) throws Exception {
        Map<String, Object> headers = frame.getHeaders();
        String destination = (String)headers.get("destination");
        String selector = (String)headers.get("selector");
        String ack = (String)headers.get("ack");
        String id = (String)headers.get("id");
        String durableSubscriptionName = (String)headers.get("durable-subscriber-name");
        boolean noLocal = false;
        if (headers.containsKey("no-local")) {
            noLocal = Boolean.parseBoolean((String)headers.get("no-local"));
        }
        if (noLocal) {
            String noLocalFilter = "__HQ_CID <> '" + connection.getID().toString() + "'";
            selector = selector == null ? noLocalFilter : selector + " AND " + noLocalFilter;
        }
        if (ack == null) {
            ack = "auto";
        }
        String subscriptionID = null;
        if (id != null) {
            subscriptionID = id;
        } else {
            if (destination == null) {
                throw new StompException("Client must set destination or id header to a SUBSCRIBE command");
            }
            subscriptionID = "subscription/" + destination;
        }
        this.validateDestination(new SimpleString(destination));
        StompSession stompSession = this.getSession(connection);
        stompSession.setNoLocal(noLocal);
        if (stompSession.containsSubscription(subscriptionID)) {
            throw new StompException("There already is a subscription for: " + subscriptionID + ". Either use unique subscription IDs or do not create multiple subscriptions for the same destination");
        }
        long consumerID = this.server.getStorageManager().generateUniqueID();
        String clientID = connection.getClientID() != null ? connection.getClientID() : null;
        stompSession.addSubscription(consumerID, subscriptionID, clientID, durableSubscriptionName, destination, selector, ack);
        return null;
    }

    private StompFrame onUnsubscribe(StompFrame frame, StompConnection connection) throws Exception {
        Map<String, Object> headers = frame.getHeaders();
        String destination = (String)headers.get("destination");
        String id = (String)headers.get("id");
        String subscriptionID = null;
        if (id != null) {
            subscriptionID = id;
        } else {
            if (destination == null) {
                throw new StompException("Must specify the subscription's id or the destination you are unsubscribing from");
            }
            subscriptionID = "subscription/" + destination;
        }
        StompSession stompSession = this.getSession(connection);
        boolean unsubscribed = stompSession.unsubscribe(subscriptionID);
        if (!unsubscribed) {
            throw new StompException("Cannot unsubscribe as no subscription exists for id: " + subscriptionID);
        }
        return null;
    }

    private StompFrame onAck(StompFrame frame, StompConnection connection) throws Exception {
        Map<String, Object> headers = frame.getHeaders();
        String messageID = (String)headers.get("message-id");
        String txID = (String)headers.get("transaction");
        StompSession stompSession = null;
        if (txID != null) {
            log.warn("Transactional acknowledgement is not supported");
        }
        stompSession = this.getSession(connection);
        stompSession.acknowledge(messageID);
        return null;
    }

    private StompFrame onBegin(StompFrame frame, HornetQServer server, StompConnection connection) throws Exception {
        Map<String, Object> headers = frame.getHeaders();
        String txID = (String)headers.get("transaction");
        if (txID == null) {
            throw new StompException("transaction header is mandatory to BEGIN a transaction");
        }
        if (this.transactedSessions.containsKey(txID)) {
            throw new StompException("Transaction already started: " + txID);
        }
        this.getTransactedSession(connection, txID);
        return null;
    }

    private StompFrame onCommit(StompFrame frame, StompConnection connection) throws Exception {
        Map<String, Object> headers = frame.getHeaders();
        String txID = (String)headers.get("transaction");
        if (txID == null) {
            throw new StompException("transaction header is mandatory to COMMIT a transaction");
        }
        StompSession session = this.getTransactedSession(connection, txID);
        if (session == null) {
            throw new StompException("No transaction started: " + txID);
        }
        this.transactedSessions.remove(txID);
        session.getSession().commit();
        return null;
    }

    private StompFrame onAbort(StompFrame frame, StompConnection connection) throws Exception {
        Map<String, Object> headers = frame.getHeaders();
        String txID = (String)headers.get("transaction");
        if (txID == null) {
            throw new StompException("transaction header is mandatory to ABORT a transaction");
        }
        StompSession session = this.getTransactedSession(connection, txID);
        if (session == null) {
            throw new StompException("No transaction started: " + txID);
        }
        this.transactedSessions.remove(txID);
        session.getSession().rollback(false);
        return null;
    }

    private void checkConnected(StompConnection connection) throws StompException {
        if (!connection.isValid()) {
            throw new StompException("Not connected");
        }
    }

    private StompSession getSession(StompConnection connection) throws Exception {
        StompSession stompSession = this.sessions.get(connection.getID());
        if (stompSession == null) {
            stompSession = new StompSession(connection, this, this.server.getStorageManager().newContext(this.server.getExecutorFactory().getExecutor()));
            String name = UUIDGenerator.getInstance().generateStringUUID();
            ServerSession session = this.server.createSession(name, connection.getLogin(), connection.getPasscode(), 102400, connection, true, false, false, false, null, stompSession);
            stompSession.setServerSession(session);
            this.sessions.put(connection.getID(), stompSession);
        }
        this.server.getStorageManager().setContext(stompSession.getContext());
        return stompSession;
    }

    private StompSession getTransactedSession(StompConnection connection, String txID) throws Exception {
        StompSession stompSession = this.transactedSessions.get(txID);
        if (stompSession == null) {
            stompSession = new StompSession(connection, this, this.server.getStorageManager().newContext(this.executor));
            String name = UUIDGenerator.getInstance().generateStringUUID();
            ServerSession session = this.server.createSession(name, connection.getLogin(), connection.getPasscode(), 102400, connection, false, false, false, false, null, stompSession);
            stompSession.setServerSession(session);
            this.transactedSessions.put(txID, stompSession);
        }
        this.server.getStorageManager().setContext(stompSession.getContext());
        return stompSession;
    }

    private StompFrame onDisconnect(StompFrame frame, StompConnection connection) throws Exception {
        this.cleanup(connection);
        return null;
    }

    private StompFrame onSend(StompFrame frame, StompConnection connection) throws Exception {
        this.checkConnected(connection);
        Map<String, Object> headers = frame.getHeaders();
        String destination = (String)headers.remove("destination");
        String txID = (String)headers.remove("transaction");
        long timestamp = System.currentTimeMillis();
        SimpleString address = SimpleString.toSimpleString(destination);
        ServerMessageImpl message = new ServerMessageImpl(this.server.getStorageManager().generateUniqueID(), 512);
        message.setTimestamp(timestamp);
        message.setAddress(address);
        this.validateDestination(address);
        StompUtils.copyStandardHeadersFromFrameToMessage(frame, message);
        if (headers.containsKey("content-length")) {
            message.setType((byte)4);
            message.getBodyBuffer().writeBytes(frame.getContent());
        } else {
            message.setType((byte)3);
            String text = new String(frame.getContent(), "UTF-8");
            message.getBodyBuffer().writeNullableSimpleString(SimpleString.toSimpleString(text));
        }
        StompSession stompSession = null;
        stompSession = txID == null ? this.getSession(connection) : this.getTransactedSession(connection, txID);
        if (stompSession.isNoLocal()) {
            message.putStringProperty(CONNECTION_ID_PROP, connection.getID().toString());
        }
        stompSession.sendInternal(message, true);
        return null;
    }

    private void validateDestination(SimpleString address) throws Exception, HornetQException {
        Bindings binding;
        if (!(!address.startsWith(JMS_QUEUE_PREFIX) && !address.startsWith(JMS_TOPIC_PREFIX) || address.equals(this.managementAddress) || (binding = this.server.getPostOffice().lookupBindingsForAddress(address)) != null && binding.getBindings().size() != 0)) {
            throw new HornetQException(106, "Address " + address + " has not been deployed");
        }
    }

    private StompFrame onConnect(StompFrame frame, StompConnection connection) throws Exception {
        Map<String, Object> headers = frame.getHeaders();
        String login = (String)headers.get("login");
        String passcode = (String)headers.get("passcode");
        String clientID = (String)headers.get("client-id");
        String requestID = (String)headers.get("request-id");
        HornetQSecurityManager sm = this.server.getSecurityManager();
        if (sm != null && this.server.getConfiguration().isSecurityEnabled()) {
            sm.validateUser(login, passcode);
        }
        connection.setLogin(login);
        connection.setPasscode(passcode);
        connection.setClientID(clientID);
        connection.setValid(true);
        HashMap<String, Object> h = new HashMap<String, Object>();
        h.put("session", connection.getID());
        if (requestID != null) {
            h.put("response-id", requestID);
        }
        return new StompFrame("CONNECTED", h);
    }

    public void cleanup(final StompConnection connection) {
        connection.setValid(false);
        this.executor.execute(new Runnable(){

            @Override
            public void run() {
                StompSession session = (StompSession)StompProtocolManager.this.sessions.remove(connection.getID());
                if (session != null) {
                    try {
                        session.getSession().rollback(true);
                        session.getSession().close(false);
                    }
                    catch (Exception e) {
                        log.warn(e.getMessage(), e);
                    }
                }
                Iterator iterator = StompProtocolManager.this.transactedSessions.entrySet().iterator();
                while (iterator.hasNext()) {
                    Map.Entry entry = iterator.next();
                    if (((StompSession)entry.getValue()).getConnection() != connection) continue;
                    ServerSession serverSession = ((StompSession)entry.getValue()).getSession();
                    try {
                        serverSession.rollback(true);
                        serverSession.close(false);
                    }
                    catch (Exception e) {
                        log.warn(e.getMessage(), e);
                    }
                    iterator.remove();
                }
            }
        });
    }

    private void sendReply(final StompConnection connection, final StompFrame frame) {
        this.server.getStorageManager().afterCompleteOperations(new IOAsyncTask(){

            @Override
            public void onError(int errorCode, String errorMessage) {
                log.warn("Error processing IOCallback code = " + errorCode + " message = " + errorMessage);
                StompFrame error = StompProtocolManager.createError(new HornetQException(errorCode, errorMessage), frame);
                StompProtocolManager.this.send(connection, error);
            }

            @Override
            public void done() {
                StompProtocolManager.this.send(connection, frame);
            }
        });
    }
}

