package org.jboss.ip.dbtool;

import org.apache.tools.ant.DefaultLogger;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.ProjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;

/**
 * Example of usage:
 * DbSchema dbSchema = new DbSchema(new File("/usr/share/jboss-eap"));
 * dbSchema.create("username", "password", "jdbc:postgresql://localhost:5432/db",
 * "org.postgresql.Driver", "PostgreSQLDialect",
 * new File("/usr/share/jboss-eap/modules/base/org/postgresql/main"));
 *
 * @author Vaclav Dedik <vdedik@redhat.com>
 */
public class DbSchema {
    private static Logger logger = LoggerFactory.getLogger(Main.LOGGER_NAME);

    /**
     * Location of the sql ant script relative to jboss home directory
     */
    private static final String ANT_SCRIPT_RELATIVE_PATH = "sql/build.xml";

    /**
     * Location of the default sql log file relative to jboss home directory
     */
    private static final String SQL_LOG_RELATIVE_PATH = "sql/sqlresults.log";

    /**
     * Ant target to be run
     */
    private static final String ANT_TARGET = "all";

    /**
     * JBoss home location
     */
    private File jbossHome;

    /**
     * Location of the ant output log file, ${jbossHome}/sql/sqlresults.log by default
     */
    private File antOutput;

    public DbSchema(File jbossHome) {
        this(jbossHome, new File(jbossHome.getAbsolutePath() + "/" + SQL_LOG_RELATIVE_PATH));
    }

    public DbSchema(File jbossHome, File antOutput) {
        this.jbossHome = jbossHome;
        this.antOutput = antOutput;
    }

    /**
     * Creates database dbtool by calling ${jbossHome}/sql/build.xml ant script
     *
     * @param username  - Database username, e.g. "postgresUser"
     * @param password  - Database password, e.g. "passw0rd"
     * @param url       - Database connection url, e.g. jdbc:postgresql://localhost:5432/db
     * @param driver    - Database driver class, fully qualified, e.g. org.postgresql.Driver
     * @param dialect   - Database dialect class, only "simple name" (i.e. not fully qualified)
     *                  e.g. PostgreSQLDialect
     * @param driverJar - Jar file with the jdbc driver
     */
    public void create(String username, String password, String url, String driver, String dialect, File driverJar)
            throws FileNotFoundException {
        logger.info("Starting export of sql dbtool... username='{}', password='**REDACTED**', url='{}', " +
                "driver='{}', dialect='{}', driverJar='{}', jbossHome='{}'", username, url, driver, dialect, driverJar,
                jbossHome);

        Project project = new Project();
        String buildFileName = jbossHome.getAbsolutePath() + "/" + ANT_SCRIPT_RELATIVE_PATH;
        File buildFile = new File(buildFileName);
        project.setUserProperty("ant.file", buildFile.getAbsolutePath());
        project.init();

        project.setProperty("driverjarloc", driverJar.getParent());
        project.setProperty("db.driver", driver);
        project.setProperty("db.url", url);
        project.setProperty("db.user", username);
        project.setProperty("db.password", password);
        project.setProperty("db.dialect", dialect);

        ProjectHelper helper = ProjectHelper.getProjectHelper();
        helper.parse(project, buildFile);

        project.addReference("ant.projectHelper", helper);

        project.addBuildListener(getDefaultLogger());
        project.executeTarget(ANT_TARGET);

        logger.info("Export of sql dbtool completed.");
    }

    private DefaultLogger getDefaultLogger() throws FileNotFoundException {
        DefaultLogger consoleLogger = new DefaultLogger();
        PrintStream printStream = new PrintStream(antOutput);
        consoleLogger.setErrorPrintStream(printStream);
        consoleLogger.setOutputPrintStream(printStream);
        consoleLogger.setMessageOutputLevel(Project.MSG_VERBOSE);
        return consoleLogger;
    }

}
