package org.jboss.ip.dbtool.cliutil;


import org.jboss.as.cli.CliInitializationException;
import org.jboss.as.cli.CommandContext;
import org.jboss.as.cli.CommandContextFactory;
import org.jboss.as.cli.CommandLineException;
import org.jboss.ip.dbtool.Main;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jboss.ip.dbtool.Utils;

import java.io.*;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;


/**
 * @author Alex Creasy <acreasy@redhat.com>
 */
public final class CommandExecutor {

    private static final String RELOAD_CMD = ":reload";
    private static final String DOMAIN_CMD_PREFIX = "/host=master";
    private static final String SHUTDOWN_CMD = ":shutdown";

    private static final String SRAMP_CFG = "s-ramp-add-sramp-conf.${driverName}.cli";

    private static final Logger logger = LoggerFactory.getLogger(Main.LOGGER_NAME);

    private final CommandContext ctx;
    private final String hostname;
    private final int port;

    public CommandExecutor() throws CliInitializationException {
        this(null, null, null, -1);
    }

    public CommandExecutor(final String username, final char[] password) throws CliInitializationException {
        this(username, password, null, -1);
    }

    public CommandExecutor(final String username, final char[] password, final String hostname,
                           final int port) throws CliInitializationException {

        logger.trace("CommandExecutor.CommandExecutor( username='{}', password=****, hostname='{}', port='{}' )",
                username, hostname, port);

        ctx = CommandContextFactory.getInstance().newCommandContext(username, password);

        ctx.setSilent(true);

        this.hostname = hostname;
        this.port = port;
    }

    public void connect() throws CommandLineException {
        logger.trace("CommandExecutor.connect() > Connecting to hostname='{}', port='{}'", hostname, port);
        ctx.connectController(hostname, port);
    }

    public void disconnect() {
        logger.trace("CommandExecutor.disconnect()");
        ctx.disconnectController();
    }

    public void backupConfig() throws CommandLineException {
        logger.trace("CommandExecutor.backupConfig()");
        submitCommand(":take-snapshot");
    }

    public void shutdownServer() throws CommandLineException {
        logger.trace("CommandExecutor.shutDownServer()");
        submitCommand(ctx.isDomainMode() ? DOMAIN_CMD_PREFIX + SHUTDOWN_CMD : SHUTDOWN_CMD);
    }

    public void reloadServer() throws CommandLineException {
        logger.trace("CommandExecutor.reloadServer()");
        submitCommand(ctx.isDomainMode() ? DOMAIN_CMD_PREFIX + RELOAD_CMD : RELOAD_CMD);
    }

    public void registerJdbcDriver(final String name, final String module, final String xaDatasourceClass)
            throws CommandLineException {
        registerJdbcDriver(null, name, module, xaDatasourceClass);
    }

    public void registerJdbcDriver(final String domainProfile, final String name, final String module,
                                   final String xaDatasourceClass)
            throws CommandLineException {

        logger.trace("CommandExecutor.registerJdbcDriver( name='{}', module='{}', xaDatasourceClass='{}', " +
                "domainProfile='{}' )", name, module, xaDatasourceClass, domainProfile);

        final StringBuilder cmdBuilder = new StringBuilder();

        if (domainProfile != null)
            cmdBuilder.append("/profile=").append(domainProfile);

        cmdBuilder.append("/subsystem=datasources/jdbc-driver=").append(name);
        cmdBuilder.append(":add(driver-name=\"").append(name);
        cmdBuilder.append("\",driver-module-name=\"").append(module);
        cmdBuilder.append("\",driver-xa-datasource-class-name=\"").append(xaDatasourceClass);
        cmdBuilder.append("\")");

        try {
            submitCommand(cmdBuilder.toString());
        } catch (CommandLineException e) {
            // IPFSW-28 - when the driver is already installed, first remove and then add it in a batch command

            final StringBuilder cmdRemoveBuilder = new StringBuilder();

            if (domainProfile != null)
                cmdRemoveBuilder.append("/profile=").append(domainProfile);
            cmdRemoveBuilder.append("/subsystem=datasources/jdbc-driver=").append(name).append(":remove");

            List<String> batch = new ArrayList<String>() {{
                add(cmdRemoveBuilder.toString());
                add(cmdBuilder.toString());
            }};

            submitBatch(batch);
        }
    }

    public void updateDatasource(final String dsName, final DatasourceValues dsValues) throws NoSuchDatasourceException,
            WriteAttributeException {
        updateDatasource(null, dsName, dsValues);
    }

    public void updateDatasource(final String domainProfile, final String dsName, final DatasourceValues dsValues)
            throws NoSuchDatasourceException, WriteAttributeException {

        if (!datasourceExists(domainProfile, dsName)) {
            String msg = "DatasourceValues: '" + dsName + "', ";

            if (domainProfile != null && !domainProfile.isEmpty())
                msg += "on domain profile: '" + domainProfile + "', ";

            msg += "does not exist.";
            throw new NoSuchDatasourceException(msg);
        }

        updateDatasourceAttribute(domainProfile, dsName, "connection-url", dsValues.getConnectionUrl());
        updateDatasourceAttribute(domainProfile, dsName, "driver-name", dsValues.getDriverName());
        updateDatasourceAttribute(domainProfile, dsName, "min-pool-size", String.valueOf(dsValues.getMinPoolSize()));
        updateDatasourceAttribute(domainProfile, dsName, "max-pool-size", String.valueOf(dsValues.getMaxPoolSize()));
        updateDatasourceAttribute(domainProfile, dsName, "user-name", dsValues.getUsername());
        updateDatasourceAttribute(domainProfile, dsName, "password", String.valueOf(dsValues.getPassword()));

        if (dsValues.getDriverExceptionSorter() != null && !dsValues.getDriverExceptionSorter().isEmpty())
            updateDatasourceAttribute(domainProfile, dsName, "exception-sorter-class-name",
                    dsValues.getDriverExceptionSorter());

        if (dsValues.getDriverValidConnectionChecker() != null && !dsValues.getDriverValidConnectionChecker().isEmpty())
            updateDatasourceAttribute(domainProfile, dsName, "valid-connection-checker-class-name",
                    dsValues.getDriverValidConnectionChecker());

        if (dsValues.getDriverStaleConnectionChecker() != null && !dsValues.getDriverStaleConnectionChecker().isEmpty())
            updateDatasourceAttribute(domainProfile, dsName, "stale-connection-checker-class-name",
                    dsValues.getDriverStaleConnectionChecker());
    }


    public boolean datasourceExists(final String dsName) {
        return datasourceExists(null, dsName);
    }

    public boolean datasourceExists(final String domainProfile, final String dsName) {

        final StringBuilder cmdBuilder = new StringBuilder();

        if (domainProfile != null)
            cmdBuilder.append("/profile=").append(domainProfile);

        cmdBuilder.append("/subsystem=datasources/data-source=").append(dsName).append(":read-resource");

        try {
            submitCommand(cmdBuilder.toString());
        } catch (CommandLineException cliEx) {
            return false;
        }
        return true;
    }

    public void updateDatasourceAttribute(final String domainProfile, final String dsName, final String attribute,
            final String value) throws WriteAttributeException {

        logger.trace("CommandExecutor.updateDatasourceAttribute( dsName='{}', attribute='{}', value='{}', " +
                "domainProfile='{}' )", dsName, attribute, value, domainProfile);

        final StringBuilder cmdBuilder = new StringBuilder();

        if (domainProfile != null)
            cmdBuilder.append("/profile=").append(domainProfile);

        cmdBuilder.append("/subsystem=datasources/data-source=").append(dsName)
                .append(":write-attribute(name=\"").append(attribute)
                .append("\",value=\"").append(value)
                .append("\")");

        try {
            submitCommand(cmdBuilder.toString());
        } catch (CommandLineException cliEx) {
            throw new WriteAttributeException("Unable to write '" + attribute + "' => '" + value + "' to datasource: " +
                    "'" + dsName + "'", cliEx);
        }
    }

    public void updateBpelDialect(final String domainProfile, final String dialect) throws WriteAttributeException {
        logger.trace("CommandExecutor.updateBpelDialect( domainProfile='{}', dialect='{}' )", domainProfile, dialect);

        final StringBuilder cmdBuilder = new StringBuilder();

        if (domainProfile != null)
            cmdBuilder.append("/profile=").append(domainProfile);

        cmdBuilder.append("/subsystem=switchyard/module=org.switchyard.component.bpel:" +
                "write-attribute(name=properties,value={bpel.db.mode=EXTERNAL,db.emb.create=false," +
                "bpel.db.ext.dataSource=java:jboss/datasources/BpelDS,hibernate.dialect=\"").append(dialect)
                .append("\"})");

        try {
            submitCommand(cmdBuilder.toString());
        } catch (CommandLineException cliEx) {
            throw new WriteAttributeException("Unable to write 'properties' => " +
                    "'{bpel.db.mode=EXTERNAL,db.emb.create=false,bpel.db.ext.dataSource=" +
                    "java:jboss/datasources/BpelDS,hibernate.dialect=" + dialect + "'", cliEx);
        }
    }

    public void redeploySrampConfig(final String domainProfile, final File jbossHomeDir, final String driverName)
            throws CommandLineException {
        logger.trace("CommandExecutor.redeploySrampConfig( domainProfile='{}', driverName='{}' )",
                domainProfile, driverName);

        List<String> batch = new ArrayList<String>();
        batch.add("/subsystem=infinispan/cache-container=modeshape/local-cache=sramp:remove");
        batch.add("/subsystem=modeshape/repository=sramp:remove");

        //load config
        String srampConfig = SRAMP_CFG.replaceAll("\\$\\{driverName\\}", driverName);

        List<String> lines = Utils.loadFile(new File(jbossHomeDir.getAbsolutePath() + "/cli-scripts/" + srampConfig));
        for (String line : lines) {
            if (domainProfile == null && !line.isEmpty()) {
                batch.add(line);
            } else if (!line.isEmpty()) {
                batch.add("/profile=" + domainProfile + line);
            }
        }

        submitBatch(batch);
    }

    public void seedSramp(final File srampShell, final File commandsPath) throws MalformedURLException {
        URL srampURL = srampShell.toURI().toURL();
        System.setProperty("one-jar.jar.path", srampShell.getAbsolutePath());

        ClassLoader loader = URLClassLoader.newInstance(
                new URL[]{srampURL},
                getClass().getClassLoader()
        );

        Class<?> clazz = null;
        try {
            clazz = Class.forName("OneJar", true, loader);
            Method method = clazz.getMethod("main", String[].class);
            Object arg = new String[] {"-f", commandsPath.getAbsolutePath()};
            method.invoke(null, arg);
        } catch (Exception e) {
            throw new RuntimeException("S-RAMP seed failed");
        }
    }

    private void submitCommand(final String cmd) throws CommandLineException {
        logger.trace("CommandExecutor.submitCommand => '{}'", cmd);
        try {
            ctx.handle(cmd);
        } catch (CommandLineException cliEx) {
            logger.warn("CommandExecutor.submitCommand => Error executing command", cliEx);
            throw cliEx;
        }
    }

    private void submitBatch(final List<String> cmds) throws CommandLineException {
        logger.trace("CommandExecutor.submitBatch => '{}'", cmds);
        try {
            ctx.handle("batch");
            for (String cmd : cmds) {
                ctx.handle(cmd);
            }
            ctx.handle("run-batch");
        } catch (CommandLineException cliEx) {
            logger.warn("CommandExecutor.submitBatch => Error executing command", cliEx);
            throw cliEx;
        }
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("CommandExecutor{");
        sb.append("ctx=").append(ctx);
        sb.append(", hostname='").append(hostname).append('\'');
        sb.append(", port=").append(port);
        sb.append('}');
        return sb.toString();
    }
}
