/*
 * #%L
 * Fuse EAP :: Config
 * %%
 * Copyright (C) 2015 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package org.jboss.fuse.eap.config;

import static org.wildfly.extras.config.LayerConfig.Type.INSTALLING;
import static org.wildfly.extras.config.LayerConfig.Type.OPTIONAL;
import static org.wildfly.extras.config.LayerConfig.Type.REQUIRED;

import java.net.URL;
import java.util.Arrays;
import java.util.List;

import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.Text;
import org.wildfly.extras.config.ConfigContext;
import org.wildfly.extras.config.ConfigPlugin;
import org.wildfly.extras.config.ConfigSupport;
import org.wildfly.extras.config.LayerConfig;

public abstract class RTGovConfigPlugin implements ConfigPlugin {

    private static final Namespace NS_DOMAIN = Namespace.getNamespace ("urn:jboss:domain:1.7");
    private static final Namespace NS_SECURITY = Namespace.getNamespace ("urn:jboss:domain:security:1.2");
    protected static final Namespace NS_OVERLORD_CONFIG = Namespace.getNamespace ("urn:jboss:domain:overlord-configuration:1.0");
    protected static final Namespace NS_OVERLORD_DEPLOY = Namespace.getNamespace ("urn:jboss:domain:overlord-deployment:1.0");

    @Override
    public abstract String getConfigName();

    @Override
    public List<LayerConfig> getLayerConfigs() {
        return Arrays.asList(new LayerConfig("fuse", REQUIRED, -10), new LayerConfig("bpms", OPTIONAL, -9), new LayerConfig("soa", OPTIONAL, -11), new LayerConfig ("rtgov", INSTALLING, -12));
    }

    @Override
    public void applyStandaloneConfigChange(ConfigContext context, boolean enable) {
        updateExtensions (context, enable);
        updateVault (context, enable);
        updateSecurityDomain (context, enable);
        updateOverlordConfigSubsystem (context, enable);
        updateOverlordDeploySubsystem (context, enable);
    }

    @Override
    public void applyDomainConfigChange(ConfigContext context, boolean enable) {
         applyStandaloneConfigChange(context, enable);
    }

    // client & server
    private void updateExtensions(ConfigContext context, boolean enable) {
        String extPrefix = "org.overlord.commons.eap.extensions.";
        String[] extensionNames = new String[2];
        extensionNames[0] = extPrefix + "config";
        extensionNames[1] = extPrefix + "deploy";

        Element extensions = context.getDocument().getRootElement().getChild("extensions", NS_DOMAIN);
        ConfigSupport.assertExists(extensions, "Did not find the <extensions> element");
        for (String extName : extensionNames) {
            Element element = ConfigSupport.findElementWithAttributeValue(extensions, "extension", NS_DOMAIN, "module", extName);
            if (enable && element == null) {
                extensions.addContent(new Text("    "));
                extensions.addContent(new Element("extension", NS_DOMAIN).setAttribute("module", extName));
                extensions.addContent(new Text("\n    "));
            }
            if (!enable && element != null) {
                element.getParentElement().removeContent(element);
            }
        }
    }

    // client & server
    private void updateVault (ConfigContext context, boolean enable) {
        Element root = context.getDocument().getRootElement();
        Element vault = root.getChild ("vault", NS_DOMAIN);

        if (enable && vault == null) {
            // Insert the vault section before the management section.
            Element management = root.getChild("management", NS_DOMAIN);

            ConfigSupport.assertExists (management, "Did not find the <management> element");
            int managementIndex = root.indexOf (management);
            URL resource = RTGovConfigPlugin.class.getResource ("/common/vault.xml");
            root.addContent (managementIndex - 1, ConfigSupport.loadElementFrom (resource));
            root.addContent (managementIndex - 1, new Text ("\n"));
        }

        if (!enable && vault != null) {
            root.removeContent (vault);
        }
    }

    // client & server
    private void updateSecurityDomain (ConfigContext context, boolean enable) {
        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument (), NS_DOMAIN);
        for (Element profile : profiles) {
            Element security = profile.getChild ("subsystem", NS_SECURITY);
            ConfigSupport.assertExists (security, "Did not find the security subsystem");
            Element domains = security.getChild ("security-domains", NS_SECURITY);
            ConfigSupport.assertExists(domains, "Did not find the <security-domains> element");

            String[] domainNames = { "overlord-idp", "overlord-sp", "overlord-jaxrs" };
            for (String domainName : domainNames) {
                Element domain = ConfigSupport.findElementWithAttributeValue (domains, "security-domain", NS_SECURITY, "name", domainName);
                if (enable && domain == null) {
                    URL resource = RTGovConfigPlugin.class.getResource ("/common/security-" + domainName + ".xml");
                    domains.addContent (new Text("    "));
                    domains.addContent (ConfigSupport.loadElementFrom (resource));
                    domains.addContent (new Text ("\n            "));
                }

                if (!enable && domain != null) {
                    domain.getParentElement().removeContent (domain);
                }
            }
        }
    }

    protected abstract void updateOverlordConfigSubsystem (ConfigContext context, boolean enable);

    protected abstract void updateOverlordDeploySubsystem (ConfigContext context, boolean enable);

    protected void updateSubsystem (ConfigContext context, boolean enable, Namespace ns, String resourceName) {
        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument(), NS_DOMAIN);
        for (Element profile : profiles) {
            Element element = profile.getChild ("subsystem", ns);
            if (enable && element == null) {
                URL resource = RTGovConfigPlugin.class.getResource (resourceName);
                profile.addContent (new Text ("    "));
                profile.addContent (ConfigSupport.loadElementFrom (resource));
                profile.addContent (new Text ("\n    "));
            }
            if (!enable && element != null) {
                element.getParentElement().removeContent (element);
            }
        }
    }
}
