/*
 * Copyright 2017 Red Hat Inc. and/or its affiliates and other contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.switchyard.tests;

import javax.xml.namespace.QName;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.StringReader;

import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.switchyard.Exchange;
import org.switchyard.Message;
import org.switchyard.MockDomain;
import org.switchyard.MockHandler;
import org.switchyard.Scope;
import org.switchyard.ServiceReference;
import org.switchyard.common.lang.Strings;
import org.switchyard.runtime.util.ExchangeFormatter;

/**
 * ExchangeFormatterTest.
 *
 */
public class ExchangeFormatterTest {
    private static final String INDENT = System.getProperty("line.separator");
    private MockDomain _domain;

    @Before
    public void setUp() throws Exception {
        _domain = new MockDomain();
    }

    @Test
    public void testStringFormatBody() {
        final QName serviceName = new QName("exchangeFormatter");
        final String input = new String("Hello");
        MockHandler provider = new MockHandler();
        ServiceReference service = _domain.createInOnlyService(serviceName, provider);

        Exchange exchange = service.createExchange();
        Message msg = exchange.createMessage().setContent(input);

        msg.setContent(input);
        exchange.send(msg);

        String result = ExchangeFormatter.format(exchange, true);
        String expectedResult = indent(0) +  "Message Content -> " + indent(0) + input;
        if (result.contains(expectedResult)) {
        } else {
            Assert.fail("Expected to see [" + expectedResult + "] in result, result was [" + result + "]");
        }
    }

    @Test
    public void testStreamFormatBody() throws Exception {
        final QName serviceName = new QName("exchangeFormatter");
        final String input = new String("Hello");
        MockHandler provider = new MockHandler();
        ServiceReference service = _domain.createInOnlyService(serviceName, provider);

        Exchange exchange = service.createExchange();
        Message msg = exchange.createMessage().setContent(input);
        msg.getContext().setProperty("CamelCharsetName", "UTF-8");
        exchange.getContext().setProperty("CamelCharsetName", "UTF-8", Scope.EXCHANGE);
        msg.setContent(new ByteArrayInputStream(input.getBytes("UTF-8")));
        exchange.send(msg);

        String result = ExchangeFormatter.format(exchange, true);
        String expectedResult = indent(0) +  "Message Content -> " + indent(0) + input;
        if (result.contains(expectedResult)) {
        } else {
            Assert.fail("Expected to see [" + expectedResult + "] in result, result was [" + result + "]");
        }
    }

    @Test
    public void testReaderFormatBody() throws Exception {
        final QName serviceName = new QName("exchangeFormatter");
        final String input = new String("Hello");
        MockHandler provider = new MockHandler();
        ServiceReference service = _domain.createInOnlyService(serviceName, provider);

        Exchange exchange = service.createExchange();
        Message msg = exchange.createMessage().setContent(input);
        msg.getContext().setProperty("CamelCharsetName", "UTF-8");
        exchange.getContext().setProperty("CamelCharsetName", "UTF-8", Scope.EXCHANGE);

        msg.setContent(new StringReader(input));
        exchange.send(msg);

        String result = ExchangeFormatter.format(exchange, true);
        String expectedResult = indent(0) +  "Message Content -> " + indent(0) + input;
        if (result.contains(expectedResult)) {
        } else {
            Assert.fail("Expected to see [" + expectedResult + "] in result, result was [" + result + "]");
        }
    }

    private static String indent(int column) {
        return INDENT + Strings.repeat("    ", column);
    }

}
