/*
 * #%L
 * Wildfly Camel :: Testsuite
 * %%
 * Copyright (C) 2013 - 2014 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

package org.wildfly.camel.test.paho;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.camel.CamelContext;
import org.apache.camel.PollingConsumer;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.impl.DefaultCamelContext;
import org.eclipse.paho.mqttv5.client.IMqttToken;
import org.eclipse.paho.mqttv5.client.MqttCallback;
import org.eclipse.paho.mqttv5.client.MqttClient;
import org.eclipse.paho.mqttv5.client.MqttConnectionOptions;
import org.eclipse.paho.mqttv5.client.MqttDisconnectResponse;
import org.eclipse.paho.mqttv5.client.persist.MemoryPersistence;
import org.eclipse.paho.mqttv5.common.MqttException;
import org.eclipse.paho.mqttv5.common.MqttMessage;
import org.eclipse.paho.mqttv5.common.packet.MqttProperties;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.as.arquillian.api.ServerSetup;
import org.jboss.as.arquillian.api.ServerSetupTask;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wildfly.camel.test.common.utils.AvailablePortFinder;
import org.wildfly.camel.test.common.utils.TestUtils;
import org.wildfly.camel.test.dockerjava.DockerManager;
import org.wildfly.extension.camel.CamelAware;

@CamelAware
@RunWith(Arquillian.class)
@ServerSetup({ PahoMQTT5IntegrationTest.BrokerSetup.class })
public class PahoMQTT5IntegrationTest {

    final static Logger LOG = LoggerFactory.getLogger(PahoMQTT5IntegrationTest.class);

    static class BrokerSetup implements ServerSetupTask {

        static final int PORT = 1883;
        static final String DOCKER_HOST;
        static {
            String dockerHost = "127.0.0.1";
            try {
                dockerHost = TestUtils.getDockerHost();
            } catch (Exception e) {
                LOG.error(e.getMessage(), e);
            }
            DOCKER_HOST = dockerHost;
        }
        static final String TCP_CONNECTION = "tcp://" + DOCKER_HOST + ":" + PORT;
        static final String TEST_TOPIC = "ComponentTestTopic";

        private DockerManager dockerManager;

        @Override
        public void setup(ManagementClient managementClient, String containerId) throws Exception {
            dockerManager = new DockerManager()
                    .createContainer("eclipse-mosquitto:1.6.14")
                    .withName("mosquitto")
                    .withPortBindings(PORT)
                    .startContainer();

            if (!dockerManager
                    .withAwaitLogMessage("mosquitto version 1.6.14 running")
                    .awaitCompletion(30, TimeUnit.SECONDS)) {
                throw new IllegalStateException("Mosquitto server could not be started in time");
            }

        }

        @Override
        public void tearDown(ManagementClient managementClient, String containerId) throws Exception {
            if (dockerManager != null) {
                dockerManager.removeContainer("mosquitto");
            }
        }
    }

    @Deployment
    public static JavaArchive deployment() {
        JavaArchive archive = ShrinkWrap.create(JavaArchive.class, "mqtt-tests");
        archive.addClasses(TestUtils.class);
        archive.addAsResource(new StringAsset(BrokerSetup.TCP_CONNECTION), "tcp-connection");
        return archive;
    }

    @Test
    public void testPahoConsumer() throws Exception {

        String conUrl = TestUtils.getResourceValue(getClass(), "/tcp-connection");

        CamelContext camelctx = new DefaultCamelContext();
        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() {
                from("paho-mqtt5:" + BrokerSetup.TEST_TOPIC + "?brokerUrl=" + conUrl).
                transform(body().prepend("Hello ")).to("seda:end");
            }
        });

        camelctx.start();
        try {
            PollingConsumer consumer = camelctx.getEndpoint("seda:end").createPollingConsumer();
            consumer.start();

            MqttClient client = null;
            try {
                client = new MqttClient(conUrl, "MqttClient", new MemoryPersistence());
                MqttConnectionOptions opts = new MqttConnectionOptions();
                opts.setCleanStart(true);
                client.connect(opts);
                MqttMessage message = new MqttMessage("Kermit".getBytes());
                message.setQos(2);
                client.publish(BrokerSetup.TEST_TOPIC, message);
            } finally {
                if (client != null) {
                    client.disconnect();
                }
            }
            String result = consumer.receive(3000).getIn().getBody(String.class);
            Assert.assertEquals("Hello Kermit", result);
        } finally {
            camelctx.stop();
        }
    }

    @Test
    public void testMQTTProducer() throws Exception {

        String conUrl = TestUtils.getResourceValue(getClass(), "/tcp-connection");

        CamelContext camelctx = new DefaultCamelContext();
        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() {
                from("direct:start").
                transform(body().prepend("Hello ")).
                to("paho-mqtt5:" + BrokerSetup.TEST_TOPIC + "?brokerUrl=" + conUrl);
            }
        });

        camelctx.start();
        try {
            MqttClient client = new MqttClient(conUrl, "MqttClient", new MemoryPersistence());
            MqttConnectionOptions opts = new MqttConnectionOptions();
            opts.setCleanStart(true);
            client.connect(opts);
            client.subscribe(BrokerSetup.TEST_TOPIC, 2);

            final List<String> result = new ArrayList<>();
            final CountDownLatch latch = new CountDownLatch(1);

            client.setCallback(new MqttCallback() {

                @Override
                public void disconnected(MqttDisconnectResponse disconnectResponse) {
                }

                @Override
                public void mqttErrorOccurred(MqttException exception) {
                }

                @Override
                public void messageArrived(String topic, MqttMessage message) {
                    result.add(new String (message.getPayload()));
                    latch.countDown();
                }

                @Override
                public void deliveryComplete(IMqttToken token) {
                }

                @Override
                public void connectComplete(boolean reconnect, String serverURI) {

                }

                @Override
                public void authPacketArrived(int reasonCode, MqttProperties properties) {
                }});

            ProducerTemplate producer = camelctx.createProducerTemplate();
            producer.asyncSendBody("direct:start", "Kermit");

            Assert.assertTrue(latch.await(10, TimeUnit.SECONDS));
            Assert.assertEquals("One message", 1, result.size());
            Assert.assertEquals("Hello Kermit", result.get(0));
        } finally {
            camelctx.stop();
        }
    }
}
