package templates

// TestRunScript is a simple test script that verifies the S2I image.
const TestRunScript = `#!/bin/bash
#
# The 'run' performs a simple test that verifies the S2I image.
# The main focus here is to exercise the S2I scripts.
#
# For more information see the documentation:
# https://github.com/openshift/source-to-image/blob/master/docs/builder_image.md
#
# IMAGE_NAME specifies a name of the candidate image used for testing.
# The image has to be available before this script is executed.
#
IMAGE_NAME=${IMAGE_NAME-{{.ImageName}}-candidate}

# Determining system utility executables (darwin compatibility check)
READLINK_EXEC="readlink"
MKTEMP_EXEC="mktemp"
if [[ "$OSTYPE" =~ 'darwin' ]]; then
  ! type -a "greadlink" &>"/dev/null" || READLINK_EXEC="greadlink"
  ! type -a "gmktemp" &>"/dev/null" || MKTEMP_EXEC="gmktemp"
fi

test_dir="$($READLINK_EXEC -zf $(dirname "${BASH_SOURCE[0]}"))"
image_dir=$($READLINK_EXEC -zf ${test_dir}/..)
cid_file=$($MKTEMP_EXEC -u --suffix=.cid)

# Since we built the candidate image locally, we don't want S2I to attempt to pull
# it from Docker hub
s2i_args="--pull-policy=never --loglevel=2"

# Port the image exposes service to be tested
test_port=8080

image_exists() {
  docker inspect $1 &>/dev/null
}

container_exists() {
  image_exists $(cat $cid_file)
}

container_ip() {
  if [ ! -z "$DOCKER_HOST" ] && [[ "$OSTYPE" =~ 'darwin' ]]; then
    docker-machine ip
  else
    docker inspect --format="{{"{{"}} .NetworkSettings.IPAddress {{"}}"}}" $(cat $cid_file)
  fi
}

container_port() {
  if [ ! -z "$DOCKER_HOST" ] && [[ "$OSTYPE" =~ 'darwin' ]]; then
    docker inspect --format="{{"{{"}}(index .NetworkSettings.Ports \"$test_port/tcp\" 0).HostPort{{"}}"}}" "$(cat "${cid_file}")"
  else
    echo $test_port
  fi
}

run_s2i_build() {
  s2i build --incremental=true ${s2i_args} file://${test_dir}/test-app ${IMAGE_NAME} ${IMAGE_NAME}-testapp
}

prepare() {
  if ! image_exists ${IMAGE_NAME}; then
    echo "ERROR: The image ${IMAGE_NAME} must exist before this script is executed."
    exit 1
  fi
  # s2i build requires the application is a valid 'Git' repository
  pushd ${test_dir}/test-app >/dev/null
  git init
  git config user.email "build@localhost" && git config user.name "builder"
  git add -A && git commit -m "Sample commit"
  popd >/dev/null
  run_s2i_build
}

run_test_application() {
  docker run --rm --cidfile=${cid_file} -p ${test_port} ${IMAGE_NAME}-testapp
}

cleanup() {
  if [ -f $cid_file ]; then
    if container_exists; then
      docker stop $(cat $cid_file)
    fi
  fi
  if image_exists ${IMAGE_NAME}-testapp; then
    docker rmi ${IMAGE_NAME}-testapp
  fi
}

check_result() {
  local result="$1"
  if [[ "$result" != "0" ]]; then
    echo "S2I image '${IMAGE_NAME}' test FAILED (exit code: ${result})"
    cleanup
    exit $result
  fi
}

wait_for_cid() {
  local max_attempts=10
  local sleep_time=1
  local attempt=1
  local result=1
  while [ $attempt -le $max_attempts ]; do
    [ -f $cid_file ] && break
    echo "Waiting for container to start..."
    attempt=$(( $attempt + 1 ))
    sleep $sleep_time
  done
}

test_usage() {
  echo "Testing 's2i usage'..."
  s2i usage ${s2i_args} ${IMAGE_NAME} &>/dev/null
}

test_connection() {
  echo "Testing HTTP connection (http://$(container_ip):$(container_port))"
  local max_attempts=10
  local sleep_time=1
  local attempt=1
  local result=1
  while [ $attempt -le $max_attempts ]; do
    echo "Sending GET request to http://$(container_ip):$(container_port)/"
    response_code=$(curl -s -w %{http_code} -o /dev/null http://$(container_ip):$(container_port)/)
    status=$?
    if [ $status -eq 0 ]; then
      if [ $response_code -eq 200 ]; then
        result=0
      fi
      break
    fi
    attempt=$(( $attempt + 1 ))
    sleep $sleep_time
  done
  return $result
}

# Build the application image twice to ensure the 'save-artifacts' and
# 'restore-artifacts' scripts are working properly
prepare
run_s2i_build
check_result $?

# Verify the 'usage' script is working properly
test_usage
check_result $?

# Verify that the HTTP connection can be established to test application container
run_test_application &

# Wait for the container to write its CID file
wait_for_cid

test_connection
check_result $?

cleanup

`

// Makefile contains a sample Makefile which can build and test a container.
const Makefile = `
IMAGE_NAME = {{.ImageName}}

.PHONY: build
build:
	docker build -t $(IMAGE_NAME) .

.PHONY: test
test:
	docker build -t $(IMAGE_NAME)-candidate .
	IMAGE_NAME=$(IMAGE_NAME)-candidate test/run
`

// Index contains a sample index.html file.
const Index = `
<!doctype html>
<html>
	<head>
		<title>Hello World!</title>
	</head>
	<body>
		<h1>Hello World!</h1>
	</body>
</html>
 `
