# Reactive Code Planner Design Document

## Overview

The **Code Planning Loop** is a reactive agent-based system designed to automate the process of identifying, prioritizing, and resolving tasks in a software repository. The system uses validators to assess repository state and taskrunners to handle resolution. It operates on a git-based virtual file system (VFS) to track changes and manage repository states, allowing for potential rollback and historical analysis.

When a task is resolved, the system reruns validators to detect any new tasks. These new tasks are added as children to the original task at the same priority level, ensuring that they are fully resolved before moving to the next task. Additionally, if a task is incidentally resolved during the resolution of another task (e.g., fixing a dependency resolves a related error), this is detected, and the resolved task is automatically removed from the queue.

The current implementation includes specific validator/taskrunner pairs for resolving Maven compiler errors, static analysis issues, and dependency management. The system is designed to be extensible, supporting integration with external tools like IDEs.

## Architecture

### 1. **Validators**

Validators analyze the repository and generate tasks based on the issues they identify. Current validators include:

- **MavenCompileStep**: Detects and reports Maven compiler errors.
- **AnalyzerLSPStep**: Uses the Analyzer LSP server to run static analysis rules and find code issues.

Validators track the current set of issues in the repository and report them to the task manager.

### 2. **Task Management System**

The **Task Manager** is responsible for managing tasks identified by validators. Tasks are organized into a priority queue, with seeded tasks automatically assigned the highest priority. The task manager enforces a depth-first execution strategy, ensuring child tasks are resolved before their parent tasks are considered complete. Key components of the task management system include:

- **Task Prioritization**: Tasks are assigned priority levels, with seeded tasks assigned priority 0 (highest).
- **Task Stacks**: Tasks are stored in stacks by priority level, allowing for ordered execution.
- **Task Resolution & Queue Management**: If a task is resolved incidentally due to another task's completion, it is automatically detected and removed from the queue. New tasks discovered after resolving a task are added as children at the same priority level to ensure complete resolution.

External tools can interact with the task manager to seed high-priority tasks and query tasks based on priority levels, making it adaptable for integration with IDEs or other external applications.

### 3. **Taskrunners**

Taskrunners are responsible for resolving tasks generated by the validators. Upon receiving a task, a taskrunner checks whether it can handle that task and uses an AI agent-based workflow to perform the resolution. Current taskrunners include:

- **MavenCompilerTaskRunner**: Resolves Maven compilation issues identified by the MavenCompileStep.
- **AnalyzerTaskRunner**: Resolves static analysis findings generated by the AnalyzerLSPStep.
- **DependencyTaskRunner**: Handles Maven dependency-related issues by leveraging an LLM to modify dependencies in the `pom.xml` file as needed and committing changes through the **RepositoryContextManager**. This allows it to handle complex dependency issues and ensure that the project can build successfully.

### 4. **RepositoryContextManager (RCM)**

The **RCM** manages repository write operations using a **git-based VFS** middleware. This approach allows changes to be written to a temporary repository rather than directly modifying the project files on disk. Key functionalities include:

- **Tracking Changes**: All modifications are tracked in a temporary repository, allowing for version control.
- **Rollback Support**: The git VFS provides the infrastructure for rolling back changes, though user-accessible rollback functionality is planned for a future release.
- **Reflection Agent**: This agent evaluates the quality of a change before it is written to the repository. If the change is deemed unsatisfactory, the reflection agent may prevent it from being committed.

### 5. **Task Lifecycle**

The lifecycle of a task is as follows:

1. **Validation**: Validators assess the repository and generate tasks based on the issues they detect.
2. **Task Prioritization**: Tasks are added to the priority queue, with seeded tasks assigned the highest priority.
3. **Resolution**: Taskrunners resolve tasks using AI-based workflows and commit changes to the repository.
4. **Revalidation**: Validators are rerun to detect new tasks after each resolution. These new tasks are added as children of the resolved task, at the same priority level. If a task is resolved indirectly due to another task's resolution, it is automatically removed from the queue.
5. **Task Completion**: Once all tasks (including child tasks) are resolved, the parent task is marked as complete.

### 6. **Seeding Tasks**

External tools, such as IDEs, can interact with the task manager by **seeding tasks**. Seeded tasks receive the highest priority (priority 0) and are processed before any other tasks. This feature allows external tools to drive the task resolution process by highlighting critical issues that must be resolved first.

Additionally, systems can query the task manager to retrieve tasks of a certain priority or higher, providing external tools with control over which tasks should be worked on at any given time.

### 7. **Task Results and Feedback**

Upon resolving a task, the taskrunner supplies the result back to the task manager. Task results include:

- **Modified Files**: A list of files that were changed during the resolution.
- **Errors Encountered**: Any errors encountered during the resolution process.

Based on the result, the task manager continues processing or raises new tasks for revalidation.

## API Overview for External Tools

### Integration Flow for IDEs

The Code Planning Loop API provides a simple yet powerful interface for interacting with the system. External tools such as IDEs can use this API to seed tasks, retrieve tasks by priority, and monitor task progress. Here are the key components:

A typical flow for IDE integration would be as follows:

1. **Task Seeding**: The IDE requests a task to be processed, which is added as a high-priority task.
2. **Task Processing**: The task is processed, and any new issues detected are added as children until all related tasks are fully resolved.&#x20;
3. **Full Diff Evaluation**: Once all tasks are resolved, the set of changes is sent back to the IDE for evaluation, providing a complete understanding of the modifications made.

### 1. **Task Seeding**

External tools can seed tasks into the task manager, which automatically assigns the highest priority (priority 0) to ensure these tasks are processed first.

### 2. **Retrieving Tasks by Priority**

External tools can request tasks of a specific priority or higher, allowing them to focus on critical tasks without overwhelming the system with lower-priority work.

### 3. **Tracking Task Results**

Once a task is resolved, external tools can track the results, including any changes made and errors encountered. This information can be fed back into the external system to inform users or developers about task resolution.

### 4. **Commit and Version Control via RCM**

The **RepositoryContextManager** ensures all changes are tracked in a git-based VFS, allowing external tools to potentially roll back changes in the future. Though rollback functionality is not currently enabled for users, the system tracks changes in a temporary repository, ensuring a clear history of modifications.

## Future Enhancements

Planned enhancements to the Code Planning Loop include:

- **JSON-RPC Based Driver**: A JSON-RPC based driver for all functionality, providing an API that allows the IDE to drive all behavior.

- **User-Accessible Rollbacks**: The ability for users to roll back changes via the git-based VFS middleware.

- **Expanded Validator and Taskrunner Support**: Integration of additional validators and taskrunners for handling more diverse types of repository issues.

- **Tighter IDE Integration**: External tools will be able to interact more seamlessly with the task manager, receiving detailed feedback on task execution and resolution.

## Conclusion

The **Code Planning Loop** is a powerful, flexible system for managing repository tasks and issues using AI-driven taskrunners and validators. Its ability to integrate with external tools, such as IDEs, makes it a valuable component for ensuring code quality and repository consistency. With planned expansions for rollback functionality and additional taskrunners, the system will continue to evolve and provide greater control over repository management.
