import functools
import os
from dataclasses import dataclass
from pathlib import Path


@dataclass(init=False)
class Language:
    name: str
    extensions: set[str]
    exact_names: set[str]
    aliases: list[str]

    def __init__(
        self, name: str, extensions: str | list[str] | None, aliases: str | list[str]
    ) -> None:
        if isinstance(extensions, str):
            extensions = [e.strip() for e in extensions.split(",")]
        elif extensions is None:
            extensions = []

        if isinstance(aliases, str):
            aliases = [a.strip() for a in aliases.split(",")]

        self.name = name
        self.aliases = aliases

        self.extensions: set[str] = set()
        self.exact_names: set[str] = set()

        for ext in extensions:
            if ext.startswith("*."):
                self.extensions.add(ext)
            else:
                self.exact_names.add(ext)


@functools.cache
def get_exact_name_map() -> dict[str, Language]:
    result: dict[str, Language] = {}
    for language in LANGUAGES:
        for name in language.exact_names:
            if name in result:
                raise ValueError(
                    f"Error adding to name map: {language} collides with {result[name]}"
                )
            else:
                result[name] = language

    return result


@functools.cache
def get_extension_map() -> dict[str, Language]:
    result: dict[str, Language] = {}
    for language in LANGUAGES:
        for ext in language.extensions:
            if ext in result:
                raise ValueError(
                    f"Error adding to extension map: {language} collides with {result[ext]}"
                )
            else:
                result[ext] = language

    return result


def guess_language(code: str, filename: str | None = None) -> str:
    if filename is None:
        return ""

    exact_name_map = get_exact_name_map()
    extension_map = get_extension_map()
    filename = os.path.basename(filename)

    if filename in exact_name_map:
        return exact_name_map[filename].aliases[0]

    ext = "*" + Path(filename).suffix
    if ext in extension_map:
        return extension_map[ext].aliases[0]

    return ""


# Sourced from https://pygments.org/languages/
# cSpell:disable
LANGUAGES = [
    Language("ABAP", "*.abap, *.ABAP", "abap"),
    Language("ABNF", "*.abnf", "abnf"),
    Language("ActionScript", "*.as", "actionscript, as"),
    # Language("ActionScript 3","*.as","actionscript3, as3"),
    Language("Ada", "*.adb, *.ads, *.ada", "ada, ada95, ada2005"),
    Language("ADL", "*.adl, *.adls, *.adlf, *.adlx", "adl"),
    Language("Agda", "*.agda", "agda"),
    Language("Aheui", "*.aheui", "aheui"),
    Language("Alloy", "*.als", "alloy"),
    Language("AmbientTalk", "*.at", "ambienttalk, ambienttalk/2, at"),
    Language("AMDGPU", "*.isa", "amdgpu"),
    Language("Ampl", "*.run", "ampl"),
    Language("Angular2", None, "ng2"),
    Language("ANSYS parametric design language", "*.ans", "ansys, apdl"),
    # Language("ANTLR",None,"antlr"),
    # Language("ANTLR With ActionScript Target","*.G, *.g","antlr-actionscript, antlr-as"),
    # Language("ANTLR With C# Target","*.G, *.g","antlr-csharp, antlr-c#"),
    # Language("ANTLR With CPP Target","*.G, *.g","antlr-cpp"),
    # Language("ANTLR With Java Target","*.G, *.g","antlr-java"),
    # Language("ANTLR With ObjectiveC Target","*.G, *.g","antlr-objc"),
    # Language("ANTLR With Perl Target","*.G, *.g","antlr-perl"),
    # Language("ANTLR With Python Target","*.G, *.g","antlr-python"),
    # Language("ANTLR With Ruby Target","*.G, *.g","antlr-ruby, antlr-rb"),
    Language(
        "ApacheConf",
        ".htaccess, apache.conf, apache2.conf",
        "apacheconf, aconf, apache",
    ),
    Language("APL", "*.apl, *.aplf, *.aplo, *.apln, *.aplc, *.apli, *.dyalog", "apl"),
    Language("AppleScript ", "*.applescript", "applescript"),
    Language("Arduino", "*.ino", "arduino"),
    Language("Arrow", "*.arw", "arrow"),
    Language("Arturo", "*.art", "arturo, art"),
    Language(
        "ASCII armored",
        "*.asc, *.pem, iddsa, idecdsa, idecdsask, ided25519, ided25519sk, idrsa",
        "asc, pem",
    ),
    Language("ASN.1", "*.asn1", "asn1"),
    Language("AspectJ", "*.aj", "aspectj"),
    Language("aspx-cs", "*.aspx, *.asax, *.ascx, *.ashx, *.asmx, *.axd", "aspx-cs"),
    # Language("aspx-vb","*.aspx, *.asax, *.ascx, *.ashx, *.asmx, *.axd","aspx-vb"),
    Language("Asymptote", "*.asy", "asymptotfe, asy"),
    Language("Augeas", "*.aug", "augeas"),
    Language("autohotkey", "*.ahk, *.ahkl", "autohotkey, ahk"),
    Language("AutoIt", "*.au3", "autoit"),
    Language("Awk", "*.awk", "awk, gawk, mawk, nawk"),
    Language("BARE", "*.bare", "bare"),
    Language("Base Makefile ", None, "basemake"),
    Language(
        "Bash",
        "*.sh, *.ksh, *.bash, *.ebuild, *.eclass, *.exheres-0, *.exlib, *.zsh, .bashrc, bashrc, .bash\\*, bash\\*, zshrc, .zshrc, .kshrc, kshrc, PKGBUILD",
        "bash, sh, ksh, zsh, shell, openrc",
    ),
    Language("Bash Session", "*.sh-session, *.shell-session", "console, shell-session"),
    Language("Batchfile", "*.bat, *.cmd", "batch, bat, dosbatch, winbatch"),
    Language("BBC Basic", "*.bbc", "bbcbasic"),
    Language("BBCode", None, "bbcode"),
    Language("BC", "*.bc", "bc"),
    # Language("Bdd","*.feature","bdd"),
    Language("Befunge", "*.befunge", "befunge"),
    Language("Berry", "*.be", "berry, be"),
    Language("BibTeX", "*.bib", "bibtex, bib"),
    Language("BlitzBasic", "*.bb, *.decls", "blitzbasic, b3d, bplus"),
    Language("BlitzMax", "*.bmx", "blitzmax, bmax"),
    Language("Blueprint", "*.blp", "blueprint"),
    Language("BNF ", "*.bnf", "bnf"),
    Language("Boa", "*.boa", "boa"),
    Language("Boo", "*.boo", "boo"),
    Language("Boogie", "*.bpl", "boogie"),
    Language("BQN", "*.bqn", "bqn"),
    # Language("Brainfuck","*.bf, *.b","brainfuck, bf"),
    Language("BST", "*.bst", "bst, bst-pybtex"),
    # Language("BUGS ","*.bug","bugs, winbugs, openbugs"),
    Language("C", "*.c, *.h, *.idc, *.x[bp]m", "c"),
    Language("C#", "*.cs", "csharp, c#, cs"),
    Language(
        "C++",
        "*.cpp, *.hpp, *.c++, *.h++, *.cc, *.hh, *.cxx, *.hxx, *.C, *.H, *.cp, *.CPP, *.tpp",
        "cpp, c++",
    ),
    Language("c-objdump", "*.c-objdump", "c-objdump"),
    # Language("ca65 assembler","*.s","ca65"),
    Language("cADL ", "*.cadl", "cadl"),
    Language("CAmkES", "*.camkes, *.idl4", "camkes, idl4"),
    Language("Cap’n Proto", "*.capnp", "capnp"),
    Language("CapDL ", "*.cdl", "capdl"),
    Language("Carbon", "*.carbon", "carbon"),
    # Language("CBM BASIC V2 ","*.bas","cbmbas"),
    Language("CDDL", "*.cddl", "cddl"),
    Language("Ceylon", "*.ceylon", "ceylon"),
    Language("CFEngine3", "*.cf", "cfengine3, cf3"),
    Language("cfstatement", None, "cfs"),
    Language("ChaiScript", "*.chai", "chaiscript, chai"),
    Language("Chapel", "*.chpl", "chapel, chpl"),
    Language("Charmci", "*.ci", "charmci"),
    Language("Cheetah", "*.tmpl, *.spt", "cheetah, spitfire"),
    Language("Cirru", "*.cirru", "cirru"),
    Language("Clay", "*.clay", "clay"),
    Language("Clean", "*.icl, *.dcl", "clean"),
    Language("Clojure", "*.clj, *.cljc", "clojure, clj"),
    Language("ClojureScript", "*.cljs", "clojurescript, cljs"),
    Language("CMake", "*.cmake, CMakeLists.txt", "cmake"),
    Language("COBOL", "*.cob, *.COB, *.cpy, *.CPY", "cobol"),
    Language("COBOLFree", "*.cbl, *.CBL", "cobolfree"),
    Language("CodeQL", "*.ql, *.qll", "codeql, ql"),
    Language("CoffeeScript", "*.coffee", "coffeescript, coffee-script, coffee"),
    Language("Coldfusion CFC", "*.cfc", "cfc"),
    Language("Coldfusion HTML", "*.cfm, *.cfml", "cfm"),
    Language("COMAL-80", "*.cml, *.comal", "comal, comal80"),
    Language("Common Lisp", "*.cl, *.lisp", "common-lisp, cl, lisp"),
    # Language("Component Pascal","*.cp, *.cps","componentpascal, cp"),
    # Language("Coq","*.v","coq"),
    # Language("cplint","*.ecl, *.prolog, *.pro, *.pl, *.P, *.lpad, *.cpl","cplint"),
    Language(
        "cpp-objdump",
        "*.cpp-objdump, *.c++-objdump, *.cxx-objdump",
        "cpp-objdump, c++-objdumb, cxx-objdump",
    ),
    Language("CPSA", "*.cpsa", "cpsa"),
    Language("Crmsh", "*.crmsh, *.pcmk", "crmsh, pcmk"),
    Language("Croc", "*.croc", "croc"),
    Language("Cryptol", "*.cry", "cryptol, cry"),
    Language("Crystal", "*.cr", "cr, crystal"),
    Language("Csound Document", "*.csd", "csound-document, csound-csd"),
    Language("Csound Orchestra", "*.orc, *.udo", "csound, csound-orc"),
    Language("Csound Score", "*.sco", "csound-score, csound-sco"),
    Language("CSS", "*.css", "css"),
    # Language("CSS+Django/Jinja","*.css.j2, *.css.jinja2, *.css","css+django, css+jinja"),
    # Language("CSS+Genshi Text","*.css","css+genshitext, css+genshi"),
    # Language("CSS+Lasso","*.css","css+lasso"),
    # Language("CSS+Mako",None,"css+mako"),
    # Language("CSS+mozpreproc ","*.css.in","css+mozpreproc"),
    # Language("CSS+Myghty",None,"css+myghty"),
    # Language("CSS+PHP","*.css","css+php"),
    # Language("CSS+Ruby","*.css","css+ruby, css+erb"),
    # Language("CSS+Smarty","*.css, *.tpl","css+smarty"),
    # Language("CSS+UL4","*.cssul4","css+ul4"),
    Language("CUDA ", "*.cu, *.cuh", "cuda, cu"),
    Language("Cypher ", "*.cyp, *.cypher", "cypher"),
    Language("Cython", "*.pyx, *.pxd, *.pxi", "cython, pyx, pyrex"),
    Language("D", "*.d, *.di", "d"),
    Language("d-objdump", "*.d-objdump", "d-objdump"),
    Language("Darcs Patch", "*.dpatch, *.darcspatch", "dpatch"),
    Language("Dart", "*.dart", "dart"),
    Language("DASM16", "*.dasm16, *.dasm", "dasm16"),
    Language("Dax", "*.dax", "dax"),
    Language("Debian Control file ", "control", "debcontrol, control"),
    Language(
        "Debian Sourcelist", "sources.list", "debsources, sourceslist, sources.list"
    ),
    Language("Debian Sources file", "*.sources", "debian.sources"),
    Language("Delphi", "*.pas, *.dpr", "delphi, pas, pascal, objectpascal"),
    Language("Desktop file", "*.desktop", "desktop"),
    Language("Devicetree", "*.dts, *.dtsi", "devicetree, dts"),
    Language("dg", "*.dg", "dg"),
    Language("Diff", "*.diff, *.patch", "diff, udiff"),
    Language("Django/Jinja", None, "django, jinja"),
    Language("Docker", "Dockerfile, *.docker", "docker, dockerfile"),
    Language("DTD ", "*.dtd", "dtd"),
    Language("Duel", "*.duel, *.jbst", "duel, jbst, jsonml+bst"),
    Language("Dylan", "*.dylan, *.dyl, *.intr", "dylan"),
    Language("Dylan session", "*.dylan-console", "dylan-console, dylan-repl"),
    Language("DylanLID", "*.lid, *.hdp", "dylan-lid, lid"),
    Language("E-mail ", "*.eml", "email, eml"),
    Language("Earl Grey", "*.eg", "earl-grey, earlgrey, eg"),
    Language("Easytrieve ", "*.ezt, *.mac", "easytrieve"),
    Language("EBNF", "*.ebnf", "ebnf"),
    Language("eC", "*.ec, *.eh", "ec"),
    Language("ECL", "*.ecl", "ecl"),
    Language("Eiffel", "*.e", "eiffel"),
    Language("Elixir", "*.ex, *.eex, *.exs, *.leex", "elixir, ex, exs"),
    Language("Elixir iex session", None, "iex"),
    Language("Elm", "*.elm", "elm"),
    Language("Elpi", "*.elpi", "elpi"),
    Language("EmacsLisp", "*.el", "emacs-lisp, elisp, emacs"),
    Language("Embedded Ragel", "*.rl", "ragel-em"),
    Language("ERB", None, "erb"),
    Language("Erlang", "*.erl, *.hrl, *.es, *.escript", "erlang"),
    Language("Erlang erl session", "*.erl-sh", "erl"),
    Language("Evoque", "*.evoque", "evoque"),
    Language("execline", "*.exec", "execline"),
    Language("Ezhil", "*.n", "ezhil"),
    Language("F#", "*.fs, *.fsi, *.fsx", "fsharp, f#"),
    Language("Factor", "*.factor", "factor"),
    Language("Fancy", "*.fy, *.fancypack", "fancy, fy"),
    Language("Fantom", "*.fan", "fan"),
    Language("Felix", "*.flx, *.flxh", "felix, flx"),
    Language("Fennel", "*.fnl", "fennel, fnl"),
    Language("Fift", "*.fif", "fift, fif"),
    Language("Fish", "*.fish, *.load", "fish, fishshell"),
    Language("Flatline", None, "flatline"),
    Language("FloScript", "*.flo", "floscript, flo"),
    # Language("Forth","*.frt, *.fs","forth"),
    Language("Fortran", "*.f03, *.f90, *.F03, *.F90", "fortran, f90"),
    Language("FortranFixed", "*.f, *.F", "fortranfixed"),
    Language("FoxPro", "*.PRG, *.prg", "foxpro, vfp, clipper, xbase"),
    Language("Freefem", "*.edp", "freefem"),
    Language("FStar", "*.fst, *.fsti", "fstar"),
    Language("FunC", "*.fc, *.func", "func, fc"),
    Language("Futhark", "*.fut", "futhark"),
    Language("g-code", "*.gcode", "gcode"),
    # Language("GAP","*.g, *.gd, *.gi, *.gap","gap"),
    Language("GAP session", "*.tst", "gap-console, gap-repl"),
    Language("GAS", "*.s, *.S", "gas, asm"),
    Language("GDScript", "*.gd", "gdscript, gd"),
    # Language("Genshi","*.kid, *.xml","genshi, kid, xml+genshi, xml+kid"),
    Language("Genshi Text", None, "genshitext"),
    Language("Gettext Catalog", "*.pot, *.po", "pot, po"),
    Language("Gherkin", "*.feature", "gherkin, cucumber"),
    Language("Gleam", "*.gleam", "gleam"),
    Language("GLSL", "*.vert, *.frag, *.geo", "glsl"),
    Language("Gnuplot", "*.plot, *.plt", "gnuplot"),
    Language("Go", "*.go", "go, golang"),
    Language("Golo", "*.golo", "golo"),
    Language("GoodData-CL", "*.gdc", "gooddata-cl"),
    Language("GoogleSQL", "*.googlesql, *.googlesql.sql", "googlesql, zetasql"),
    Language("Gosu", "*.gs, *.gsx, *.gsp, *.vark", "gosu"),
    Language("Gosu Template", "*.gst", "gst"),
    Language("GraphQL", "*.graphql", "graphql"),
    Language("Graphviz", "*.gv, *.dot", "graphviz, dot"),
    Language("Groff", "*.[1-9], *.man, *.1p, *.3pm", "groff, nroff, man"),
    Language("Groovy", "*.groovy, *.gradle", "groovy"),
    Language("GSQL", "*.gsql", "gsql"),
    Language("Haml", "*.haml", "haml"),
    Language("Handlebars", None, "handlebars"),
    Language("Hare", "*.ha", "hare"),
    Language("Haskell", "*.hs", "haskell, hs"),
    Language("Haxe", "*.hx, *.hxsl", "haxe, hxsl, hx"),
    Language("Hexdump", None, "hexdump"),
    Language("HLSL ", "*.hlsl, *.hlsli", "hlsl"),
    Language("HSAIL ", "*.hsail", "hsail, hsa"),
    Language("Hspec", "*Spec.hs", "hspec"),
    Language("HTML", "*.html, *.htm, *.xhtml, *.xslt", "html"),
    Language("HTML + Angular2", "*.ng2", "html+ng2"),
    # Language("HTML+Cheetah",None,"html+cheetah, html+spitfire, htmlcheetah"),
    # Language("HTML+Django/Jinja","*.html.j2, *.htm.j2, *.xhtml.j2, *.html.jinja2, *.htm.jinja2, *.xhtml.jinja2, *.html, *.htm, *.xhtml","html+django, html+jinja, htmldjango"),
    # Language("HTML+Evoque","*.html","html+evoque"),
    # Language("HTML+Genshi","*.html, *.htm, *.xhtml","html+genshi, html+kid"),
    # Language("HTML+Handlebars","*.handlebars, *.hbs","html+handlebars"),
    # Language("HTML+Lasso","*.html, *.htm, *.xhtml, *.lasso, *.lasso[89], *.incl, *.inc, *.las","html+lasso"),
    # Language("HTML+Mako",None,"html+mako"),
    # Language("HTML+Myghty",None,"html+myghty"),
    # Language("HTML+PHP","*.phtml, *.php, *.html, *.htm, *.xhtml, *.php[345]","html+php"),
    # Language("HTML+Smarty","*.html, *.htm, *.xhtml, *.tpl","html+smarty"),
    # Language("HTML+Twig","*.twig","html+twig"),
    # Language("HTML+UL4","*.htmlul4","html+ul4"),
    # Language("HTML+Velocity","*.html, *.fhtml","html+velocity"),
    Language("HTTP", None, "http"),
    Language("Hxml", "*.hxml", "haxeml, hxml"),
    Language("Hy", "*.hy", "hylang, hy"),
    Language("Hybris", "*.hyb", "hybris"),
    Language("Icon", "*.icon, *.ICON", "icon"),
    Language("IDL", "*.pro", "idl"),
    Language("Idris", "*.idr", "idris, idr"),
    Language("Igor", "*.ipf", "igor, igorpro"),
    # Language("Inform 6","*.inf","inform6, i6"),
    # Language("Inform 6 template","*.i6t","i6t"),
    # Language("Inform 7","*.ni, *.i7x","inform7, i7"),
    Language("INI", "*.ini, *.cfg, *.inf, .editorconfig", "ini, cfg, dosini"),
    Language("Io", "*.io", "io"),
    Language("Ioke", "*.ik", "ioke, ik"),
    Language("IRC logs ", "*.weechatlog", "irc"),
    Language("Isabelle", "*.thy", "isabelle"),
    Language("J", "*.ijs", "j"),
    Language("JAGS", "*.jag, *.bug", "jags"),
    Language("Janet", "*.janet, *.jdn", "janet"),
    Language("Jasmin", "*.j", "jasmin, jasminxt"),
    Language("Java", "*.java", "java"),
    Language("Java Server Page", "*.jsp", "jsp"),
    Language("JavaScript ", "*.js, *.jsm, *.mjs, *.cjs", "javascript, js"),
    # Language("JavaScript+Cheetah",None,"javascript+cheetah, js+cheetah, javascript+spitfire, js+spitfire"),
    # Language("JavaScript+Django/Jinja","*.js.j2, *.js.jinja2, *.js","javascript+django, js+django, javascript+jinja, js+jinja"),
    # Language("JavaScript+Genshi Text","*.js","js+genshitext, js+genshi, javascript+genshitext, javascript+genshi"),
    # Language("JavaScript+Lasso","*.js","javascript+lasso, js+lasso"),
    # Language("JavaScript+Mako",None,"javascript+mako, js+mako"),
    # Language("Javascript+mozpreproc ","*.js.in","javascript+mozpreproc"),
    # Language("JavaScript+Myghty",None,"javascript+myghty, js+myghty"),
    # Language("JavaScript+PHP","*.js","javascript+php, js+php"),
    # Language("JavaScript+Ruby","*.js","javascript+ruby, js+ruby, javascript+erb, js+erb"),
    # Language("JavaScript+Smarty","*.js, *.tpl","javascript+smarty, js+smarty"),
    # Language("Javascript+UL4","*.jsul4","js+ul4"),
    Language("JCL", "*.jcl", "jcl"),
    Language("JMESPath", "*.jp", "jmespath, jp"),
    Language("JSGF", "*.jsgf", "jsgf"),
    Language("JSLT", "*.jslt", "jslt"),
    Language("JSON", "*.json, *.jsonl, *.ndjson, Pipfile.lock", "json, json-object"),
    Language("JSON-LD", "*.jsonld", "jsonld, json-ld"),
    Language("JSON5", "*.json5", "json5"),
    Language("JSONBareObject", None, ""),
    Language("Jsonnet", "*.jsonnet, *.libsonnet", "jsonnet"),
    Language("JSX", "*.jsx, *.react", "jsx, react"),
    Language("Julia", "*.jl", "julia, jl"),
    Language("Julia console", None, "jlcon, julia-repl"),
    Language("Juttle", "*.juttle", "juttle"),
    Language("K", "*.k", "k"),
    Language("Kal", "*.kal", "kal"),
    Language(
        "Kconfig",
        "Kconfig*, *Config.in*, external.in*, standard-modules.in",
        "kconfig, menuconfig, linux-config, kernel-config",
    ),
    Language("Kernel log", "*.kmsg, *.dmesg", "kmsg, dmesg"),
    Language("Koka", "*.kk, *.kki", "koka"),
    Language("Kotlin", "*.kt, *.kts", "kotlin"),
    Language("Kuin", "*.kn", "kuin"),
    Language("Kusto", "*.kql, *.kusto, .csl", "kql, kusto"),
    # Language("Lasso","*.lasso, *.lasso[89], *.incl, *.inc, *.las","lasso, lassoscript"),
    Language(
        "LDAP configuration file", ".ldaprc, ldaprc, ldap.conf", "ldapconf, ldaprc"
    ),
    Language("LDIF", "*.ldif", "ldif"),
    Language("Lean", "*.lean", "lean, lean3"),
    # Language("Lean4","*.lean","lean4"),
    Language("LessCss", "*.less", "less"),
    Language("Lighttpd configuration file", "lighttpd.conf", "lighttpd, lighty"),
    Language("LilyPond", "*.ly", "lilypond"),
    # Language("Limbo","*.b","limbo"),
    Language("liquid", "*.liquid", "liquid"),
    Language("Literate Agda ", "*.lagda", "literate-agda, lagda"),
    Language("Literate Cryptol", "*.lcry", "literate-cryptol, lcryptol, lcry"),
    Language("Literate Haskell", "*.lhs", "literate-haskell, lhaskell, lhs"),
    Language("Literate Idris", "*.lidr", "literate-idris, lidris, lidr"),
    Language("LiveScript", "*.ls", "livescript, live-script"),
    Language("LLVM", "*.ll", "llvm"),
    Language("LLVM-MIR", "*.mir", "llvm-mir"),
    Language("LLVM-MIR Body", None, "llvm-mir-body"),
    Language("Logos", "*.x, *.xi, *.xm, *.xmi", "logos"),
    Language("Logtalk", "*.lgt, *.logtalk", "logtalk"),
    Language("LSL", "*.lsl", "lsl"),
    Language("Lua", "*.lua, *.wlua", "lua"),
    Language("Luau", "*.luau", "luau"),
    Language("Macaulay2", "*.m2", "macaulay2"),
    Language(
        "Makefile",
        "*.mak, *.mk, Makefile, makefile, Makefile.*, GNUmakefile",
        "make, makefile, mf, bsdmake",
    ),
    Language("Mako", "*.mao", "mako"),
    Language("Maple", "*.mpl, *.mi, *.mm", "maple"),
    Language("MAQL ", "*.maql", "maql"),
    Language("Markdown", "*.md, *.markdown", "markdown, md"),
    Language("Mask", "*.mask", "mask"),
    # Language("Mason","*.m, *.mhtml, *.mc, *.mi, autohandler, dhandler","mason"),
    Language("Mathematica", "*.nb, *.cdf, *.nbp, *.ma", "mathematica, mma, nb"),
    Language("Matlab", "*.m", "matlab"),
    Language("Matlab session", None, "matlabsession"),
    # Language("Maxima","*.mac, *.max","maxima, macsyma"),
    Language("MCFunction", "*.mcfunction", "mcfunction, mcf"),
    Language("MCSchema ", "*.mcschema", "mcschema"),
    Language("Meson", "meson.build, mesonoptions.txt", "meson, meson.build"),
    Language("MIME", None, "mime"),
    Language("MiniD", None, "minid"),
    Language("MiniScript", "*.ms", "miniscript, ms"),
    Language("MIPS", "*.mips, *.MIPS", "mips"),
    Language("Modelica", "*.mo", "modelica"),
    Language("Modula-2", "*.def, *.mod", "modula2, m2"),
    Language("MoinMoin/Trac Wiki markup", None, "trac-wiki, moin"),
    Language("Mojo", "*.mojo, *.🔥", "mojo, 🔥"),
    Language("Monkey", "*.monkey", "monkey"),
    Language("Monte", "*.mt", "monte"),
    Language("MOOCode", "*.moo", "moocode, moo"),
    Language("MoonScript", "*.moon", "moonscript, moon"),
    Language("Mosel ", "*.mos", "mosel"),
    Language("mozhashpreproc ", None, "mozhashpreproc"),
    Language("mozpercentpreproc ", None, "mozpercentpreproc"),
    Language("MQL", "*.mq4, *.mq5, *.mqh", "mql, mq4, mq5, mql4, mql5"),
    Language("Mscgen", "*.msc", "mscgen, msc"),
    Language("MSDOS Session", None, "doscon"),
    Language("MuPAD", "*.mu", "mupad"),
    Language("MXML", "*.mxml", "mxml"),
    Language("Myghty", "*.myt, autodelegate", "myghty"),
    Language("MySQL", None, "mysql"),
    Language("NASM", "*.asm, *.ASM, *.nasm", "nasm"),
    Language("NCL", "*.ncl", "ncl"),
    # Language("Nemerle","*.n","nemerle"),
    Language("nesC ", "*.nc", "nesc"),
    Language("NestedText", "*.nt", "nestedtext, nt"),
    Language("NewLisp", "*.lsp, *.nl, *.kif", "newlisp"),
    Language("Newspeak", "*.ns2", "newspeak"),
    Language("Nginx configuration file", "nginx.conf", "nginx"),
    Language("Nimrod", "*.nim, *.nimrod", "nimrod, nim"),
    Language("Nit", "*.nit", "nit"),
    Language("Nix", "*.nix", "nixos, nix"),
    Language("Node.js REPL console session", None, "nodejsrepl"),
    Language("Notmuch", None, "notmuch"),
    Language("NSIS", "*.nsi, *.nsh", "nsis, nsi, nsh"),
    Language("Numba_IR ", "*.numbair", "numba_ir, numbair"),
    Language("NumPy", None, "numpy"),
    Language("NuSMV", "*.smv", "nusmv"),
    Language("objdump", "*.objdump", "objdump"),
    Language("objdump-nasm", "*.objdump-intel", "objdump-nasm"),
    # Language("Objective-C ","*.m, *.h","objective-c, objectivec, obj-c, objc"),
    # Language("Objective-C++","*.mm, *.hh","objective-c++, objectivec++, obj-c++, objc++"),
    # Language("Objective-J","*.j","objective-j, objectivej, obj-j, objj"),
    Language("OCaml", "*.ml, *.mli, *.mll, *.mly", "ocaml"),
    # Language("Octave","*.m","octave"),
    Language("ODIN", "*.odin", "odin"),
    Language("OMG Interface Definition Language", "*.idl, *.pidl", "omg-idl"),
    Language("Ooc", "*.ooc", "ooc"),
    Language("Opa", "*.opa", "opa"),
    Language("OpenEdge ABL ", "*.p, *.cls", "openedge, abl, progress"),
    Language("OpenSCAD", "*.scad", "openscad"),
    Language("Org Mode", "*.org", "org, orgmode, org-mode"),
    Language("PacmanConf", "pacman.conf", "pacmanconf"),
    Language("Pan", "*.pan", "pan"),
    Language("ParaSail", "*.psi, *.psl", "parasail"),
    # Language("Pawn","*.p, *.pwn, *.inc","pawn"),
    Language("PDDL", "*.pddl", "pddl"),
    Language("PEG", "*.peg", "peg"),
    Language("Perl", "*.pl, *.pm, *.t, *.perl", "perl, pl"),
    # Language("Perl6","*.pl, *.pm, *.nqp, *.p6, *.6pl, *.p6l, *.pl6, *.6pm, *.p6m, *.pm6, *.t, *.raku, *.rakumod, *.rakutest, *.rakudoc","perl6, pl6, raku"),
    Language("Phix", "*.exw", "phix"),
    Language("PHP", "*.php, *.php[345], *.inc", "php, php3, php4, php5"),
    Language("Pig", "*.pig", "pig"),
    Language("Pike", "*.pike, *.pmod", "pike"),
    Language("PkgConfig", "*.pc", "pkgconfig"),
    Language("PL/pgSQL", None, "plpgsql"),
    Language("Pointless", "*.ptls", "pointless"),
    Language("Pony", "*.pony", "pony"),
    Language("Portugol", "*.alg, *.portugol", "portugol"),
    Language(
        "PostgreSQL console (psql)", None, "psql, postgresql-console, postgres-console"
    ),
    Language("PostgreSQL EXPLAIN dialect ", "*.explain", "postgres-explain"),
    Language("PostgreSQL SQL dialect", None, "postgresql, postgres"),
    Language("PostScript", "*.ps, *.eps", "postscript, postscr"),
    # Language("POVRay","*.pov, *.inc","pov"),
    Language("PowerShell", "*.ps1, *.psm1", "powershell, pwsh, posh, ps1, psm1"),
    Language("PowerShell Session", None, "pwsh-session, ps1con"),
    Language("Praat", "*.praat, *.proc, *.psc", "praat"),
    Language("Procfile", "Procfile", "procfile"),
    # Language("Prolog","*.ecl, *.prolog, *.pro, *.pl","prolog"),
    # Language("Promela","*.pml, *.prom, *.prm, *.promela, *.pr, *.pm","promela"),
    Language("PromQL", "*.promql", "promql"),
    Language("Properties", "*.properties", "properties, jproperties"),
    Language("Protocol Buffer", "*.proto", "protobuf, proto"),
    Language("PRQL", "*.prql", "prql"),
    Language("PsySH console session for PHP", None, "psysh"),
    Language("PTX ", "*.ptx", "ptx"),
    Language("Pug", "*.pug, *.jade", "pug, jade"),
    Language("Puppet", "*.pp", "puppet"),
    Language("PyPy Log", "*.pypylog", "pypylog, pypy"),
    Language(
        "Python",
        "*.py, *.pyw, *.pyi, *.jy, *.sage, *.sc, SConstruct, SConscript, *.bzl, BUCK, BUILD, BUILD.bazel, WORKSPACE, *.tac",
        "python, py, sage, python3, py3, bazel, starlark, pyi",
    ),
    Language("Python 2.x", None, "python2, py2"),
    Language("Python 2.x Traceback", "*.py2tb", "py2tb"),
    Language("Python console session", None, "pycon, python-console"),
    Language("Python Traceback", "*.pytb, *.py3tb", "pytb, py3tb"),
    Language("Python+UL4", "*.pyul4", "py+ul4"),
    Language("Q", "*.q", "q"),
    # Language("QBasic","*.BAS, *.bas","qbasic, basic"),
    Language("Qlik", "*.qvs, *.qvw", "qlik, qlikview, qliksense, qlikscript"),
    Language("QML", "*.qml, *.qbs", "qml, qbs"),
    Language("QVTO", "*.qvto", "qvto, qvt"),
    Language("Racket", "*.rkt, *.rktd, *.rktl", "racket, rkt"),
    Language("Ragel", None, "ragel"),
    # Language("Ragel in C Host","*.rl","ragel-c"),
    # Language("Ragel in CPP Host","*.rl","ragel-cpp"),
    # Language("Ragel in D Host","*.rl","ragel-d"),
    # Language("Ragel in Java Host ","*.rl","ragel-java"),
    # Language("Ragel in Objective C Host","*.rl","ragel-objc"),
    # Language("Ragel in Ruby Host ","*.rl","ragel-ruby, ragel-rb"),
    Language("Raw token data", None, ""),
    Language("RConsole", "*.Rout", "rconsole, rout"),
    Language("Rd", "*.Rd", "rd"),
    Language("ReasonML", "*.re, *.rei", "reasonml, reason"),
    Language("REBOL", "*.r, *.r3, *.reb", "rebol"),
    Language("Red", "*.red, *.reds", "red, red/system"),
    Language("Redcode", "*.cw", "redcode"),
    Language("reg ", "*.reg", "registry"),
    Language("Rego", "*.rego", "rego"),
    Language("Relax-NG Compact", "*.rnc", "rng-compact, rnc"),
    Language("ResourceBundle", None, "resourcebundle, resource"),
    Language("reStructuredText ", "*.rst, *.rest", "restructuredtext, rst, rest"),
    Language("Rexx", "*.rexx, *.rex, *.rx, *.arexx", "rexx, arexx"),
    # Language("RHTML","*.rhtml, *.html, *.htm, *.xhtml","rhtml, html+erb, html+ruby"),
    Language("Ride", "*.ride", "ride"),
    Language("Rita", "*.rita", "rita"),
    Language("Roboconf Graph", "*.graph", "roboconf-graph"),
    Language("Roboconf Instances", "*.instances", "roboconf-instances"),
    Language("RobotFramework", "*.robot, *.resource", "robotframework"),
    Language("RPMSpec", "*.spec", "spec"),
    Language("RQL", "*.rql", "rql"),
    Language("RSL", "*.rsl", "rsl"),
    Language(
        "Ruby",
        "*.rb, *.rbw, Rakefile, *.rake, *.gemspec, *.rbx, *.duby, Gemfile, Vagrantfile",
        "ruby, rb, duby",
    ),
    Language("Ruby irb session", None, "rbcon, irb"),
    Language("Rust", "*.rs, *.rs.in", "rust, rs"),
    # Language("S","*.S, *.R, .Rhistory, .Rprofile, .Renviron","splus, s, r"),
    Language("SARL", "*.sarl", "sarl"),
    Language("SAS", "*.SAS, *.sas", "sas"),
    Language("Sass", "*.sass", "sass"),
    Language("Savi", "*.savi", "savi"),
    Language("Scala", "*.scala", "scala"),
    Language("Scalate Server Page", "*.ssp", "ssp"),
    Language("Scaml", "*.scaml", "scaml"),
    Language("scdoc", "*.scd, *.scdoc", "scdoc, scd"),
    Language("Scheme", "*.scm, *.ss", "scheme, scm"),
    # Language("Scilab","*.sci, *.sce, *.tst","scilab"),
    Language("SCSS", "*.scss", "scss"),
    Language("Sed", "*.sed, *.[gs]sed", "sed, gsed, ssed"),
    Language("Shen", "*.shen", "shen"),
    Language("ShExC", "*.shex", "shexc, shex"),
    Language("Sieve", "*.siv, *.sieve", "sieve"),
    Language("Silver", "*.sil, *.vpr", "silver"),
    # Language("Singularity","*.def, Singularity","singularity"),
    Language("Slash", "*.sla", "slash"),
    Language("Slim", "*.slim", "slim"),
    Language("Slurm", "*.sl", "slurm, sbatch"),
    Language("Smali", "*.smali", "smali"),
    Language("Smalltalk", "*.st", "smalltalk, squeak, st"),
    Language("SmartGameFormat", "*.sgf", "sgf"),
    Language("Smarty", "*.tpl", "smarty"),
    Language("Smithy", "*.smithy", "smithy"),
    Language("SNBT", "*.snbt", "snbt"),
    Language("Snobol", "*.snobol", "snobol"),
    Language("Snowball", "*.sbl", "snowball"),
    Language("Solidity", "*.sol", "solidity"),
    Language("Soong", "Android.bp", "androidbp, bp, soong"),
    Language("Sophia", "*.aes", "sophia"),
    Language("SourcePawn", "*.sp", "sp"),
    Language("SPARQL", "*.rq, *.sparql", "sparql"),
    Language("Spice", "*.spice", "spice, spicelang"),
    Language("SQL", "*.sql", "sql"),
    # Language("SQL+Jinja","*.sql, *.sql.j2, *.sql.jinja2","sql+jinja"),
    Language("sqlite3con", "*.sqlite3-console", "sqlite3"),
    Language("SquidConf", "squid.conf", "squidconf, squid.conf, squid"),
    Language("Srcinfo", ".SRCINFO", "srcinfo"),
    Language("Stan", "*.stan", "stan"),
    Language("Standard ML", "*.sml, *.sig, *.fun", "sml"),
    Language("Stata", "*.do, *.ado", "stata, do"),
    # Language("SuperCollider","*.sc, *.scd","supercollider, sc"),
    Language("Swift", "*.swift", "swift"),
    Language("SWIG", "*.swg, *.i", "swig"),
    Language(
        "Systemd",
        "*.service, *.socket, *.device, *.mount, *.automount, *.swap, *.target, *.path, *.timer, *.slice, *.scope",
        "systemd",
    ),
    Language("systemverilog ", "*.sv, *.svh", "systemverilog, sv"),
    Language("TableGen", "*.td", "tablegen, td"),
    Language("Tact", "*.tact", "tact"),
    # Language("TADS 3","*.t","tads3"),
    Language("Tal", "*.tal", "tal, uxntal"),
    Language("TAP", "*.tap", "tap"),
    # Language("TASM","*.asm, *.ASM, *.tasm","tasm"),
    Language("Tcl", "*.tcl, *.rvt", "tcl"),
    Language("Tcsh", "*.tcsh, *.csh", "tcsh, csh"),
    Language("Tcsh Session", None, "tcshcon"),
    Language("Tea", "*.tea", "tea"),
    Language("teal", "*.teal", "teal"),
    Language("Tera Term macro", "*.ttl", "teratermmacro, teraterm, ttl"),
    Language("Termcap", "termcap, termcap.src", "termcap"),
    Language("Terminfo", "terminfo, terminfo.src", "terminfo"),
    Language("Terraform", "*.tf, *.hcl", "terraform, tf, hcl"),
    Language("TeX", "*.tex, *.aux, *.toc", "tex, latex"),
    Language("Text only", "*.txt", "text"),
    Language("Text output", None, "output"),
    Language("ThingsDB", "*.ti", "ti, thingsdb"),
    Language("Thrift", "*.thrift", "thrift"),
    Language("tiddler", "*.tid", "tid"),
    Language("Tl-b", "*.tlb", "tlb"),
    Language("TLS Presentation Language", None, "tls"),
    Language("Todotxt", "todo.txt, *.todotxt", "todotxt"),
    Language("TOML", "*.toml, Pipfile, poetry.lock", "toml"),
    Language("TrafficScript", "*.rts", "trafficscript, rts"),
    # Language("Transact-SQL","*.sql","tsql, t-sql"),
    Language("Treetop", "*.treetop, *.tt", "treetop"),
    Language("TSX ", "*.tsx", "tsx"),
    # Language("Turtle","*.ttl","turtle"),
    Language("Twig", None, "twig"),
    Language("TypeScript", "*.ts", "typescript, ts"),
    Language("Typographic Number Theory", "*.tnt", "tnt"),
    Language("TypoScript", "*.typoscript", "typoscript"),
    Language("TypoScriptCssData", None, "typoscriptcssdata"),
    Language("TypoScriptHtmlData", None, "typoscripthtmldata"),
    Language("Typst", "*.typ", "typst"),
    Language("ucode", "*.u, *.u1, *.u2", "ucode"),
    Language("UL4", "*.ul4", "ul4"),
    Language("Unicon", "*.icn", "unicon"),
    Language("Unix/Linux config files", None, "unixconfig, linuxconfig"),
    # Language("UrbiScript","*.u","urbiscript"),
    Language("urlencoded ", None, "urlencoded"),
    Language("USD", "*.usd, *.usda", "usd, usda"),
    Language("Vala", "*.vala, *.vapi", "vala, vapi"),
    Language(
        "VB.net ",
        "*.vb, *.bas",
        "vb.net, vbnet, lobas, oobas, sobas, visual-basic, visualbasic",
    ),
    Language("VBScript", "*.vbs, *.VBS", "vbscript"),
    Language("VCL ", "*.vcl", "vcl"),
    Language("VCLSnippets ", None, "vclsnippets, vclsnippet"),
    Language("VCTreeStatus", None, "vctreestatus"),
    Language("Velocity", "*.vm, *.fhtml", "velocity"),
    Language("Verifpal", "*.vp", "verifpal"),
    Language("verilog", "*.v", "verilog, v"),
    Language("VGL", "*.rpf", "vgl"),
    Language("vhdl", "*.vhdl, *.vhd", "vhdl"),
    Language(
        "VimL",
        "*.vim, .vimrc, .exrc, .gvimrc, vimrc, exrc, gvimrc, vimrc, gvimrc",
        "vim",
    ),
    # Language("Visual Prolog","*.pro, *.cl, *.i, *.pack, *.ph","visualprolog"),
    Language("Visual Prolog Grammar", "*.vipgrm", "visualprologgrammar"),
    Language("Vue", "*.vue", "vue"),
    Language("Vyper", "*.vy", "vyper"),
    Language("WDiff", "*.wdiff", "wdiff"),
    Language("Web IDL", "*.webidl", "webidl"),
    Language("WebAssembly", "*.wat, *.wast", "wast, wat"),
    Language("WebGPU Shading Language", "*.wgsl", "wgsl"),
    Language("Whiley", "*.whiley", "whiley"),
    Language("Wikitext", None, "wikitext, mediawiki"),
    # Language("World of Warcraft TOC","*.toc","wowtoc"),
    Language("Wren", "*.wren", "wren"),
    Language("X++ ", "*.xpp", "xpp, x++"),
    Language("X10", "*.x10", "x10, xten"),
    Language("XML", "*.xml, *.xsl, *.rss, *.xsd, *.wsdl, *.wsf", "xml"),
    # Language("XML","*.xml, *.xsl, *.rss, *.xslt, *.xsd, *.wsdl, *.wsf","xml"),
    # Language("XML+Cheetah",None,"xml+cheetah, xml+spitfire"),
    # Language("XML+Django/Jinja","*.xml.j2, *.xml.jinja2, *.xml","xml+django, xml+jinja"),
    # Language("XML+Evoque","*.xml","xml+evoque"),
    # Language("XML+Lasso","*.xml, *.lasso, *.lasso[89], *.incl, *.inc, *.las","xml+lasso"),
    # Language("XML+Mako",None,"xml+mako"),
    # Language("XML+Myghty",None,"xml+myghty"),
    # Language("XML+PHP","*.xml, *.php, *.php[345]","xml+php"),
    # Language("XML+Ruby","*.xml","xml+ruby, xml+erb"),
    # Language("XML+Smarty","*.xml, *.tpl","xml+smarty"),
    # Language("XML+UL4","*.xmlul4","xml+ul4"),
    # Language("XML+Velocity","*.xml, *.vm","xml+velocity"),
    Language("Xorg", "xorg.conf", "xorg.conf"),
    Language(
        "XQuery", "*.xqy, *.xquery, *.xq, *.xql, *.xqm", "xquery, xqy, xq, xql, xqm"
    ),
    # Language("XSLT","*.xsl, *.xslt, *.xpl","xslt"),
    Language("Xtend", "*.xtend", "xtend"),
    Language("xtlang", "*.xtm", "extempore"),
    Language("XUL+mozpreproc ", "*.xul.in", "xul+mozpreproc"),
    Language("YAML", "*.yaml, *.yml", "yaml"),
    Language(
        "YAML+Jinja",
        "*.sls, *.yaml.j2, *.yml.j2, *.yaml.jinja2, *.yml.jinja2",
        "yaml+jinja, salt, sls",
    ),
    Language("YANG", "*.yang", "yang"),
    Language("YARA", "*.yar", "yara, yar"),
    Language("Zeek", "*.zeek, *.bro", "zeek, bro"),
    Language("Zephir", "*.zep", "zephir"),
    Language("Zig", "*.zig", "zig"),
    Language("Zone", "*.zone", "zone"),
]
# cSpell:enable
