/*******************************************************************************
 * Copyright (c) 2010, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package com.ibm.ws.monitor.internal;

import java.lang.annotation.Annotation;
import java.lang.instrument.ClassFileTransformer;
import java.lang.instrument.Instrumentation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.objectweb.asm.Type;
import org.osgi.service.component.ComponentContext;

import com.ibm.websphere.monitor.MonitorManager;
import com.ibm.websphere.monitor.Probe;
import com.ibm.websphere.monitor.annotation.Monitor;
import com.ibm.websphere.monitor.annotation.ProbeSite;
import com.ibm.websphere.pmi.server.PmiAbstractModule;
import com.ibm.websphere.ras.Tr;
import com.ibm.websphere.ras.TraceComponent;
import com.ibm.websphere.ras.annotation.Sensitive;
import com.ibm.ws.monitor.internal.bci.ClassAvailableTransformer;
import com.ibm.ws.monitor.internal.bci.ProbeClassFileTransformer;

/**
 * Declarative services component that is responsible for tracking
 * monitoring components and activating probes of interest.
 */

public class ProbeManagerImpl implements MonitorManager {

    private static final TraceComponent tc = Tr.register(ProbeManagerImpl.class);
    /**
     * Probe identifier generation. The probe identifiers are only guaranteed
     * to be unique for the life of this probe manager instance.
     *
     * TODO: Need to save high water mark for the ID in work area to handle the
     * case where the probe manager is restarted.
     */
    private final AtomicLong probeIdCount = new AtomicLong();

    // TODO: Proper configuration
    /**
     * Flag that indicates whether classes loaded by the bootstrap class
     * loader should be eligible for probe injection.
     */
    boolean includeBootstrap = Boolean.getBoolean("probes.include.bootstrap");

    /**
     * The declarative services {@link ComponentContext} for this component.
     */
    ComponentContext componentContext = null;

    /**
     * The {@link Instrumentation} reference for this JVM.
     */
    Instrumentation instrumentation = null;

    /**
     * {@link ClassFileTransformer} implementation that adds a proxied {@link #classAvailable} callback.
     */
    ClassAvailableTransformer classAvailableTransformer = null;

    public static Map<Object, String> moduleInstanceToBundleMap = new HashMap<Object, String>();
    /**
     * {@link ClassFileTransformer} implementation for monitoring.
     */
    ProbeClassFileTransformer transformer = null;

    /**
     * Reference to the component that manages the runtime availability
     * and visibility of the bootstrap proxy implementations.
     */
    MonitoringProxyActivator proxyActivator = null;

    /**
     * Set of classes that have been processed that are not monitorable.
     */
    final Set<Class<?>> notMonitorable = Collections.synchronizedSet(Collections.newSetFromMap(new WeakHashMap<Class<?>, Boolean>()));

    /**
     * Set of classes that have been processed that are monitorable.
     */
    final Set<Class<?>> monitorable = Collections.synchronizedSet(Collections.newSetFromMap(new WeakHashMap<Class<?>, Boolean>()));

    /**
     * Read / write lock used to synchronize listener collections in a way
     * that allows for concurrent read and iteration.
     */
    final ReadWriteLock listenersLock = new ReentrantReadWriteLock();

    /**
     * Collection of listeners that were generated by the registration of a
     * monitor.
     */
    Map<Object, Set<ProbeListener>> listenersForMonitor = new HashMap<Object, Set<ProbeListener>>();

    /**
     * The set all listeners that have been created through monitor registration.
     */
    Set<ProbeListener> allRegisteredListeners = new HashSet<ProbeListener>();

    /**
     * The set of registered listeners with a probe site filter that matches the {@link Class} instance used as a key. This is used during transformation
     * to find the subset of listeners that need to be processed.
     */
    Map<Class<?>, Set<ProbeListener>> listenersByClass = new WeakHashMap<Class<?>, Set<ProbeListener>>();

    /**
     * Maps of probe names to probes associated with the class literal the are fired from. The
     * class is used to scope the probe the correct class loader scope.
     */
    Map<Class<?>, Map<String, ProbeImpl>> probesByKey = new WeakHashMap<Class<?>, Map<String, ProbeImpl>>();

    /**
     * The set of all active probes that will be delivered to the listener. This is
     * used to determine which probes need to updated where a listener is removed.
     */
    Map<ProbeListener, Set<ProbeImpl>> probesByListener = new HashMap<ProbeListener, Set<ProbeImpl>>();

    /**
     * Map of probe identifier to probe instance. This is used where firing probes
     * to listeners and is on the performance critical path.
     */
    Map<Long, ProbeImpl> activeProbesById = new ConcurrentHashMap<Long, ProbeImpl>();

    /**
     * Map of probe implementations to associated listeners. This is used where firing
     * probes to listeners and is on the performance critical path.
     */
    ConcurrentMap<ProbeImpl, Set<ProbeListener>> listenersByProbe = new ConcurrentHashMap<ProbeImpl, Set<ProbeListener>>();

    /**
     * Flag that's set as the component is being deactivated.
     */
    boolean shuttingDown = false;
    long prevTime;

    /* Set Which maintains the name's of the classes which needs to be instrumented. */
    private final Set<String> probeMonitorSet = Collections.newSetFromMap(new ConcurrentHashMap<String, Boolean>());

    /**
     * Activation callback from the Declarative Services runtime where the
     * component is ready for activation.
     *
     * @param bundleContext the bundleContext
     */
    synchronized void activate(ComponentContext componentContext) throws Exception {
        this.componentContext = componentContext;
        this.classAvailableTransformer = new ClassAvailableTransformer(this, instrumentation, includeBootstrap);
        this.transformer = new ProbeClassFileTransformer(this, instrumentation, includeBootstrap);
        this.proxyActivator = new MonitoringProxyActivator(componentContext.getBundleContext(), this, this.instrumentation);
        this.proxyActivator.activate();
        // The class available transformer is registered as retransform incapable
        // to avoid having to keep track of which classes have been updated with
        // static initializers and serialVersionUID fields.  The probe transformer
        // however, must run through the listener config every time a retransform
        // occurs to allow it to discover new probes and replace currently active
        // probes

        //RTCD 89497-Update the set with the classes
        for (Class<?> clazz : MonitoringUtility.loadMonitoringClasses(componentContext.getBundleContext().getBundle())) {
            for (int i = 0; i < clazz.getMethods().length; i++) {
                Annotation anno = ((clazz.getMethods()[i]).getAnnotation(ProbeSite.class));
                if (anno != null) {
                    String temp = ((ProbeSite) anno).clazz();
                    probeMonitorSet.add(temp);
                }
            }
        }
        //Update to the set ended.

        //RTC D89497 adding instrumentors moved below as the MonitoringUtility.loadMonitoringClasses
        //was resulting in loading classes and in turn getting called during with transform process which
        //might cause hang in some situations
        this.instrumentation.addTransformer(this.transformer, true);
        this.instrumentation.addTransformer(this.classAvailableTransformer);
        // We're active so if we have any listeners, we can run down the loaded
        // classes.
        for (Class<?> clazz : this.instrumentation.getAllLoadedClasses()) {
            classAvailable(clazz);
        }
    }

    /**
     * Deactivation callback from the Declarative Services runtime where the
     * component is deactivated.
     *
     * @param bundleContext the bundleContext
     */
    synchronized void deactivate() throws Exception {
        this.proxyActivator.deactivate();
        this.instrumentation.removeTransformer(this.classAvailableTransformer);
        this.instrumentation.removeTransformer(this.transformer);
        this.shuttingDown = true;

        // Save the classes that have listeners so we can retransform
        Collection<Class<?>> probedClasses = processRemovedListeners(allRegisteredListeners);

        // Clear all data structures
        listenersLock.writeLock().lock();
        try {
            listenersForMonitor.clear();
            allRegisteredListeners.clear();
        } finally {
            listenersLock.writeLock().unlock();
        }
        activeProbesById.clear();
        listenersByProbe.clear();
        listenersByClass.clear();
        probesByKey.clear();
        probesByListener.clear();

        this.proxyActivator = null;
        this.transformer = null;
        this.componentContext = null;

        // Retransform the probed classes without our transformers in play
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "deactivate: probedClasses.size() = " + probedClasses.size());
        }
        for (Class<?> clazz : probedClasses) {
            instrumentation.retransformClasses(clazz);
        }
        this.instrumentation = null;
    }

    /**
     * Inject reference to the {@link java.lang.instrument.Instrumentation} implementation.
     *
     * @param instrumentationAgent the JVM's {@code Instrumentation) reference
     */
    protected void setInstrumentation(Instrumentation instrumentation) {
        this.instrumentation = instrumentation;
    }

    /**
     * Clear the reference to the {@link java.lang.instrument.Instrumentation} implementation.
     *
     * @param instrumentationAgent the reference to invalidate
     */
    protected void unsetInstrumentation(Instrumentation instrumentation) {
        if (instrumentation == this.instrumentation) {
            this.instrumentation = null;
        }
    }

    /**
     * Register an annotated object as a monitor. The annotations will be used
     * to generate the appropriate configuration.
     *
     * @param monitor an annotated pojo that acts as a probe listener
     */
    @Override
    public boolean registerMonitor(Object monitor) {
        // Process the annotated object, build a configuration, and create probe
        // listeners that represent each call site of the monitor.
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "monitor = " + monitor);
        }
        long startTime = System.nanoTime();
        Set<ProbeListener> listeners = buildListenersFromAnnotated(monitor);
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "buildListenersFromAnnotated time = " + TimeUnit.NANOSECONDS.toMicros(System.nanoTime() - startTime) + "usec");
        }

        // Update collections of listeners
        startTime = System.nanoTime();
        if (!addListenersForMonitor(monitor, listeners)) {
            return false;
        }
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "addListenersForMonitorTime = " + TimeUnit.NANOSECONDS.toMicros(System.nanoTime() - startTime) + "usec");
        }

        // Process all of the listeners and determine which of the available
        // classes needs to be transformed
        startTime = System.nanoTime();
        Collection<Class<?>> effectedClasses = processNewListeners(listeners);
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "processNewListeners time = " + TimeUnit.NANOSECONDS.toMicros(System.nanoTime() - startTime) + "usec");
        }

        // Update the classes with the configured probes
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "effectedClasses.size() = " + effectedClasses.size());
        }
        startTime = System.nanoTime();
        this.transformer.instrumentWithProbes(effectedClasses);
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "instrumentWithProbes time = " + TimeUnit.NANOSECONDS.toMicros(System.nanoTime() - startTime) + "usec");
        }

        return true;
    }

    @Override
    public boolean registerMonitor(Object monitor, Map<String, Object> config) {
        // TODO: Configuration map support
        return false;
    }

    /**
     * Unregister the specified monitor. Unregistering a monitor may cause one
     * or more probes to be deactivated and the associated classes retransformed.
     *
     * @param the monitor to unregister
     *
     * @return true iff the monitor was successfully unregistered
     */
    @Override
    public boolean unregisterMonitor(Object annotatedObject) {
        // Clear the listener data
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "unregisteringMonitor =  " + annotatedObject);
        }
        Set<ProbeListener> listeners = removeListenersForMonitor(annotatedObject);
        if (listeners == null) {
            return false;
        }

        // Remove the listeners associated with the inactive monitor and
        // find out which classes need to be transformed to remove stale probes
        Collection<Class<?>> effectedClasses = processRemovedListeners(listeners);

        // Update the classes to remove dead probes (optional)
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
            Tr.debug(this, tc, "effectedClasses.size() =  " + effectedClasses.size());
        }
        this.transformer.instrumentWithProbes(effectedClasses);
        Set set = moduleInstanceToBundleMap.entrySet();
        Iterator i = set.iterator();
        while (i.hasNext()) {
            Map.Entry mapE = (Map.Entry) i.next();
            if (mapE != null && mapE.getValue().equals(annotatedObject.getClass().getName())) {
                PmiAbstractModule s = (PmiAbstractModule) mapE.getKey();
                s.unregister();
            }
        }
        return true;
    }

    /**
     * Get the probe associated with the specified id.
     * TODO: Restrict visible probe identifiers to the bundle.
     */
    public Probe getProbe(long probeId) {
        return activeProbesById.get(probeId);
    }

    /**
     * Create a set of {@link ProbeListener}s that delegate annotated
     * methods on the specified monitor.
     *
     * @return the set of listeners to activate
     */
    Set<ProbeListener> buildListenersFromAnnotated(Object monitor) {
        Set<ProbeListener> listeners = new HashSet<ProbeListener>();

        Class<?> clazz = monitor.getClass();
        for (Method method : ReflectionHelper.getMethods(clazz)) {
            ProbeSite probeSite = method.getAnnotation(ProbeSite.class);
            if (probeSite == null) {
                continue;
            }
            synchronized (notMonitorable) {
                Class<?> rClass = null;
                for (Class<?> tClass : notMonitorable) {
                    if (tClass != null && ((tClass.getName()).equals(probeSite.clazz()))) {
                        rClass = tClass;
                        monitorable.add(tClass);
                        break;
                    }
                }
                if (rClass != null) {
                    notMonitorable.remove(rClass);
                }
            }
            // TODO: Handle monitoring groups
            Monitor monitorData = clazz.getAnnotation(Monitor.class);
            ListenerConfiguration config = new ListenerConfiguration(monitorData, probeSite, method);
            //            ProbeListener listener = new ProbeListener(this, config, monitor, method);
            ProbeListener listener = new ProbeListener(null, config, monitor, method);
            listeners.add(listener);
        }

        return listeners;
    }

    /**
     * Register a {@link Set} of listeners with its associated monitor. The
     * configuration of each of the registered listeners is evaluated each time
     * a class is initialized in the JVM.
     *
     * @param monitor the monitor instance that owns the listeners
     * @param listeners the listeners associated with the monitor
     *
     * @return true iff the listener set was successfully associated
     */
    boolean addListenersForMonitor(Object monitor, Set<ProbeListener> listeners) {
        listenersLock.writeLock().lock();
        try {
            if (listenersForMonitor.containsKey(monitor)) {
                return false;
            }
            listenersForMonitor.put(monitor, listeners);
            allRegisteredListeners.addAll(listeners);
        } finally {
            listenersLock.writeLock().unlock();
        }

        return true;
    }

    /**
     * Remove the listeners associated with the specified monitor from the set
     * of registered listeners.
     *
     * @param monitor the monitor that owns the listeners
     *
     * @return the set of listeners that were associated with the monitor
     */
    Set<ProbeListener> removeListenersForMonitor(Object monitor) {
        Set<ProbeListener> listeners = null;

        listenersLock.writeLock().lock();
        try {
            listeners = listenersForMonitor.remove(monitor);
            if (listeners == null) {
                listeners = Collections.emptySet();
            } else {
                allRegisteredListeners.removeAll(listeners);
            }
        } finally {
            listenersLock.writeLock().unlock();
        }

        return listeners;
    }

    /**
     * Process recently registered listeners and calculate which classes need
     * to be transformed with probes that drive the listeners.
     *
     * @param listeners the collection of recently registered listeners
     *
     * @return the classes that must be retransformed with probes
     */
    Collection<Class<?>> processNewListeners(Collection<ProbeListener> listeners) {
        Set<Class<?>> classesToTransform = new HashSet<Class<?>>();

        Set<ProbeListener> matchingListeners = new HashSet<ProbeListener>();
        Set<Class<?>> monitorableTemp;
        synchronized (this.monitorable) {
            monitorableTemp = new HashSet<Class<?>>(this.monitorable);
        }
        for (Class<?> clazz : monitorableTemp /* instrumentation.getAllLoadedClasses() */) {
            if (!isMonitorable(clazz)) {
                continue;
            }
            for (ProbeListener listener : listeners) {
                ProbeFilter filter = listener.getProbeFilter();
                if (filter.matches(clazz)) {
                    matchingListeners.add(listener);
                }
            }
            if (!matchingListeners.isEmpty()) {
                classesToTransform.add(clazz);
                addInterestedByClass(clazz, matchingListeners);
            }
            matchingListeners.clear();
        }

        return classesToTransform;
    }

    /**
     * Process the recently unregistered listeners and calculate which classes
     * need to be transformed to remove deactivated probes.
     *
     * @param listeners the collection of recently removed listeners
     *
     * @return the classes that must be retransformed to remove probes
     */
    synchronized Collection<Class<?>> processRemovedListeners(Collection<ProbeListener> listeners) {
        Set<Class<?>> classesToTransform = new HashSet<Class<?>>();

        for (ProbeListener listener : listeners) {
            Collection<ProbeImpl> listenerProbes = removeProbesByListener(listener);
            for (ProbeImpl probe : listenerProbes) {
                Class<?> clazz = probe.getSourceClass();
                removeInterestedByClass(clazz, listener);
                if (removeListenerByProbe(probe, listener)) {
                    classesToTransform.add(clazz);
                }
            }
        }

        return classesToTransform;
    }

    /**
     * Process a recently defined class to determine if it needs to be
     * transformed with probes.
     *
     * @param clazz a recently defined class
     */
    public void classAvailable(Class<?> clazz) {
        if (!isMonitorable(clazz)) {
            return;
        }
        listenersLock.readLock().lock();
        try {
            for (ProbeListener listener : allRegisteredListeners) {
                ProbeFilter filter = listener.getProbeFilter();
                if (filter.matches(clazz)) {
                    addInterestedByClass(clazz, Collections.singleton(listener));
                }
            }
        } finally {
            listenersLock.readLock().unlock();
        }

        // Update the new class with the configured probes
        if (!getInterestedByClass(clazz).isEmpty()) {
            this.transformer.instrumentWithProbes(Collections.<Class<?>> singleton(clazz));
        }
    }

    /**
     * Associate the collection of listeners with {@link ProbeFilter}s that
     * match the specified class with the specified class.
     *
     * @param clazz the probe source candidate
     * @param newListeners listeners with filters that match {@code clazz}
     */
    synchronized void addInterestedByClass(Class<?> clazz, Collection<ProbeListener> newListeners) {
        Set<ProbeListener> currentListeners = listenersByClass.get(clazz);
        if (currentListeners == null) {
            currentListeners = new HashSet<ProbeListener>();
            listenersByClass.put(clazz, currentListeners);
        }
        currentListeners.addAll(newListeners);
    }

    /**
     * Remove the specified listener from the collection of listeners with {@link ProbeFilter}s that match the specified class.
     *
     * @param clazz the candidate probe source
     * @param listener the listener with a filter that matched {@code clazz}
     */
    synchronized void removeInterestedByClass(Class<?> clazz, ProbeListener listener) {
        Set<ProbeListener> listeners = listenersByClass.get(clazz);
        if (listeners != null) {
            listeners.remove(listener);
            if (listeners.isEmpty()) {
                listenersByClass.remove(clazz);
            }
        }
    }

    /**
     * Get the set of listeners with a {@link ProbeFilter} that matches
     * the specified class. The class transformer will call this during
     * class transformation to get the subset of listeners that need to be
     * processed.
     *
     * @param clazz the candidate probe source
     *
     * @return the subset of listeners with filters that match {@code clazz}
     */
    public synchronized Set<ProbeListener> getInterestedByClass(Class<?> clazz) {
        Set<ProbeListener> listeners = listenersByClass.get(clazz);
        if (listeners == null) {
            listeners = Collections.emptySet();
        }
        // Make copy to allow caller to iterate over contents
        return new HashSet<ProbeListener>(listeners);
    }

    /**
     * Get an existing instance of a probe with the specified source class and
     * key.
     *
     * @param probedClass the probe source class
     * @param key the probe key
     *
     * @return an existing probe or null
     */
    public synchronized ProbeImpl getProbe(Class<?> probedClass, String key) {
        Map<String, ProbeImpl> classProbes = probesByKey.get(probedClass);
        if (classProbes != null) {
            return classProbes.get(key);
        }
        return null;
    }

    /**
     * Create a new {@link ProbeImpl} with the specified information.
     *
     * @param probedClass the probe source
     * @param key the unique name of the probe within the source class
     * @param ctor the probed constructor or null
     * @param method the probed method or null
     * @param probeKind the kind of probe
     *
     * @return the new probe implementation
     */
    public synchronized ProbeImpl createProbe(Class<?> probedClass, String key, Constructor<?> ctor, Method method) {
        ProbeImpl probeImpl = getProbe(probedClass, key);
        if (probeImpl == null) {
            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
                Tr.debug(this, tc, "createProbe: " + key);
            }
            probeImpl = new ProbeImpl(this, probedClass, key, ctor, method);
            activeProbesById.put(Long.valueOf(probeImpl.getIdentifier()), probeImpl);
            Map<String, ProbeImpl> classProbes = probesByKey.get(probedClass);
            if (classProbes == null) {
                classProbes = new HashMap<String, ProbeImpl>();
                probesByKey.put(probedClass, classProbes);
            }
            classProbes.put(key, probeImpl);
        }
        return probeImpl;
    }

    //    public synchronized ProbeImpl createProbe(Class<?> probedClass, String key, Constructor<?> ctor, Method method, Where where, When when, Member member) {
    //    }

    /**
     * Generate a new probe identifier.
     *
     * @return a new probe identifier
     */
    long generateProbeId() {
        return probeIdCount.incrementAndGet();
    }

    /**
     * Update the appropriate collections to reflect recently activated probes
     * for the specified listener.
     *
     * @param listener the listener with recently activated probes
     * @param probes the collection of probes that were activated
     */
    public void addActiveProbesforListener(ProbeListener listener, Collection<ProbeImpl> probes) {
        // Add the probes for the specified listener.  This is purely additive.
        // Since a listener's configuration can't be updated after registration,
        // we're adding probes for recently initialized / modified classes.
        addProbesByListener(listener, probes);

        for (ProbeImpl probe : probes) {
            addListenerByProbe(probe, listener);
        }
    }

    /**
     * Add the specified listener to the collection of listeners associated
     * with the specified probe.
     *
     * @param probe the probe that fires for the listener
     * @param listener the listener that's driven by the probe
     */
    void addListenerByProbe(ProbeImpl probe, ProbeListener listener) {
        Set<ProbeListener> listeners = listenersByProbe.get(probe);
        if (listeners == null) {
            listeners = new CopyOnWriteArraySet<ProbeListener>();
            listenersByProbe.putIfAbsent(probe, listeners);
            listeners = listenersByProbe.get(probe);
        }
        listeners.add(listener);
    }

    /**
     * Remove the specified listener from the collection of listeners associated
     * with the specified probe.
     *
     * @param probe the probe that fired for the listener
     * @param listener the listener that was driven by the probe
     *
     * @return true if removing the listener caused the probe to become disabled
     */
    boolean removeListenerByProbe(ProbeImpl probe, ProbeListener listener) {
        boolean deactivatedProbe = false;

        Set<ProbeListener> listeners = listenersByProbe.get(probe);
        if (listeners != null) {
            listeners.remove(listener);
            if (listeners.isEmpty()) {
                deactivateProbe(probe);
                deactivatedProbe = true;
            }
        }

        return deactivatedProbe;
    }

    /**
     * Deactivate the specified probe and remove all data associated with the
     * probe.
     *
     * @param probe the probe to deactivate
     */
    synchronized void deactivateProbe(ProbeImpl probe) {
        listenersByProbe.remove(probe);
        activeProbesById.remove(probe.getIdentifier());

        Class<?> clazz = probe.getSourceClass();
        Map<String, ProbeImpl> classProbesByKey = probesByKey.get(clazz);
        classProbesByKey.remove(probe.getName());
        if (classProbesByKey.isEmpty()) {
            probesByKey.remove(clazz);
        }
    }

    /**
     * Associate the specified collection of probes with the specified listener.
     *
     * @param listener the listener that probes are fired to
     * @param probes the probes fired to the listener
     */
    synchronized void addProbesByListener(ProbeListener listener, Collection<ProbeImpl> probes) {
        Set<ProbeImpl> listenerProbes = probesByListener.get(listener);
        if (listenerProbes == null) {
            listenerProbes = new HashSet<ProbeImpl>();
            probesByListener.put(listener, listenerProbes);
        }
        listenerProbes.addAll(probes);
    }

    /**
     * Remove all probes that were fired at the specified listener.
     *
     * @param listener the listener that probes were fired to
     *
     * @return the probes that were assocaited with the listener
     */
    synchronized Collection<ProbeImpl> removeProbesByListener(ProbeListener listener) {
        Set<ProbeImpl> listenerProbes = probesByListener.remove(listener);
        if (listenerProbes == null) {
            listenerProbes = Collections.emptySet();
        }
        return listenerProbes;
    }

    /**
     * Determine if the named class is one that should be excluded from
     * monitoring via probe injection. The patterns below generally
     * include classes required to implement JVM function on which the
     * monitoring code depends.
     *
     * @param className the internal name of the class
     */
    public boolean isExcludedClass(String className) {
        // We rely heavily on reflection to deliver probes
        if (className.startsWith("java/lang/reflect")) {
            return true;
        }
        // Miscellaneous sun.misc classes
        if (className.startsWith("sun/misc")) {
            return true;
        }
        // Sun VM generated accessors wreak havoc
        if (className.startsWith("sun/reflect")) {
            return true;
        }
        // IBM J9 VM internals
        if (className.startsWith("com/ibm/oti/")) {
            return true;
        }
        // Don't allow hooking into the monitoring code
        if (className.startsWith("com/ibm/ws/monitor/internal")) {
            return true;
        }
        if (className.startsWith("com/ibm/websphere/monitor")) {
            return true;
        }
        if (className.startsWith("com/ibm/ws/boot/delegated/monitoring")) {
            return true;
        }
        if ((className.startsWith("com/ibm/ws/pmi")) || (className.startsWith("com/ibm/websphere/pmi")) || (className.startsWith("com/ibm/wsspi/pmi"))) {
            return true;
        }
        //D89497-Excluding those classes which are not part of the Set.
        if ((!(probeMonitorSet.contains((className.replace("/", ".")))))) {
            return true;
        }
        return false;
    }

    /**
     * Determine of the specified class can be monitored via the probes
     * infrastructure.
     *
     * @param clazz the monitoring candidate
     *
     * @return false if the class can't be monitored
     */
    public boolean isMonitorable(Class<?> clazz) {
        // Faster(?) path
        if (notMonitorable.contains(clazz)) {
            return false;
        } else if (monitorable.contains(clazz)) {
            return true;
        }

        boolean isMonitorable = true;
        if (!instrumentation.isModifiableClass(clazz)) {
            isMonitorable = false;
        } else if (clazz.isInterface()) {
            isMonitorable = false;
        } else if (clazz.isArray()) {
            isMonitorable = false;
        } else if (Proxy.isProxyClass(clazz)) {
            isMonitorable = false;
        } else if (clazz.isPrimitive()) {
            isMonitorable = false;
        } else if (isExcludedClass(Type.getInternalName(clazz))) {
            isMonitorable = false;
        } else if (!includeBootstrap && clazz.getClassLoader() == null) {
            isMonitorable = false;
        }

        // Update collections
        if (!isMonitorable) {
            synchronized (notMonitorable) {
                notMonitorable.add(clazz);
            }
        } else {
            monitorable.add(clazz);
        }
        return isMonitorable;
    }

    /**
     * Determine if a class that's being defined is one that can be
     * instrumented with probes.
     *
     * @param classInternalName the JVM internal class name
     *
     * @return true if the class can be instrumented with probes
     */
    public boolean isProbeCandidate(String classInternalName) {
        return !isExcludedClass(classInternalName);
    }

    void checkFireProbePermission(ProbeImpl probe) {
        // TODO: Implement
    }

    /**
     * Fire the listeners associated with the specified probe identifier.
     *
     * @param probeId the probe identifier
     * @param instance the object instance that's the source of the probe
     * @param payload the probe payload
     */
    final void fireProbe(long probeId, @Sensitive Object instance, @Sensitive Object target, @Sensitive Object payload) {
        try {
            //new Exception().printStackTrace();
            ProbeImpl probe = activeProbesById.get(probeId);
            ConcurrentMap<ProbeImpl, Set<ProbeListener>> listenersByProbe = this.listenersByProbe;
            if (shuttingDown || listenersByProbe == null || probe == null) {
                return;
            }
            Set<ProbeListener> listeners = listenersByProbe.get(probe);
            if (listeners == null) {
                return;
            }
            for (ProbeListener listener : listeners) {
                try {
                    Method method = listener.getTargetMethod();
                    //Class<?>[] parameterTypes = method.getParameterTypes();
                    Class<?>[] parameterTypes = listener.getParameterTypes();
                    //Annotation[][] parameterAnnotations = method.getParameterAnnotations();
                    //Annotation[][] parameterAnnotations = listener.getparameterannotations();//Performance Improvement
                    Object[] args = listener.getdefaultvalues();
                    if (args != null)
                        args = Arrays.copyOf(args, args.length);
                    boolean validated = listener.isValidated();
                    ArrayList<Integer> annList = listener.getAnnotationsList();
                    for (int i = 0; i < parameterTypes.length; i++) {

                        switch (annList.get(i)) {
                            case MonitorConstants.This:
                                if (validated || parameterTypes[i].isInstance(instance)) {
                                    args[i] = instance;
                                }
                                break;
                            case MonitorConstants.Args:
                                if (validated || Object[].class.isInstance(payload)) {
                                    args[i] = payload;
                                }
                                break;
                            case MonitorConstants.Elapsed:
                                break;
                            case MonitorConstants.Returned:
                                if (validated || parameterTypes[i].isInstance(payload)) {
                                    args[i] = payload;
                                }
                                break;
                            case MonitorConstants.FieldValue:
                                if (parameterTypes[i].isInstance(payload)) {
                                    args[i] = payload;
                                }
                                break;
                            case MonitorConstants.TargetInstance:
                                if (parameterTypes[i].isInstance(target)) {
                                    args[i] = target;
                                }
                                break;
                            case MonitorConstants.TargetMember:
                                break;
                            case MonitorConstants.Thrown:
                                if (Throwable.class.isInstance(payload) && parameterTypes[i].isInstance(payload)) {
                                    args[i] = payload;
                                }
                                break;
                            case MonitorConstants.Arg:
                                break;
                            default:
                                break;
                        }
//Below code causes performance overhead ..specifically the instanceof tag is causing overhead hence using switch case which is based on
//                        structures built the ProbeListener constructor
//                        Annotation[] annotations = parameterAnnotations[i];
//                        for (Annotation a : annotations) {
//                            if (a instanceof This) {
//                                if (parameterTypes[i].isInstance(instance)) {
//                                    System.out.println("i is " + i + " ins is This");
//                                    args[i] = instance;
//                                }
//                            } else if (a instanceof Args) { // @Args
//                                // payload instance of Object[] and Object[] can be assigned to parameter
//                                if (Object[].class.isInstance(payload) && parameterTypes[i].isAssignableFrom(Object[].class)) {
//                                    args[i] = payload;
//                                }
//                            }
//                            else if (a instanceof Elapsed) {
//                                // TODO
//                                System.out.println("elapsed");
//                                //args[i] = getDefaultValue(parameterTypes[i]);
//                            } else if (a instanceof Returned) {
//                                System.out.println("Returned");
//                                if (parameterTypes[i].isInstance(payload)) {
//                                    args[i] = payload;
//                                }
//                            } else if (a instanceof FieldValue) {
//                                System.out.println("FieldValue");
//                                if (parameterTypes[i].isInstance(payload)) {
//                                    args[i] = payload;
//                                }
//                            } else if (a instanceof TargetInstance) {
//                                System.out.println("TargetInstance");
//                                if (parameterTypes[i].isInstance(target)) {
//                                    args[i] = target;
//                                }
//                            } else if (a instanceof TargetMember) {
//                                System.out.println("TargetInstance");
//                                // TODO
//                                // args[i] = getDefaultValue(parameterTypes[i]);
//                            } else if (a instanceof Thrown) {
//                                System.out.println("Thrown");
//                                if (Throwable.class.isInstance(payload) && parameterTypes[i].isInstance(payload)) {
//                                    args[i] = payload;
//                                }
//                            } else if (a instanceof Arg) { // @Arg(n)
//                                System.out.println("Arg");
//                                Arg arg = (Arg) a;
//                                int index = arg.value() - 1;
//                                Object[] params = null;
//                                if (Object[].class.isInstance(payload)) {
//                                    params = (Object[]) payload;
//                                    if (params.length >= index && parameterTypes[i].isInstance(params[index])) {
//                                        args[i] = params[index];
//                                    }
//                                }
//                            }
//                        }
//                    }
                    }
                    listener.setValidated(true);
                    method.invoke(listener.getProbeTarget(), args);
                } catch (Throwable t) {
                    if (tc.isDebugEnabled()) {
                        Tr.debug(tc, t.getMessage());
                    }
                }
            }
        } catch (Throwable t) {
            Tr.error(tc, "PMI9999E", t.toString());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.ibm.websphere.monitor.MonitorManager#updateNonExcludedClassesSet(java.lang.String)
     */
    //D89497-Call to this method is done from MFE to update the set which contains the list of classes which needs to be instrumented
    @Override
    public void updateNonExcludedClassesSet(String className) {
        if (this.probeMonitorSet != null) {
            this.probeMonitorSet.add(className);
        }
    }

}
