/*******************************************************************************
 * Copyright (c) 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.crypto.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Arrays;

import org.junit.Test;

/**
 * Tests for the password utility class.
 */
public class PasswordHashGeneratorTest {

    /**
     * Test generateSalt
     */
    @Test
    public void testGenerateSaltDefault() {
        byte[] output = PasswordHashGenerator.generateSalt(null);
        // check the output is 8 bytes.
        assertTrue("the generated salt should not be null", (output != null));
        try {
            Thread.sleep(100);
        } catch (Exception e) {
            //do nothing
        }
        assertEquals(8, output.length);
        byte[] output2 = PasswordHashGenerator.generateSalt(null);
        assertFalse("two auto generated salt should be different", Arrays.equals(output, output2));
    }

    /**
     * Test generateSalt with parameter
     */
    @Test
    public void testGenerateSaltGiven() {
        String salt = "salt";
        byte[] output = PasswordHashGenerator.generateSalt(salt);
        // check the output is 4 bytes
        assertTrue("the generated salt should not be null", (output != null));
        assertEquals(4, output.length);
        byte[] output2 = PasswordHashGenerator.generateSalt(salt);
        assertTrue("two specific salt should be the same", Arrays.equals(output, output2));
    }

    /**
     * Test digest with invalid algorithm
     */
    @Test
    public void testDigestWithInvalidAlgorithm() {
        // if the parameters are all the same, the same result should be generated.
        String plain = "password";
        String salt = "salt"; //cannot be autogenerated in order to compare with pre-hashed data.
        String algorithm = "PBKDF222";
        int iteration = 989;
        int length = 256;
        try {
            PasswordHashGenerator.digest(plain.toCharArray(), salt.getBytes(), algorithm, iteration, length);
        } catch (InvalidPasswordCipherException ipce) {
            // normal
        } catch (Exception e) {
            fail("invalid exception.");
        }
    }

    /**
     * Test default values
     */
    @Test
    public void testDefaultValues() {
        // check iteration
        assertEquals(6384, PasswordHashGenerator.getDefaultIteration());
        // check algorithm
        assertTrue("default algorithm should be PBKDF2WithHmacSHA1", "PBKDF2WithHmacSHA1".equals(PasswordHashGenerator.getDefaultAlgorithm()));
        // check output length
        assertEquals(256, PasswordHashGenerator.getDefaultOutputLength());
    }

    /**
     * Test digest with hashed data
     */
    @Test
    public void testDigestWithDefaults() {
        // if the parameters are all the same, the same result should be generated.
        String plain = "password";
        String salt = "salt"; //cannot be autogenerated in order to compare with pre-hashed data.
        String algorithm = "PBKDF2WithHmacSHA1";
        int iteration = 6384;
        int length = 256;
        byte expect[] = { 0x0A, (byte) (0xFC & 0xFF), (byte) (0xB9 & 0xFF), (byte) (0xE1 & 0xFF), (byte) (0xC2 & 0xFF), 0x1A, (byte) (0xE2 & 0xFF), (byte) (0xCA & 0xFF), 0x56,
                         0x53, (byte) (0xF1 & 0xFF), (byte) (0xFD & 0xFF), 0x73, (byte) (0xC5 & 0xFF), (byte) (0x88 & 0xFF), (byte) (0x8D & 0xFF), 0x32, 0x0A,
                         (byte) (0xFE & 0xFF), (byte) (0x97 & 0xFF), (byte) (0x99 & 0xFF), (byte) (0xDB & 0xFF), 0x11, (byte) (0xFA & 0xFF), 0x13, (byte) (0xB5 & 0xFF), 0x16,
                         0x31, 0x6B, (byte) (0xB3 & 0xFF), (byte) (0xFD & 0xFF), 0x07 };

        try {
            byte[] output = PasswordHashGenerator.digest(plain.toCharArray(), salt.getBytes(), algorithm, iteration, length);
            for (int i = 0; i < expect.length; i++) {
                if (expect[i] != output[i]) {
                    System.out.println("value is different i:" + i);
                }
            }

            assertTrue("output should be the same as pregenerated one.", Arrays.equals(output, expect));
        } catch (Exception e) {
            fail("an excetption is caught :" + e.getMessage());
        }
    }

    /**
     * Test digest with hashed data
     */
    @Test
    public void testDigestWithSpecificValues() {
        // if the parameters are all the same, the same result should be generated.
        String plain = "password2";
        String salt = "salt2"; //cannot be autogenerated in order to compare with pre-hashed data.
        String algorithm = "PBKDF2WithHmacSHA1"; // this is the only one which works properly.
        int iteration = 989;
        int length = 64;
        byte expected[] = { (byte) (0x90 & 0xFF), (byte) (0xCC & 0xFF), 0x41, 0x0D, 0x6D, (byte) (0x93 & 0xFF), (byte) (0xD1 & 0xFF), 0x1A };

        try {
            byte[] output = PasswordHashGenerator.digest(plain.toCharArray(), salt.getBytes(), algorithm, iteration, length);
            assertTrue("output should be the same as pregenerated one.", Arrays.equals(output, expected));
        } catch (Exception e) {
            fail("An exception is caught: " + e.getMessage());
        }
    }

}
