/*******************************************************************************
 * Copyright (c) 2020 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package com.ibm.ws.jpa.fvt.ordercolumns;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.List;

import javax.persistence.Query;

import org.junit.Assert;

import com.ibm.ws.jpa.fvt.ordercolumns.model.BOrderNameEntity;
import com.ibm.ws.jpa.fvt.ordercolumns.model.BOrderNameXMLEntity;
import com.ibm.ws.jpa.fvt.ordercolumns.model.OrderColumnEntity;
import com.ibm.ws.jpa.fvt.ordercolumns.model.UOrderNameEntity;
import com.ibm.ws.jpa.fvt.ordercolumns.model.XMLOrderColumnEntity;
import com.ibm.ws.testtooling.testinfo.TestExecutionContext;
import com.ibm.ws.testtooling.testlogic.AbstractTestLogic;
import com.ibm.ws.testtooling.testlogic.JPAEntityClassEnum;
import com.ibm.ws.testtooling.vehicle.resources.JPAResource;
import com.ibm.ws.testtooling.vehicle.resources.TestExecutionResources;

/*
 * Testing logic to validate the following @OrderColumn specification sections:
 *
 * 11.1.39 Order Column Annotation
 *
 * 1) The persistence provider is responsible for updating the ordering upon flushing
 *    to the database to reflect any insertion, deletion, or reordering affecting the list.
 *
 * 2) The OrderColumn annotation is specified on a one-to-many or many-to-many
 *    relationship or on an element collection.
 *
 * 3) The OrderColumn annotation is specified on the side of the relationship that
 *    references the collection that is to be ordered.
 *
 * 4) The OrderBy annotation is not used when OrderColumn is specified.
 *
 * 5) If name is not specified, the column name is the concatenation of the following:
 *    the name of the referencing relationship property or field of the referencing
 *    entity or embeddable class; "_"; "ORDER".
 *
 * 6) The order column must be of integral type.
 *
 * 7) The persistence provider must maintain a contiguous (non-sparse) ordering of the
 *    values of the order column when updating the association or element collection.
 *
 * 8) The ordering column value for the first element of the list must be 0.
 *
 * 9) Annotation Attributes:
 *     name - name (Optional) The name of the ordering column.
 *            The concatenation of the name of the referencing property or field;
 *            "_"; "ORDER".
 *     nullable (Optional) Whether the database column is nullable.
 *     insertable (Optional) Whether the column is included in SQL INSERT statements
 *                  generated by the persistence provider.
 *     updatable (Optional) Whether the column is included in SQL UPDATE statements
 *                generated by the persistence provider.
 *     columnDefinition (Optional) The SQL fragment that is used when generating the
 *                 DDL for the column.
 *     table (Optional) The name of the table that contains the column.
 *                The table element specifies the table containing the order column.
 *                By default: if the relationship is a many-to-many or unidirectional
 *                one-to-many relationship, the table is the join table for the relationship;
 *                if the relationship is a bidirectional one-to-many or unidirectional
 *                one-to-many mapped by a join column, the table is the primary table
 *                for the entity on the many side of the relationship; if the ordering is
 *                for a collection of elements, the table is the collection table for
 *                the element collection.
 *
 *  12.3 ORM Schema
 *      <xsd:complexType name="order-column">
 *          <xsd:annotation>
 *              <xsd:documentation>
 *                  @Target({METHOD, FIELD}) @Retention(RUNTIME)
 *                  public @interface OrderColumn {
 *                      String name() default "";
 *                      boolean nullable() default true;
 *                      boolean insertable() default true;
 *                      boolean updatable() default true;
 *                      String columnDefinition() default "";
 *                      String table() default "";
 *                  }
 *              </xsd:documentation>
 *          </xsd:annotation>
 *          <xsd:attribute name="name" type="xsd:string"/>
 *          <xsd:attribute name="nullable" type="xsd:boolean"/>
 *          <xsd:attribute name="insertable" type="xsd:boolean"/>
 *          <xsd:attribute name="updatable" type="xsd:boolean"/>
 *          <xsd:attribute name="column-definition" type="xsd:string"/>
 *          <xsd:attribute name="table" type="xsd:string"/>
 *      </xsd:complexType>
 */
public class OrderColumnTestLogic extends AbstractTestLogic {

    public enum EntityType {
        XML, Annotated
    };

    private enum OrderColumnEntityClasses implements JPAEntityClassEnum {
        OrderColumnEntity(OrderColumnEntity.class),
        XMLOrderColumnEntity(XMLOrderColumnEntity.class),
        UOrderNameEntity(UOrderNameEntity.class),
        BOrderNameEntity(BOrderNameEntity.class),
        BOrderNameXMLEntity(BOrderNameXMLEntity.class);

        private Class<?> clazz;
        private String fullEntityName;
        private String entityName;

        OrderColumnEntityClasses(Class<?> clazz) {
            this.clazz = clazz;
            fullEntityName = clazz.getName();
            entityName = fullEntityName.substring(getEntityClassName()
                            .lastIndexOf('.') + 1);
        }

        public Class<?> getEntityClass() {
            return clazz;
        }

        @Override
        public String getEntityClassName() {
            return fullEntityName;
        }

        @Override
        public String getEntityName() {
            return entityName;
        }
    }

    public OrderColumnTestLogic() {
    }

    /**
     * Test Logic: testOrderColumnMain
     *
     */
    public static final String TestOrderColumnUni = "testOrderColumnUni";
    public static final int Expected_TestOrderColumnUni_Points = 77;

    public void testOrderColumnUni(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                                   Object managedComponentObject) throws Throwable {

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail("OrderColumnTestLogic.testOrderColumnUni(): Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        // Fetch JPA Resources
        JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Identify the target entity type verified by this test case.
        String entityTypeStr = (String) testExecCtx.getProperties().get("EntityType");
        EntityType entityType = EntityType.valueOf(entityTypeStr);
        Hashtable<String, java.io.Serializable> testData = new Hashtable<String, java.io.Serializable>(testExecCtx.getProperties());
        switch (entityType) {
            case Annotated:
                doTestOrderColumn(jpaResource, testData,
                                  OrderColumnEntityClasses.OrderColumnEntity,
                                  UOrderNameEntity.class);
                break;
            case XML:
                doTestOrderColumn(jpaResource, testData,
                                  OrderColumnEntityClasses.XMLOrderColumnEntity,
                                  UOrderNameEntity.class);
                break;
            default:
                Assert.fail("Invalid entity type specified ('"
                            + entityType + "'). Cannot execute the test.");
        }
    }

    public static final String TestOrderColumnBi = "testOrderColumnBi";
    public static final int Expected_TestOrderColumnBi_Points = 77;

    public void testOrderColumnBi(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                                  Object managedComponentObject) throws Throwable {

        // Verify parameters

        if (testExecCtx == null || testExecResources == null) {
            Assert.fail("OrderColumnTestLogic.testOrderColumnBi(): Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        // Fetch JPA Resources
        JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Identify the target entity type verified by this test case.
        String entityTypeStr = (String) testExecCtx.getProperties().get("EntityType");
        EntityType entityType = EntityType.valueOf(entityTypeStr);
        Hashtable<String, java.io.Serializable> testData = new Hashtable<String, java.io.Serializable>(testExecCtx.getProperties());
        switch (entityType) {
            case Annotated:
                doTestOrderColumn(jpaResource, testData,
                                  OrderColumnEntityClasses.OrderColumnEntity,
                                  BOrderNameEntity.class);
                break;
            case XML:
                doTestOrderColumn(jpaResource, testData,
                                  OrderColumnEntityClasses.XMLOrderColumnEntity,
                                  BOrderNameXMLEntity.class);
                break;
            default:
                Assert.fail("Invalid entity type specified ('"
                            + entityType + "'). Cannot execute the test.");
        }
    }

    public static final String TestOrderColumnElement = "testOrderColumnElement";
    public static final int Expected_TestOrderColumnElement_Points = 77;

    public void testOrderColumnElement(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                                       Object managedComponentObject) throws Throwable {

        // Verify parameters

        if (testExecCtx == null || testExecResources == null) {
            Assert.fail("OrderColumnTestLogic.testOrderColumnBi(): Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        // Fetch JPA Resources
        JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Identify the target entity type verified by this test case.
        String entityTypeStr = (String) testExecCtx.getProperties().get("EntityType");
        EntityType entityType = EntityType.valueOf(entityTypeStr);
        Hashtable<String, java.io.Serializable> testData = new Hashtable<String, java.io.Serializable>(testExecCtx.getProperties());
        switch (entityType) {
            case Annotated:
                doTestOrderColumn(jpaResource, testData,
                                  OrderColumnEntityClasses.OrderColumnEntity, String.class);
                break;
            case XML:
                doTestOrderColumn(jpaResource, testData,
                                  OrderColumnEntityClasses.XMLOrderColumnEntity,
                                  String.class);
                break;
            default:
                Assert.fail("Invalid entity type specified ('" + entityType + "'). Cannot execute the test.");
        }
    }

    private static final String Element_1_App = "App";
    private static final String Element_2_Boy = "Boy";
    private static final String Element_3_Cat = "Cat";
    private static final String Element_4_Dog = "Dog";
    private static final String Element_5_Egg = "Egg";
    private static final String Element_6_Fox = "Fox";

    private <C, E> void doTestOrderColumn(JPAResource jpaRW,
                                          Hashtable<String, java.io.Serializable> testData,
                                          OrderColumnEntityClasses entityType, Class<E> elementClass) throws Throwable {

        System.out.println("** Test element modification from @OrderColumn collection using add and remove elements.");
        doTestOrderColumnAddRemoveElement(jpaRW, testData, entityType, elementClass);
        System.out.println("** Test element modification from @OrderColumn collection using collection replacement.");
        doTestOrderColumnSetElements(jpaRW, testData, entityType, elementClass);
    }

    private <C, E> void doTestOrderColumnAddRemoveElement(
                                                          JPAResource jpaRW,
                                                          Hashtable<String, java.io.Serializable> testData,
                                                          OrderColumnEntityClasses entityType, Class<E> elementClass) throws Throwable {

        // Execute the test case within a try block for proper error recovery
        try {
            Class<C> columnEntityClass = (Class<C>) Class.forName(entityType
                            .getEntityClassName());
            String columnEntityClassName = entityType.getEntityName();
            String elementClassName = elementClass.getName()
                            .substring(
                                       elementClass.getName().lastIndexOf('.') + 1);

            String listFieldName = (String) testData.get("listFieldName");
            // build *OrderColumnEntity.setId(int) Method
            String setIdMethodName = "setId";
            Method setIdMethod = columnEntityClass.getMethod(setIdMethodName,
                                                             int.class);
            // build *OrderColumnEntity.get*Names() Method
            String getNamesMethodName = toBeanMethod("get", listFieldName);
            Method getNamesMethod = columnEntityClass
                            .getMethod(getNamesMethodName);
            // build *OrderColumnEntity.add*Name() Method
            String addNameMethodName = toBeanMethod("add", listFieldName); //, 1);
            Method addNameMethod = columnEntityClass.getMethod(
                                                               addNameMethodName, elementClass);
            // build *OrderColumnEntity.remove*Name() Method
            String removeNameMethodName = toBeanMethod("remove", listFieldName); //, 1);
            Method removeNameMethod = columnEntityClass.getMethod(
                                                                  removeNameMethodName, int.class);
            // build *OrderColumnEntity.remove*Name() Method
            String insertNameMethodName = toBeanMethod("insert", listFieldName); // , 1);
            Method insertNameMethod = columnEntityClass.getMethod(
                                                                  insertNameMethodName, int.class, elementClass);

            // build BOrderNameEntity.setColumn() Method
            String setColumnMethodName = elementClass == BOrderNameEntity.class ? "setColumn" : "setXmlColumn";
            Method setColumnMethod = elementClass == BOrderNameEntity.class
                                     || elementClass == BOrderNameXMLEntity.class ? elementClass
                                                     .getMethod(setColumnMethodName, columnEntityClass) : null;
            // build BOrderNameEntity.addColumns() Method
            String addColumnsMethodName = elementClass == BOrderNameEntity.class ? "addColumns" : "addXmlColumns";
            Method addColumnsMethod = elementClass == BOrderNameEntity.class
                                      || elementClass == BOrderNameXMLEntity.class ? elementClass
                                                      .getMethod(addColumnsMethodName, columnEntityClass) : null;
            // build BOrderNameEntity.removeColumns() Method
            String removeColumnsMethodName = elementClass == BOrderNameEntity.class ? "removeColumns" : "removeXmlColumns";
            Method removeColumnsMethod = elementClass == BOrderNameEntity.class
                                         || elementClass == BOrderNameXMLEntity.class ? elementClass
                                                         .getMethod(removeColumnsMethodName, columnEntityClass) : null;

            // Cleanup the database for executing the test
            System.out.println("Cleaning up database before executing test...");
            cleanupDatabase(jpaRW.getEm(), jpaRW.getTj(), OrderColumnEntityClasses.values());
            System.out.println("Database cleanup complete.\n");

            // Clear persistence context
            System.out.println("Clearing persistence context...");
            jpaRW.getEm().clear();

            System.out.println("Create " + columnEntityClassName + "[1].");
            C newEntity = (C) constructNewEntityObject(entityType);
            setIdMethod.invoke(newEntity, 1);

            Constructor<E> elementConstrctor = elementClass
                            .getConstructor(String.class);
            E new1App = elementConstrctor.newInstance(Element_1_App);
            E new2Boy = elementConstrctor.newInstance(Element_2_Boy);
            E new3Cat = elementConstrctor.newInstance(Element_3_Cat);

            System.out.println("Add " + elementClassName + "[" + new2Boy
                               + "] to " + columnEntityClassName + "[1].");
            beginTx(jpaRW);
            if (elementClass != String.class) {
                jpaRW.getEm().persist(new2Boy);
                if (elementClass == BOrderNameEntity.class
                    || elementClass == BOrderNameXMLEntity.class) {
                    if (listFieldName.charAt(1) == 'o') {
                        setColumnMethod.invoke(new2Boy, newEntity);
                    } else {
                        addColumnsMethod.invoke(new2Boy, newEntity);
                    }
                }
            }
            addNameMethod.invoke(newEntity, new2Boy);
            jpaRW.getEm().persist(newEntity);
            commitTx(jpaRW);

            C foundEntity;
            beginTx(jpaRW);
            verifyUOrderNameEntityWithQuery(jpaRW, columnEntityClass,
                                            elementClass, listFieldName, Element_2_Boy);
            foundEntity = verifyElementCollection(jpaRW,
                                                  listFieldName, columnEntityClass, columnEntityClassName,
                                                  elementClass, elementClassName, getNamesMethod, Element_2_Boy);

            System.out.println("Add " + elementClassName + "[" + new1App
                               + "] to " + columnEntityClassName + "[1].");
            if (elementClass != String.class) {
                jpaRW.getEm().persist(new1App);
                if (elementClass == BOrderNameEntity.class
                    || elementClass == BOrderNameXMLEntity.class) {
                    if (listFieldName.charAt(1) == 'o') {
                        setColumnMethod.invoke(new1App, foundEntity);
                    } else {
                        addColumnsMethod.invoke(new1App, foundEntity);
                    }
                }
            }
            addNameMethod.invoke(foundEntity, new1App);
            commitTx(jpaRW);

            beginTx(jpaRW);
            verifyUOrderNameEntityWithQuery(jpaRW, columnEntityClass,
                                            elementClass, listFieldName, Element_2_Boy, Element_1_App);
            foundEntity = verifyElementCollection(jpaRW,
                                                  listFieldName, columnEntityClass, columnEntityClassName,
                                                  elementClass, elementClassName, getNamesMethod, Element_2_Boy,
                                                  Element_1_App);

            System.out.println("Add " + elementClassName + "[" + new3Cat
                               + "] to " + columnEntityClassName + "[1].");
            if (elementClass != String.class) {
                jpaRW.getEm().persist(new3Cat);
                if (elementClass == BOrderNameEntity.class
                    || elementClass == BOrderNameXMLEntity.class) {
                    if (listFieldName.charAt(1) == 'o') {
                        setColumnMethod.invoke(new3Cat, foundEntity);
                    } else {
                        addColumnsMethod.invoke(new3Cat, foundEntity);
                    }
                }
            }
            addNameMethod.invoke(foundEntity, new3Cat);
            commitTx(jpaRW);

            beginTx(jpaRW);
            verifyUOrderNameEntityWithQuery(jpaRW, columnEntityClass,
                                            elementClass, listFieldName, Element_2_Boy, Element_1_App, Element_3_Cat);
            foundEntity = verifyElementCollection(jpaRW,
                                                  listFieldName, columnEntityClass, columnEntityClassName,
                                                  elementClass, elementClassName, getNamesMethod, Element_2_Boy,
                                                  Element_1_App, Element_3_Cat);

            System.out.println("Remove " + elementClassName + "[" + new1App
                               + "] from " + columnEntityClassName + "[1].");
            try {
                E element = (E) removeNameMethod.invoke(foundEntity, 1);
                if (elementClass == BOrderNameEntity.class
                    || elementClass == BOrderNameXMLEntity.class) {
                    if (listFieldName.charAt(1) == 'o') {
                        setColumnMethod.invoke(element, (C) null);
                    } else {
                        removeColumnsMethod.invoke(element, foundEntity);
                    }
                }
            } catch (Exception e) {
                e.printStackTrace(System.out);
                Assert.fail("Unable to remove element at index 1: " + e);
            }
            commitTx(jpaRW);

            beginTx(jpaRW);
            verifyUOrderNameEntityWithQuery(jpaRW, columnEntityClass,
                                            elementClass, listFieldName, Element_2_Boy, Element_3_Cat);
            foundEntity = verifyElementCollection(jpaRW,
                                                  listFieldName, columnEntityClass, columnEntityClassName,
                                                  elementClass, elementClassName, getNamesMethod, Element_2_Boy,
                                                  Element_3_Cat);

            System.out.println("Remove " + elementClassName + "[" + new2Boy
                               + "] from " + columnEntityClassName + "[1].");
            try {
                E element = (E) removeNameMethod.invoke(foundEntity, 0);
                if (elementClass == BOrderNameEntity.class
                    || elementClass == BOrderNameXMLEntity.class) {
                    if (listFieldName.charAt(1) == 'o') {
                        setColumnMethod.invoke(element, (C) null);
                    } else {
                        removeColumnsMethod.invoke(element, foundEntity);
                    }
                }
            } finally {

            }
//            catch (Exception e) {
//                e.printStackTrace(System.out);
//                Assert.fail("Unable to remove element at index 1: " + e);
//            }
            commitTx(jpaRW);

            beginTx(jpaRW);
            verifyUOrderNameEntityWithQuery(jpaRW, columnEntityClass,
                                            elementClass, listFieldName, Element_3_Cat);
            foundEntity = verifyElementCollection(jpaRW,
                                                  listFieldName, columnEntityClass, columnEntityClassName,
                                                  elementClass, elementClassName, getNamesMethod, Element_3_Cat);

            System.out.println("Insert " + elementClassName + "[" + new1App
                               + "] to " + columnEntityClassName + "[1].");
            new1App = (E) (elementClass == String.class ? Element_1_App : jpaRW
                            .getEm()
                            .find(elementClass, 1));
            insertNameMethod.invoke(foundEntity, 0, new1App);
            commitTx(jpaRW);

            beginTx(jpaRW);
            verifyUOrderNameEntityWithQuery(jpaRW, columnEntityClass,
                                            elementClass, listFieldName, Element_1_App, Element_3_Cat);
            foundEntity = verifyElementCollection(jpaRW,
                                                  listFieldName, columnEntityClass, columnEntityClassName,
                                                  elementClass, elementClassName, getNamesMethod, Element_1_App,
                                                  Element_3_Cat);

            System.out.println("Remove " + elementClassName + "[" + new3Cat
                               + "] from " + columnEntityClassName + "[1].");
            try {
                E element = (E) removeNameMethod.invoke(foundEntity, 1);
                if (elementClass == BOrderNameEntity.class
                    || elementClass == BOrderNameXMLEntity.class) {
                    if (listFieldName.charAt(1) == 'o') {
                        setColumnMethod.invoke(element, (C) null);
                    } else {
                        removeColumnsMethod.invoke(element, foundEntity);
                    }
                }
            } finally {

            }
//            catch (Exception e) {
//                e.printStackTrace(System.out);
//                Assert.fail("Unable to remove element at index 1: " + e);
//            }
            commitTx(jpaRW);

            beginTx(jpaRW);
            verifyUOrderNameEntityWithQuery(jpaRW, columnEntityClass,
                                            elementClass, listFieldName, Element_1_App);
            foundEntity = verifyElementCollection(jpaRW,
                                                  listFieldName, columnEntityClass, columnEntityClassName,
                                                  elementClass, elementClassName, getNamesMethod, Element_1_App);
            rollbackTx(jpaRW);

        } finally {

        }
    }

    private <E> void verifyUOrderNameEntityWithQuery(JPAResource jpaRW,
                                                     Class<?> columnEntityClass,
                                                     Class<E> elementClass, String colName, String... expectedOrderedStrings) throws Throwable {
        String columnEntityClassName = columnEntityClass.getName();
        columnEntityClassName = columnEntityClassName
                        .substring(columnEntityClassName.lastIndexOf('.') + 1);
        System.out.println("Query " + columnEntityClassName + " and verify '"
                           + colName + "' collection has " + expectedOrderedStrings.length
                           + " elements in this order: "
                           + Arrays.toString(expectedOrderedStrings));
        System.out.println("Clearing persistence context...");
        jpaRW.getEm().clear();
        int idx = 0;
        for (String expectedOrderedString : expectedOrderedStrings) {
            Query q = jpaRW.getEm()
                            .createQuery(
                                         "select w from " + columnEntityClassName + " o join o."
                                         + colName + " w where index(w) = " + idx);
            List<E> res = q.getResultList();
            Assert.assertEquals("  Verify query returns 1 element for index " + idx, 1, res.size());
            if (res.size() == 1) {
                Object oo = res.get(0);
                Assert.assertEquals("  Verify element type is "
                                    + elementClass.getName(), elementClass.getName(),
                                    oo
                                                    .getClass()
                                                    .getName());
                String name;
                try {
                    name = (String) elementClass.getMethod("toString")
                                    .invoke(
                                            oo);
                    Assert.assertEquals("  Verify element value is '"
                                        + expectedOrderedString + "'", expectedOrderedString,
                                        name);
                } catch (Exception e) {
                    Assert.fail("  Caught unexpected exception:"
                                + e.getMessage());
                }
            }
            ++idx;
        }
    }

    private <C, E> C verifyElementCollection(JPAResource jpaRW,
                                             String colName, Class<C> columnEntityClass,
                                             String columnEntityClassName, Class<E> elementClass,
                                             String elementClassName, Method getNamesMethod,
                                             String... expectedElementStrings) throws Exception {
        System.out.println("find(" + columnEntityClassName + ",1) and verify '"
                           + colName + "' collection has " + expectedElementStrings.length
                           + " elements in this order: "
                           + Arrays.toString(expectedElementStrings));

        System.out.println("Clearing persistence context...");
        jpaRW.getEm().clear();
        C foundEntity = jpaRW.getEm().find(columnEntityClass, 1);
        List<E> names = (List<E>) getNamesMethod.invoke(foundEntity);
        Assert.assertEquals("  Verify collection count = "
                            + expectedElementStrings.length, expectedElementStrings.length,
                            names.size());
        if (expectedElementStrings.length == names.size()) {
            int idx = 0;
            for (String expectedString : expectedElementStrings) {
                Assert.assertEquals("  Verify " + elementClassName + "[" + idx
                                    + "] == '" + expectedString + "'", expectedString,
                                    names
                                                    .get(idx)
                                                    .toString());
                ++idx;
            }
        } else {
            Assert.fail("  Expecting element collection "
                        + Arrays.toString(expectedElementStrings) + ", but got "
                        + names);
        }
        return foundEntity;
    }

    private <C, E> void setAndVerifyElementsInOrderColumnEntity(
                                                                JPAResource jpaRW, String listFieldName,
                                                                Class<C> columnEntityClass,
                                                                Method getNamesMethod, Method setNamesMethod, Class<E> elementClass,
                                                                String elementClassName, Method setColumnMethod,
                                                                Method addColumnsMethod, E... expectedElementStrings) throws Exception {

        System.out.println("set" + listFieldName + "() and verify "
                           + "collection has " + expectedElementStrings.length
                           + " elements in this order: "
                           + Arrays.toString(expectedElementStrings));

        beginTx(jpaRW);
        jpaRW.getEm().clear();
        // clear inverse references from element to column entity, otherwise
        //  the element will continue to stay in the collection.
        if (elementClass == BOrderNameEntity.class
            || elementClass == BOrderNameXMLEntity.class) {
            Query q = jpaRW.getEm()
                            .createQuery(
                                         "select e from " + elementClassName + " e");
            List<E> elements = q.getResultList();
            for (E element : elements) {
                if (listFieldName.charAt(1) == 'o') {
                    setColumnMethod.invoke(element, (E) null);
                } else {
                    addColumnsMethod.invoke(element, (E) null);
                }
            }
        }
        C foundEntity = jpaRW.getEm().find(columnEntityClass, 2);
        List<E> names = new ArrayList<E>();
        for (E element : expectedElementStrings) {
            if (element instanceof String) {
                names.add(element);
            } else {
                int id = element.toString().charAt(0) - 'A' + 1;
                E foundElement = jpaRW.getEm().find(elementClass, id);
                names.add(foundElement);
                if (elementClass == BOrderNameEntity.class
                    || elementClass == BOrderNameXMLEntity.class) {
                    if (listFieldName.charAt(1) == 'o') {
                        setColumnMethod.invoke(foundElement, foundEntity);
                    } else {
                        addColumnsMethod.invoke(foundElement, foundEntity);
                    }
                }
            }
        }
        setNamesMethod.invoke(foundEntity, names);
        commitTx(jpaRW);

        System.out.println("Clearing persistence context...");
        jpaRW.getEm().clear();
        foundEntity = jpaRW.getEm().find(columnEntityClass, 2);
        names = (List<E>) getNamesMethod.invoke(foundEntity);

        Assert.assertEquals("  Verify collection count = "
                            + expectedElementStrings.length, expectedElementStrings.length,
                            names.size());
        if (expectedElementStrings.length == names.size()) {
            int idx = 0;
            for (E expectedString : expectedElementStrings) {
                String expected = expectedString.toString();
                String actual = names.get(idx).toString();
                Assert.assertEquals("  Verify " + elementClassName + "[" + idx
                                    + "] == '" + expected + "' (is '" + actual + "')", expected, actual);
                ++idx;
            }
        } else {
            Assert.fail("  Expecting element collection "
                        + Arrays.toString(expectedElementStrings) + ", but got "
                        + names);
        }
    }

    private <C, E> void doTestOrderColumnSetElements(
                                                     JPAResource jpaRW,
                                                     Hashtable<String, java.io.Serializable> testData,
                                                     OrderColumnEntityClasses entityType, Class<E> elementClass) throws Throwable {

        // Execute the test case within a try block for proper error recovery
        try {
            Class<?> columnEntityClass = Class.forName(entityType
                            .getEntityClassName());
            String columnEntityClassName = entityType.getEntityName();
            String elementClassName = elementClass.getName()
                            .substring(
                                       elementClass.getName().lastIndexOf('.') + 1);

            String listFieldName = (String) testData.get("listFieldName");
            // build *OrderColumnEntity.setId(int) Method
            String setIdMethodName = "setId";
            Method setIdMethod = columnEntityClass.getMethod(setIdMethodName,
                                                             int.class);
            // build *OrderColumnEntity.get*Names() Method
            String getNamesMethodName = toBeanMethod("get", listFieldName);
            Method getNamesMethod = columnEntityClass
                            .getMethod(getNamesMethodName);
            // build *OrderColumnEntity.set*Names() Method
            String setNamesMethodName = toBeanMethod("set", listFieldName);
            Method setNamesMethod = columnEntityClass
                            .getMethod(setNamesMethodName, List.class);
            // build BOrderNameEntity.setColumn() Method
            String setColumnMethodName = elementClass == BOrderNameEntity.class ? "setColumn" : "setXmlColumn";
            Method setColumnMethod = elementClass == BOrderNameEntity.class
                                     || elementClass == BOrderNameXMLEntity.class ? elementClass
                                                     .getMethod(setColumnMethodName, columnEntityClass) : null;
            // build BOrderNameEntity.addColumns() Method
            String addColumnsMethodName = elementClass == BOrderNameEntity.class ? "addColumns" : "addXmlColumns";
            Method addColumnsMethod = elementClass == BOrderNameEntity.class
                                      || elementClass == BOrderNameXMLEntity.class ? elementClass
                                                      .getMethod(addColumnsMethodName, columnEntityClass) : null;

            // Cleanup the database for executing the test
            System.out.println("Cleaning up database before executing test...");
            cleanupDatabase(jpaRW.getEm(), jpaRW.getTj(), OrderColumnEntityClasses.values());
            System.out.println("Database cleanup complete.\n");

            // Clear persistence context
            System.out.println("Clearing persistence context...");
            jpaRW.getEm().clear();

            System.out.println("Create " + columnEntityClassName + "[1].");
            C newEntity = (C) constructNewEntityObject(entityType);
            setIdMethod.invoke(newEntity, 2);

            Constructor<E> elementConstrctor = elementClass
                            .getConstructor(String.class);
            E new4Dog = elementConstrctor.newInstance(Element_4_Dog);
            E new5Egg = elementConstrctor.newInstance(Element_5_Egg);
            E new6Fox = elementConstrctor.newInstance(Element_6_Fox);

            System.out.println("Persist " + elementClassName + "[" + Element_4_Dog
                               + ',' + Element_5_Egg + ',' + Element_6_Fox + "] to "
                               + columnEntityClassName + "[1].");
            beginTx(jpaRW);
            if (elementClass != String.class) {
                jpaRW.getEm().persist(new4Dog);
                jpaRW.getEm().persist(new5Egg);
                jpaRW.getEm().persist(new6Fox);
            }
            jpaRW.getEm().persist(newEntity);
            commitTx(jpaRW);

            setAndVerifyElementsInOrderColumnEntity(jpaRW, listFieldName,
                                                    columnEntityClass, getNamesMethod, setNamesMethod,
                                                    elementClass, elementClassName, setColumnMethod, addColumnsMethod, new5Egg);
            setAndVerifyElementsInOrderColumnEntity(jpaRW, listFieldName,
                                                    columnEntityClass, getNamesMethod, setNamesMethod,
                                                    elementClass, elementClassName, setColumnMethod, addColumnsMethod, new5Egg, new4Dog);
            setAndVerifyElementsInOrderColumnEntity(jpaRW, listFieldName,
                                                    columnEntityClass, getNamesMethod, setNamesMethod,
                                                    elementClass, elementClassName, setColumnMethod, addColumnsMethod, new5Egg, new4Dog, new6Fox);
            setAndVerifyElementsInOrderColumnEntity(jpaRW, listFieldName,
                                                    columnEntityClass, getNamesMethod, setNamesMethod,
                                                    elementClass, elementClassName, setColumnMethod, addColumnsMethod, new5Egg, new6Fox);
            setAndVerifyElementsInOrderColumnEntity(jpaRW, listFieldName,
                                                    columnEntityClass, getNamesMethod, setNamesMethod,
                                                    elementClass, elementClassName, setColumnMethod, addColumnsMethod, new6Fox);
            setAndVerifyElementsInOrderColumnEntity(jpaRW, listFieldName,
                                                    columnEntityClass, getNamesMethod, setNamesMethod,
                                                    elementClass, elementClassName, setColumnMethod, addColumnsMethod, new6Fox, new4Dog);
            setAndVerifyElementsInOrderColumnEntity(jpaRW, listFieldName,
                                                    columnEntityClass, getNamesMethod, setNamesMethod,
                                                    elementClass, elementClassName, setColumnMethod, addColumnsMethod, new4Dog);

        } finally {

        }
    }
}
