/*******************************************************************************
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.threading.internal;

/**
 * Simple class that calculates an exponentially weighted moving average
 * of non-negative durations.
 */
class ThroughputDistribution {

    // 8/10/2012: Change alpha from 0.2 to 0.3
    // 8/11/2012: Change back to 0.2
    /**
     * The weight of the most recently added data. According to something
     * I read, 1/ALPHA determines the average age of the data in the
     * distribution.
     */
    final static double ALPHA = 0.2;

    // 8/11/2012: Generate variance for a single data point
    /**
     * A factor used to estimate the standard deviation from a single data
     * point. This is just an estimate to establish a base line early in
     * the data stream.
     */
    final static double SINGLE_VALUE_STDDEV = 1.0 / 6.0;

    /**
     * The exponentially weighted moving average of the observed data.
     */
    double ewma = 0.0;

    /**
     * The variance of the observed data. The variance is the square
     * of the standard deviation.
     * <p>
     * The value {@code Double.NEGATIVE_INFINITY} indicates no observations
     * have been made.
     */
    double ewmaVariance = Double.NEGATIVE_INFINITY;

    /**
     * The standard deviation of the observed data.
     * <p>
     * The value {@code Double.NEGATIVE_INFINITY} indicates no observations
     * have been made.
     */
    double ewmaStddev = Double.NEGATIVE_INFINITY;

    /**
     * The last controller cycle when this distribution was updated. The controller
     * will discard old data.
     * This is an int - at 1500ms controller cycles, that allows for 100+ years of
     * continuous controller operation (no restarts) before it would recycle.
     */
    int lastUpdate = 0;

    synchronized void setLastUpdate(int update) {
        lastUpdate = update;
    }

    synchronized int getLastUpdate() {
        return lastUpdate;
    }

    /**
     * How many times in a row this distribution has received an outlier value (as
     * determined by the caller).
     */
    int consecutiveOutliers = 0;

    synchronized int incrementAndGetConsecutiveOutliers() {
        consecutiveOutliers++;
        return consecutiveOutliers;
    }

    synchronized void resetConsecutiveOutliers() {
        consecutiveOutliers = 0;
    }

    /**
     * Reset the distribution by throwing away all history and
     * adding a single data point.
     *
     * @param value the single data point that makes up the distribution
     */
    synchronized void reset(double value, int updated) {
        ewma = value;
        ewmaStddev = value * SINGLE_VALUE_STDDEV;
        ewmaVariance = ewmaStddev * ewmaStddev;
        consecutiveOutliers = 0;
        lastUpdate = updated;
    }

    /**
     * Get the standard deviation of the observed data.
     *
     * @return the standard deviation of the observed data
     */
    synchronized double getStddev() {
        if (ewmaStddev == Double.NEGATIVE_INFINITY) {
            return 0.0;
        }
        return ewmaStddev;
    }

    /**
     * Get the variance of the observed data. The variance is the square
     * of the standard deviation.
     *
     * @return the variance of the observed data
     */
    synchronized double getVariance() {
        if (ewmaVariance == Double.NEGATIVE_INFINITY) {
            return 0.0;
        }
        return ewmaVariance;
    }

    /**
     * Get the exponentially weighted moving average of the observed data.
     * This value can be treated as a forecast of future values.
     *
     * @return the moving average of the observed data
     */
    synchronized double getMovingAverage() {
        return ewma;
    }

    /**
     * Add a throughput observation to the distribution, setting lastUpdate
     *
     * @param value the observed throughput
     */
    synchronized void addDataPoint(double value, int updated) {
        // 8/10/2012: Force the standard deviation to 1/3 of the first data point
        // 8/11:2012: Force it to 1/6th of the first data point (more accurate)
        if (ewmaVariance == Double.NEGATIVE_INFINITY) {
            reset(value, updated);
            return;
        }

        double delta = value - ewma;
        double increment = ALPHA * delta;

        ewma += increment;
        ewmaVariance = (1 - ALPHA) * (ewmaVariance + delta * increment);
        ewmaStddev = Math.sqrt(ewmaVariance);

        lastUpdate = updated;
    }

    /**
     * Calculate a z-score for the specified value from the current
     * throughput data. The z-score helps to normalize a distribution
     * to the <em>standard</em> normal distribution required to use
     * cumulative tables to determine probability.
     *
     * @param value the random value to evaluate against the distribution
     *
     * @return the number of standard deviations between the specified value
     *         and the mean
     *
     * @see http://en.wikipedia.org/wiki/Standard_normal_table
     */
    synchronized double getZScore(double value) {
        // Generate a random value between -0.5 and 0.5 if we have no
        // historical data.  This will result in a score that is well
        // within 1 standard deviation from the specified value and
        // avoid divide-by-zero issues.
        if (ewmaVariance == Double.NEGATIVE_INFINITY) {
            return Math.random() - 0.5;
        }

        double score = (value - ewma) / ewmaStddev;

        // Limit the z-score to the range where we've calculated the
        // standard normal distribution probabilities
        if (score < -3.4) {
            return -3.4;
        } else if (score > 3.4) {
            return 3.4;
        }

        return score;
    }

    /**
     * Calculate the probability that the throughput of the next interval
     * will be less than the specified value.
     *
     * @param the random value to be evaluated
     *
     * @return the probability that the throughput of the next interval will
     *         be less than the specified random value
     */
    double getProbabilityLessThan(double value) {
        double zScore = getZScore(value);

        int probabiltyIndex = (int) (zScore * 100.0) + 340;

        return Z_SCORE_PROBABILITY_LESS_THAN[probabiltyIndex];
    }

    /**
     * Calculate the probability that the throughput of the next interval
     * will be greater than the specified value.
     *
     * @param the random value to be evaluated
     *
     * @return the probability that the throughput of the next interval will
     *         be greater than the specified random value
     */
    double getProbabilityGreaterThan(double value) {
        return 1.0 - getProbabilityLessThan(value);
    }

    /**
     * Simple formatting function for the throughput data.
     */
    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder();
        double stddev = (ewmaStddev == Double.NEGATIVE_INFINITY) ? 0.0 : ewmaStddev;

        sb.append(" ewma=").append(String.format("%16.6f", Double.valueOf(ewma)));
        sb.append(" stddev=").append(String.format("%16.6f", Double.valueOf(stddev)));
        sb.append(" updated= ").append(String.format("%10d", Integer.valueOf(lastUpdate)));
        if (consecutiveOutliers > 0)
            sb.append(" consecutiveOutliers=").append(String.format("%2d", Integer.valueOf(consecutiveOutliers)));

        return sb.toString();
    }

    /**
     * Values from the standard normal cumulative distribution table for
     * z-scores ranging from -3.40 to +3.40 in increments of 0.01.
     *
     * This table was generated in Excel with the following formulas:
     *
     * <pre>
     * <tt>
     * B1 = 0, D1 = =B1-3.4, F1 = =NORMDIST(D1,0,1,TRUE)
     * B2 = =B1+1, D2 = =D1+.01, F2 = =NORMDIST(D2,0,1,TRUE)
     * </tt>
     * </pre>
     *
     * @see http://en.wikipedia.org/wiki/Normal_distribution
     * @see http://en.wikipedia.org/wiki/Standard_normal_table
     */
    // @formatter:off
    private static final double Z_SCORE_PROBABILITY_LESS_THAN[] = {
            /* [   0 ]: -3.40= */  0.0003369292657,
            /* [   1 ]: -3.39= */  0.0003494631183,
            /* [   2 ]: -3.38= */  0.0003624291490,
            /* [   3 ]: -3.37= */  0.0003758409184,
            /* [   4 ]: -3.36= */  0.0003897123626,
            /* [   5 ]: -3.35= */  0.0004040578019,
            /* [   6 ]: -3.34= */  0.0004188919495,
            /* [   7 ]: -3.33= */  0.0004342299204,
            /* [   8 ]: -3.32= */  0.0004500872406,
            /* [   9 ]: -3.31= */  0.0004664798561,
            /* [  10 ]: -3.30= */  0.0004834241424,
            /* [  11 ]: -3.29= */  0.0005009369138,
            /* [  12 ]: -3.28= */  0.0005190354332,
            /* [  13 ]: -3.27= */  0.0005377374218,
            /* [  14 ]: -3.26= */  0.0005570610690,
            /* [  15 ]: -3.25= */  0.0005770250424,
            /* [  16 ]: -3.24= */  0.0005976484979,
            /* [  17 ]: -3.23= */  0.0006189510904,
            /* [  18 ]: -3.22= */  0.0006409529837,
            /* [  19 ]: -3.21= */  0.0006636748614,
            /* [  20 ]: -3.20= */  0.0006871379379,
            /* [  21 ]: -3.19= */  0.0007113639686,
            /* [  22 ]: -3.18= */  0.0007363752616,
            /* [  23 ]: -3.17= */  0.0007621946881,
            /* [  24 ]: -3.16= */  0.0007888456944,
            /* [  25 ]: -3.15= */  0.0008163523128,
            /* [  26 ]: -3.14= */  0.0008447391735,
            /* [  27 ]: -3.13= */  0.0008740315156,
            /* [  28 ]: -3.12= */  0.0009042551998,
            /* [  29 ]: -3.11= */  0.0009354367195,
            /* [  30 ]: -3.10= */  0.0009676032132,
            /* [  31 ]: -3.09= */  0.0010007824766,
            /* [  32 ]: -3.08= */  0.0010350029748,
            /* [  33 ]: -3.07= */  0.0010702938547,
            /* [  34 ]: -3.06= */  0.0011066849574,
            /* [  35 ]: -3.05= */  0.0011442068310,
            /* [  36 ]: -3.04= */  0.0011828907431,
            /* [  37 ]: -3.03= */  0.0012227686936,
            /* [  38 ]: -3.02= */  0.0012638734277,
            /* [  39 ]: -3.01= */  0.0013062384488,
            /* [  40 ]: -3.00= */  0.0013498980316,
            /* [  41 ]: -2.99= */  0.0013948872355,
            /* [  42 ]: -2.98= */  0.0014412419173,
            /* [  43 ]: -2.97= */  0.0014889987452,
            /* [  44 ]: -2.96= */  0.0015381952117,
            /* [  45 ]: -2.95= */  0.0015888696474,
            /* [  46 ]: -2.94= */  0.0016410612342,
            /* [  47 ]: -2.93= */  0.0016948100193,
            /* [  48 ]: -2.92= */  0.0017501569287,
            /* [  49 ]: -2.91= */  0.0018071437808,
            /* [  50 ]: -2.90= */  0.0018658133004,
            /* [  51 ]: -2.89= */  0.0019262091322,
            /* [  52 ]: -2.88= */  0.0019883758549,
            /* [  53 ]: -2.87= */  0.0020523589949,
            /* [  54 ]: -2.86= */  0.0021182050404,
            /* [  55 ]: -2.85= */  0.0021859614549,
            /* [  56 ]: -2.84= */  0.0022556766915,
            /* [  57 ]: -2.83= */  0.0023274002067,
            /* [  58 ]: -2.82= */  0.0024011824742,
            /* [  59 ]: -2.81= */  0.0024770749988,
            /* [  60 ]: -2.80= */  0.0025551303304,
            /* [  61 ]: -2.79= */  0.0026354020779,
            /* [  62 ]: -2.78= */  0.0027179449227,
            /* [  63 ]: -2.77= */  0.0028028146328,
            /* [  64 ]: -2.76= */  0.0028900680762,
            /* [  65 ]: -2.75= */  0.0029797632351,
            /* [  66 ]: -2.74= */  0.0030719592187,
            /* [  67 ]: -2.73= */  0.0031667162774,
            /* [  68 ]: -2.72= */  0.0032640958159,
            /* [  69 ]: -2.71= */  0.0033641604067,
            /* [  70 ]: -2.70= */  0.0034669738030,
            /* [  71 ]: -2.69= */  0.0035726009524,
            /* [  72 ]: -2.68= */  0.0036811080092,
            /* [  73 ]: -2.67= */  0.0037925623477,
            /* [  74 ]: -2.66= */  0.0039070325749,
            /* [  75 ]: -2.65= */  0.0040245885428,
            /* [  76 ]: -2.64= */  0.0041453013610,
            /* [  77 ]: -2.63= */  0.0042692434091,
            /* [  78 ]: -2.62= */  0.0043964883481,
            /* [  79 ]: -2.61= */  0.0045271111330,
            /* [  80 ]: -2.60= */  0.0046611880237,
            /* [  81 ]: -2.59= */  0.0047987965971,
            /* [  82 ]: -2.58= */  0.0049400157578,
            /* [  83 ]: -2.57= */  0.0050849257490,
            /* [  84 ]: -2.56= */  0.0052336081636,
            /* [  85 ]: -2.55= */  0.0053861459541,
            /* [  86 ]: -2.54= */  0.0055426234431,
            /* [  87 ]: -2.53= */  0.0057031263330,
            /* [  88 ]: -2.52= */  0.0058677417153,
            /* [  89 ]: -2.51= */  0.0060365580804,
            /* [  90 ]: -2.50= */  0.0062096653258,
            /* [  91 ]: -2.49= */  0.0063871547649,
            /* [  92 ]: -2.48= */  0.0065691191355,
            /* [  93 ]: -2.47= */  0.0067556526071,
            /* [  94 ]: -2.46= */  0.0069468507886,
            /* [  95 ]: -2.45= */  0.0071428107353,
            /* [  96 ]: -2.44= */  0.0073436309553,
            /* [  97 ]: -2.43= */  0.0075494114163,
            /* [  98 ]: -2.42= */  0.0077602535506,
            /* [  99 ]: -2.41= */  0.0079762602607,
            /* [ 100 ]: -2.40= */  0.0081975359246,
            /* [ 101 ]: -2.39= */  0.0084241863993,
            /* [ 102 ]: -2.38= */  0.0086563190255,
            /* [ 103 ]: -2.37= */  0.0088940426303,
            /* [ 104 ]: -2.36= */  0.0091374675306,
            /* [ 105 ]: -2.35= */  0.0093867055348,
            /* [ 106 ]: -2.34= */  0.0096418699454,
            /* [ 107 ]: -2.33= */  0.0099030755592,
            /* [ 108 ]: -2.32= */  0.0101704386687,
            /* [ 109 ]: -2.31= */  0.0104440770620,
            /* [ 110 ]: -2.30= */  0.0107241100217,
            /* [ 111 ]: -2.29= */  0.0110106583244,
            /* [ 112 ]: -2.28= */  0.0113038442386,
            /* [ 113 ]: -2.27= */  0.0116037915219,
            /* [ 114 ]: -2.26= */  0.0119106254185,
            /* [ 115 ]: -2.25= */  0.0122244726550,
            /* [ 116 ]: -2.24= */  0.0125454614359,
            /* [ 117 ]: -2.23= */  0.0128737214386,
            /* [ 118 ]: -2.22= */  0.0132093838073,
            /* [ 119 ]: -2.21= */  0.0135525811464,
            /* [ 120 ]: -2.20= */  0.0139034475135,
            /* [ 121 ]: -2.19= */  0.0142621184107,
            /* [ 122 ]: -2.18= */  0.0146287307760,
            /* [ 123 ]: -2.17= */  0.0150034229737,
            /* [ 124 ]: -2.16= */  0.0153863347839,
            /* [ 125 ]: -2.15= */  0.0157776073911,
            /* [ 126 ]: -2.14= */  0.0161773833722,
            /* [ 127 ]: -2.13= */  0.0165858066836,
            /* [ 128 ]: -2.12= */  0.0170030226476,
            /* [ 129 ]: -2.11= */  0.0174291779377,
            /* [ 130 ]: -2.10= */  0.0178644205628,
            /* [ 131 ]: -2.09= */  0.0183088998517,
            /* [ 132 ]: -2.08= */  0.0187627664349,
            /* [ 133 ]: -2.07= */  0.0192261722275,
            /* [ 134 ]: -2.06= */  0.0196992704094,
            /* [ 135 ]: -2.05= */  0.0201822154057,
            /* [ 136 ]: -2.04= */  0.0206751628661,
            /* [ 137 ]: -2.03= */  0.0211782696427,
            /* [ 138 ]: -2.02= */  0.0216916937676,
            /* [ 139 ]: -2.01= */  0.0222155944294,
            /* [ 140 ]: -2.00= */  0.0227501319482,
            /* [ 141 ]: -1.99= */  0.0232954677502,
            /* [ 142 ]: -1.98= */  0.0238517643415,
            /* [ 143 ]: -1.97= */  0.0244191852802,
            /* [ 144 ]: -1.96= */  0.0249978951482,
            /* [ 145 ]: -1.95= */  0.0255880595216,
            /* [ 146 ]: -1.94= */  0.0261898449405,
            /* [ 147 ]: -1.93= */  0.0268034188771,
            /* [ 148 ]: -1.92= */  0.0274289497038,
            /* [ 149 ]: -1.91= */  0.0280666066598,
            /* [ 150 ]: -1.90= */  0.0287165598160,
            /* [ 151 ]: -1.89= */  0.0293789800404,
            /* [ 152 ]: -1.88= */  0.0300540389612,
            /* [ 153 ]: -1.87= */  0.0307419089295,
            /* [ 154 ]: -1.86= */  0.0314427629808,
            /* [ 155 ]: -1.85= */  0.0321567747956,
            /* [ 156 ]: -1.84= */  0.0328841186592,
            /* [ 157 ]: -1.83= */  0.0336249694196,
            /* [ 158 ]: -1.82= */  0.0343795024459,
            /* [ 159 ]: -1.81= */  0.0351478935840,
            /* [ 160 ]: -1.80= */  0.0359303191129,
            /* [ 161 ]: -1.79= */  0.0367269556987,
            /* [ 162 ]: -1.78= */  0.0375379803485,
            /* [ 163 ]: -1.77= */  0.0383635703629,
            /* [ 164 ]: -1.76= */  0.0392039032875,
            /* [ 165 ]: -1.75= */  0.0400591568638,
            /* [ 166 ]: -1.74= */  0.0409295089788,
            /* [ 167 ]: -1.73= */  0.0418151376136,
            /* [ 168 ]: -1.72= */  0.0427162207913,
            /* [ 169 ]: -1.71= */  0.0436329365240,
            /* [ 170 ]: -1.70= */  0.0445654627585,
            /* [ 171 ]: -1.69= */  0.0455139773215,
            /* [ 172 ]: -1.68= */  0.0464786578637,
            /* [ 173 ]: -1.67= */  0.0474596818029,
            /* [ 174 ]: -1.66= */  0.0484572262667,
            /* [ 175 ]: -1.65= */  0.0494714680336,
            /* [ 176 ]: -1.64= */  0.0505025834741,
            /* [ 177 ]: -1.63= */  0.0515507484901,
            /* [ 178 ]: -1.62= */  0.0526161384542,
            /* [ 179 ]: -1.61= */  0.0536989281481,
            /* [ 180 ]: -1.60= */  0.0547992916996,
            /* [ 181 ]: -1.59= */  0.0559174025195,
            /* [ 182 ]: -1.58= */  0.0570534332378,
            /* [ 183 ]: -1.57= */  0.0582075556385,
            /* [ 184 ]: -1.56= */  0.0593799405948,
            /* [ 185 ]: -1.55= */  0.0605707580021,
            /* [ 186 ]: -1.54= */  0.0617801767118,
            /* [ 187 ]: -1.53= */  0.0630083644640,
            /* [ 188 ]: -1.52= */  0.0642554878189,
            /* [ 189 ]: -1.51= */  0.0655217120889,
            /* [ 190 ]: -1.50= */  0.0668072012689,
            /* [ 191 ]: -1.49= */  0.0681121179667,
            /* [ 192 ]: -1.48= */  0.0694366233333,
            /* [ 193 ]: -1.47= */  0.0707808769917,
            /* [ 194 ]: -1.46= */  0.0721450369659,
            /* [ 195 ]: -1.45= */  0.0735292596096,
            /* [ 196 ]: -1.44= */  0.0749336995343,
            /* [ 197 ]: -1.43= */  0.0763585095367,
            /* [ 198 ]: -1.42= */  0.0778038405265,
            /* [ 199 ]: -1.41= */  0.0792698414534,
            /* [ 200 ]: -1.40= */  0.0807566592338,
            /* [ 201 ]: -1.39= */  0.0822644386777,
            /* [ 202 ]: -1.38= */  0.0837933224150,
            /* [ 203 ]: -1.37= */  0.0853434508220,
            /* [ 204 ]: -1.36= */  0.0869149619471,
            /* [ 205 ]: -1.35= */  0.0885079914374,
            /* [ 206 ]: -1.34= */  0.0901226724644,
            /* [ 207 ]: -1.33= */  0.0917591356503,
            /* [ 208 ]: -1.32= */  0.0934175089935,
            /* [ 209 ]: -1.31= */  0.0950979177952,
            /* [ 210 ]: -1.30= */  0.0968004845856,
            /* [ 211 ]: -1.29= */  0.0985253290497,
            /* [ 212 ]: -1.28= */  0.1002725679544,
            /* [ 213 ]: -1.27= */  0.1020423150748,
            /* [ 214 ]: -1.26= */  0.1038346811213,
            /* [ 215 ]: -1.25= */  0.1056497736669,
            /* [ 216 ]: -1.24= */  0.1074876970746,
            /* [ 217 ]: -1.23= */  0.1093485524257,
            /* [ 218 ]: -1.22= */  0.1112324374478,
            /* [ 219 ]: -1.21= */  0.1131394464440,
            /* [ 220 ]: -1.20= */  0.1150696702217,
            /* [ 221 ]: -1.19= */  0.1170231960231,
            /* [ 222 ]: -1.18= */  0.1190001074552,
            /* [ 223 ]: -1.17= */  0.1210004844210,
            /* [ 224 ]: -1.16= */  0.1230244030513,
            /* [ 225 ]: -1.15= */  0.1250719356371,
            /* [ 226 ]: -1.14= */  0.1271431505628,
            /* [ 227 ]: -1.13= */  0.1292381122400,
            /* [ 228 ]: -1.12= */  0.1313568810427,
            /* [ 229 ]: -1.11= */  0.1334995132427,
            /* [ 230 ]: -1.10= */  0.1356660609464,
            /* [ 231 ]: -1.09= */  0.1378565720320,
            /* [ 232 ]: -1.08= */  0.1400710900888,
            /* [ 233 ]: -1.07= */  0.1423096543559,
            /* [ 234 ]: -1.06= */  0.1445722996639,
            /* [ 235 ]: -1.05= */  0.1468590563759,
            /* [ 236 ]: -1.04= */  0.1491699503310,
            /* [ 237 ]: -1.03= */  0.1515050027883,
            /* [ 238 ]: -1.02= */  0.1538642303727,
            /* [ 239 ]: -1.01= */  0.1562476450212,
            /* [ 240 ]: -1.00= */  0.1586552539315,
            /* [ 241 ]: -0.99= */  0.1610870595108,
            /* [ 242 ]: -0.98= */  0.1635430593277,
            /* [ 243 ]: -0.97= */  0.1660232460635,
            /* [ 244 ]: -0.96= */  0.1685276074668,
            /* [ 245 ]: -0.95= */  0.1710561263085,
            /* [ 246 ]: -0.94= */  0.1736087803386,
            /* [ 247 ]: -0.93= */  0.1761855422453,
            /* [ 248 ]: -0.92= */  0.1787863796144,
            /* [ 249 ]: -0.91= */  0.1814112548918,
            /* [ 250 ]: -0.90= */  0.1840601253468,
            /* [ 251 ]: -0.89= */  0.1867329430372,
            /* [ 252 ]: -0.88= */  0.1894296547767,
            /* [ 253 ]: -0.87= */  0.1921502021037,
            /* [ 254 ]: -0.86= */  0.1948945212518,
            /* [ 255 ]: -0.85= */  0.1976625431227,
            /* [ 256 ]: -0.84= */  0.2004541932604,
            /* [ 257 ]: -0.83= */  0.2032693918281,
            /* [ 258 ]: -0.82= */  0.2061080535858,
            /* [ 259 ]: -0.81= */  0.2089700878716,
            /* [ 260 ]: -0.80= */  0.2118553985834,
            /* [ 261 ]: -0.79= */  0.2147638841636,
            /* [ 262 ]: -0.78= */  0.2176954375857,
            /* [ 263 ]: -0.77= */  0.2206499463426,
            /* [ 264 ]: -0.76= */  0.2236272924376,
            /* [ 265 ]: -0.75= */  0.2266273523769,
            /* [ 266 ]: -0.74= */  0.2296499971648,
            /* [ 267 ]: -0.73= */  0.2326950923009,
            /* [ 268 ]: -0.72= */  0.2357624977792,
            /* [ 269 ]: -0.71= */  0.2388520680900,
            /* [ 270 ]: -0.70= */  0.2419636522231,
            /* [ 271 ]: -0.69= */  0.2450970936743,
            /* [ 272 ]: -0.68= */  0.2482522304536,
            /* [ 273 ]: -0.67= */  0.2514288950953,
            /* [ 274 ]: -0.66= */  0.2546269146713,
            /* [ 275 ]: -0.65= */  0.2578461108059,
            /* [ 276 ]: -0.64= */  0.2610862996929,
            /* [ 277 ]: -0.63= */  0.2643472921157,
            /* [ 278 ]: -0.62= */  0.2676288934690,
            /* [ 279 ]: -0.61= */  0.2709309037830,
            /* [ 280 ]: -0.60= */  0.2742531177501,
            /* [ 281 ]: -0.59= */  0.2775953247535,
            /* [ 282 ]: -0.58= */  0.2809573088986,
            /* [ 283 ]: -0.57= */  0.2843388490463,
            /* [ 284 ]: -0.56= */  0.2877397188490,
            /* [ 285 ]: -0.55= */  0.2911596867883,
            /* [ 286 ]: -0.54= */  0.2945985162157,
            /* [ 287 ]: -0.53= */  0.2980559653949,
            /* [ 288 ]: -0.52= */  0.3015317875470,
            /* [ 289 ]: -0.51= */  0.3050257308975,
            /* [ 290 ]: -0.50= */  0.3085375387260,
            /* [ 291 ]: -0.49= */  0.3120669494174,
            /* [ 292 ]: -0.48= */  0.3156136965162,
            /* [ 293 ]: -0.47= */  0.3191775087825,
            /* [ 294 ]: -0.46= */  0.3227581102503,
            /* [ 295 ]: -0.45= */  0.3263552202879,
            /* [ 296 ]: -0.44= */  0.3299685536606,
            /* [ 297 ]: -0.43= */  0.3335978205954,
            /* [ 298 ]: -0.42= */  0.3372427268482,
            /* [ 299 ]: -0.41= */  0.3409029737723,
            /* [ 300 ]: -0.40= */  0.3445782583897,
            /* [ 301 ]: -0.39= */  0.3482682734640,
            /* [ 302 ]: -0.38= */  0.3519727075758,
            /* [ 303 ]: -0.37= */  0.3556912451994,
            /* [ 304 ]: -0.36= */  0.3594235667820,
            /* [ 305 ]: -0.35= */  0.3631693488244,
            /* [ 306 ]: -0.34= */  0.3669282639640,
            /* [ 307 ]: -0.33= */  0.3706999810593,
            /* [ 308 ]: -0.32= */  0.3744841652767,
            /* [ 309 ]: -0.31= */  0.3782804781780,
            /* [ 310 ]: -0.30= */  0.3820885778110,
            /* [ 311 ]: -0.29= */  0.3859081188011,
            /* [ 312 ]: -0.28= */  0.3897387524442,
            /* [ 313 ]: -0.27= */  0.3935801268020,
            /* [ 314 ]: -0.26= */  0.3974318867982,
            /* [ 315 ]: -0.25= */  0.4012936743171,
            /* [ 316 ]: -0.24= */  0.4051651283022,
            /* [ 317 ]: -0.23= */  0.4090458848580,
            /* [ 318 ]: -0.22= */  0.4129355773518,
            /* [ 319 ]: -0.21= */  0.4168338365175,
            /* [ 320 ]: -0.20= */  0.4207402905609,
            /* [ 321 ]: -0.19= */  0.4246545652652,
            /* [ 322 ]: -0.18= */  0.4285762840991,
            /* [ 323 ]: -0.17= */  0.4325050683250,
            /* [ 324 ]: -0.16= */  0.4364405371086,
            /* [ 325 ]: -0.15= */  0.4403823076297,
            /* [ 326 ]: -0.14= */  0.4443299951941,
            /* [ 327 ]: -0.13= */  0.4482832133454,
            /* [ 328 ]: -0.12= */  0.4522415739794,
            /* [ 329 ]: -0.11= */  0.4562046874577,
            /* [ 330 ]: -0.10= */  0.4601721627230,
            /* [ 331 ]: -0.09= */  0.4641436074148,
            /* [ 332 ]: -0.08= */  0.4681186279860,
            /* [ 333 ]: -0.07= */  0.4720968298195,
            /* [ 334 ]: -0.06= */  0.4760778173459,
            /* [ 335 ]: -0.05= */  0.4800611941616,
            /* [ 336 ]: -0.04= */  0.4840465631472,
            /* [ 337 ]: -0.03= */  0.4880335265859,
            /* [ 338 ]: -0.02= */  0.4920216862831,
            /* [ 339 ]: -0.01= */  0.4960106436854,
            /* [ 340 ]:  0.00= */  0.5000000000000,
            /* [ 341 ]: +0.01= */  0.5039893563146,
            /* [ 342 ]: +0.02= */  0.5079783137169,
            /* [ 343 ]: +0.03= */  0.5119664734141,
            /* [ 344 ]: +0.04= */  0.5159534368528,
            /* [ 345 ]: +0.05= */  0.5199388058384,
            /* [ 346 ]: +0.06= */  0.5239221826541,
            /* [ 347 ]: +0.07= */  0.5279031701805,
            /* [ 348 ]: +0.08= */  0.5318813720140,
            /* [ 349 ]: +0.09= */  0.5358563925852,
            /* [ 350 ]: +0.10= */  0.5398278372770,
            /* [ 351 ]: +0.11= */  0.5437953125423,
            /* [ 352 ]: +0.12= */  0.5477584260206,
            /* [ 353 ]: +0.13= */  0.5517167866546,
            /* [ 354 ]: +0.14= */  0.5556700048059,
            /* [ 355 ]: +0.15= */  0.5596176923702,
            /* [ 356 ]: +0.16= */  0.5635594628914,
            /* [ 357 ]: +0.17= */  0.5674949316750,
            /* [ 358 ]: +0.18= */  0.5714237159009,
            /* [ 359 ]: +0.19= */  0.5753454347348,
            /* [ 360 ]: +0.20= */  0.5792597094391,
            /* [ 361 ]: +0.21= */  0.5831661634824,
            /* [ 362 ]: +0.22= */  0.5870644226482,
            /* [ 363 ]: +0.23= */  0.5909541151420,
            /* [ 364 ]: +0.24= */  0.5948348716978,
            /* [ 365 ]: +0.25= */  0.5987063256829,
            /* [ 366 ]: +0.26= */  0.6025681132018,
            /* [ 367 ]: +0.27= */  0.6064198731980,
            /* [ 368 ]: +0.28= */  0.6102612475558,
            /* [ 369 ]: +0.29= */  0.6140918811989,
            /* [ 370 ]: +0.30= */  0.6179114221889,
            /* [ 371 ]: +0.31= */  0.6217195218220,
            /* [ 372 ]: +0.32= */  0.6255158347233,
            /* [ 373 ]: +0.33= */  0.6293000189406,
            /* [ 374 ]: +0.34= */  0.6330717360360,
            /* [ 375 ]: +0.35= */  0.6368306511756,
            /* [ 376 ]: +0.36= */  0.6405764332180,
            /* [ 377 ]: +0.37= */  0.6443087548005,
            /* [ 378 ]: +0.38= */  0.6480272924242,
            /* [ 379 ]: +0.39= */  0.6517317265360,
            /* [ 380 ]: +0.40= */  0.6554217416103,
            /* [ 381 ]: +0.41= */  0.6590970262277,
            /* [ 382 ]: +0.42= */  0.6627572731517,
            /* [ 383 ]: +0.43= */  0.6664021794045,
            /* [ 384 ]: +0.44= */  0.6700314463394,
            /* [ 385 ]: +0.45= */  0.6736447797121,
            /* [ 386 ]: +0.46= */  0.6772418897496,
            /* [ 387 ]: +0.47= */  0.6808224912174,
            /* [ 388 ]: +0.48= */  0.6843863034838,
            /* [ 389 ]: +0.49= */  0.6879330505826,
            /* [ 390 ]: +0.50= */  0.6914624612740,
            /* [ 391 ]: +0.51= */  0.6949742691025,
            /* [ 392 ]: +0.52= */  0.6984682124530,
            /* [ 393 ]: +0.53= */  0.7019440346051,
            /* [ 394 ]: +0.54= */  0.7054014837843,
            /* [ 395 ]: +0.55= */  0.7088403132116,
            /* [ 396 ]: +0.56= */  0.7122602811510,
            /* [ 397 ]: +0.57= */  0.7156611509537,
            /* [ 398 ]: +0.58= */  0.7190426911014,
            /* [ 399 ]: +0.59= */  0.7224046752465,
            /* [ 400 ]: +0.60= */  0.7257468822499,
            /* [ 401 ]: +0.61= */  0.7290690962170,
            /* [ 402 ]: +0.62= */  0.7323711065310,
            /* [ 403 ]: +0.63= */  0.7356527078843,
            /* [ 404 ]: +0.64= */  0.7389137003071,
            /* [ 405 ]: +0.65= */  0.7421538891941,
            /* [ 406 ]: +0.66= */  0.7453730853287,
            /* [ 407 ]: +0.67= */  0.7485711049047,
            /* [ 408 ]: +0.68= */  0.7517477695464,
            /* [ 409 ]: +0.69= */  0.7549029063257,
            /* [ 410 ]: +0.70= */  0.7580363477769,
            /* [ 411 ]: +0.71= */  0.7611479319100,
            /* [ 412 ]: +0.72= */  0.7642375022207,
            /* [ 413 ]: +0.73= */  0.7673049076991,
            /* [ 414 ]: +0.74= */  0.7703500028352,
            /* [ 415 ]: +0.75= */  0.7733726476231,
            /* [ 416 ]: +0.76= */  0.7763727075624,
            /* [ 417 ]: +0.77= */  0.7793500536573,
            /* [ 418 ]: +0.78= */  0.7823045624143,
            /* [ 419 ]: +0.79= */  0.7852361158364,
            /* [ 420 ]: +0.80= */  0.7881446014166,
            /* [ 421 ]: +0.81= */  0.7910299121284,
            /* [ 422 ]: +0.82= */  0.7938919464142,
            /* [ 423 ]: +0.83= */  0.7967306081719,
            /* [ 424 ]: +0.84= */  0.7995458067395,
            /* [ 425 ]: +0.85= */  0.8023374568773,
            /* [ 426 ]: +0.86= */  0.8051054787482,
            /* [ 427 ]: +0.87= */  0.8078497978963,
            /* [ 428 ]: +0.88= */  0.8105703452233,
            /* [ 429 ]: +0.89= */  0.8132670569628,
            /* [ 430 ]: +0.90= */  0.8159398746532,
            /* [ 431 ]: +0.91= */  0.8185887451082,
            /* [ 432 ]: +0.92= */  0.8212136203856,
            /* [ 433 ]: +0.93= */  0.8238144577547,
            /* [ 434 ]: +0.94= */  0.8263912196614,
            /* [ 435 ]: +0.95= */  0.8289438736915,
            /* [ 436 ]: +0.96= */  0.8314723925332,
            /* [ 437 ]: +0.97= */  0.8339767539365,
            /* [ 438 ]: +0.98= */  0.8364569406723,
            /* [ 439 ]: +0.99= */  0.8389129404892,
            /* [ 440 ]: +1.00= */  0.8413447460685,
            /* [ 441 ]: +1.01= */  0.8437523549787,
            /* [ 442 ]: +1.02= */  0.8461357696273,
            /* [ 443 ]: +1.03= */  0.8484949972117,
            /* [ 444 ]: +1.04= */  0.8508300496690,
            /* [ 445 ]: +1.05= */  0.8531409436241,
            /* [ 446 ]: +1.06= */  0.8554277003361,
            /* [ 447 ]: +1.07= */  0.8576903456441,
            /* [ 448 ]: +1.08= */  0.8599289099112,
            /* [ 449 ]: +1.09= */  0.8621434279680,
            /* [ 450 ]: +1.10= */  0.8643339390536,
            /* [ 451 ]: +1.11= */  0.8665004867572,
            /* [ 452 ]: +1.12= */  0.8686431189573,
            /* [ 453 ]: +1.13= */  0.8707618877600,
            /* [ 454 ]: +1.14= */  0.8728568494372,
            /* [ 455 ]: +1.15= */  0.8749280643628,
            /* [ 456 ]: +1.16= */  0.8769755969487,
            /* [ 457 ]: +1.17= */  0.8789995155790,
            /* [ 458 ]: +1.18= */  0.8809998925448,
            /* [ 459 ]: +1.19= */  0.8829768039769,
            /* [ 460 ]: +1.20= */  0.8849303297783,
            /* [ 461 ]: +1.21= */  0.8868605535560,
            /* [ 462 ]: +1.22= */  0.8887675625522,
            /* [ 463 ]: +1.23= */  0.8906514475743,
            /* [ 464 ]: +1.24= */  0.8925123029254,
            /* [ 465 ]: +1.25= */  0.8943502263331,
            /* [ 466 ]: +1.26= */  0.8961653188787,
            /* [ 467 ]: +1.27= */  0.8979576849252,
            /* [ 468 ]: +1.28= */  0.8997274320456,
            /* [ 469 ]: +1.29= */  0.9014746709502,
            /* [ 470 ]: +1.30= */  0.9031995154144,
            /* [ 471 ]: +1.31= */  0.9049020822048,
            /* [ 472 ]: +1.32= */  0.9065824910065,
            /* [ 473 ]: +1.33= */  0.9082408643497,
            /* [ 474 ]: +1.34= */  0.9098773275355,
            /* [ 475 ]: +1.35= */  0.9114920085626,
            /* [ 476 ]: +1.36= */  0.9130850380529,
            /* [ 477 ]: +1.37= */  0.9146565491780,
            /* [ 478 ]: +1.38= */  0.9162066775850,
            /* [ 479 ]: +1.39= */  0.9177355613223,
            /* [ 480 ]: +1.40= */  0.9192433407662,
            /* [ 481 ]: +1.41= */  0.9207301585466,
            /* [ 482 ]: +1.42= */  0.9221961594735,
            /* [ 483 ]: +1.43= */  0.9236414904633,
            /* [ 484 ]: +1.44= */  0.9250663004657,
            /* [ 485 ]: +1.45= */  0.9264707403903,
            /* [ 486 ]: +1.46= */  0.9278549630341,
            /* [ 487 ]: +1.47= */  0.9292191230083,
            /* [ 488 ]: +1.48= */  0.9305633766667,
            /* [ 489 ]: +1.49= */  0.9318878820333,
            /* [ 490 ]: +1.50= */  0.9331927987311,
            /* [ 491 ]: +1.51= */  0.9344782879111,
            /* [ 492 ]: +1.52= */  0.9357445121811,
            /* [ 493 ]: +1.53= */  0.9369916355360,
            /* [ 494 ]: +1.54= */  0.9382198232882,
            /* [ 495 ]: +1.55= */  0.9394292419979,
            /* [ 496 ]: +1.56= */  0.9406200594052,
            /* [ 497 ]: +1.57= */  0.9417924443614,
            /* [ 498 ]: +1.58= */  0.9429465667622,
            /* [ 499 ]: +1.59= */  0.9440825974805,
            /* [ 500 ]: +1.60= */  0.9452007083004,
            /* [ 501 ]: +1.61= */  0.9463010718519,
            /* [ 502 ]: +1.62= */  0.9473838615457,
            /* [ 503 ]: +1.63= */  0.9484492515099,
            /* [ 504 ]: +1.64= */  0.9494974165259,
            /* [ 505 ]: +1.65= */  0.9505285319663,
            /* [ 506 ]: +1.66= */  0.9515427737333,
            /* [ 507 ]: +1.67= */  0.9525403181971,
            /* [ 508 ]: +1.68= */  0.9535213421363,
            /* [ 509 ]: +1.69= */  0.9544860226784,
            /* [ 510 ]: +1.70= */  0.9554345372415,
            /* [ 511 ]: +1.71= */  0.9563670634760,
            /* [ 512 ]: +1.72= */  0.9572837792087,
            /* [ 513 ]: +1.73= */  0.9581848623864,
            /* [ 514 ]: +1.74= */  0.9590704910212,
            /* [ 515 ]: +1.75= */  0.9599408431362,
            /* [ 516 ]: +1.76= */  0.9607960967125,
            /* [ 517 ]: +1.77= */  0.9616364296371,
            /* [ 518 ]: +1.78= */  0.9624620196515,
            /* [ 519 ]: +1.79= */  0.9632730443013,
            /* [ 520 ]: +1.80= */  0.9640696808871,
            /* [ 521 ]: +1.81= */  0.9648521064160,
            /* [ 522 ]: +1.82= */  0.9656204975541,
            /* [ 523 ]: +1.83= */  0.9663750305804,
            /* [ 524 ]: +1.84= */  0.9671158813408,
            /* [ 525 ]: +1.85= */  0.9678432252044,
            /* [ 526 ]: +1.86= */  0.9685572370192,
            /* [ 527 ]: +1.87= */  0.9692580910705,
            /* [ 528 ]: +1.88= */  0.9699459610388,
            /* [ 529 ]: +1.89= */  0.9706210199596,
            /* [ 530 ]: +1.90= */  0.9712834401840,
            /* [ 531 ]: +1.91= */  0.9719333933402,
            /* [ 532 ]: +1.92= */  0.9725710502962,
            /* [ 533 ]: +1.93= */  0.9731965811229,
            /* [ 534 ]: +1.94= */  0.9738101550595,
            /* [ 535 ]: +1.95= */  0.9744119404784,
            /* [ 536 ]: +1.96= */  0.9750021048518,
            /* [ 537 ]: +1.97= */  0.9755808147198,
            /* [ 538 ]: +1.98= */  0.9761482356585,
            /* [ 539 ]: +1.99= */  0.9767045322498,
            /* [ 540 ]: +2.00= */  0.9772498680518,
            /* [ 541 ]: +2.01= */  0.9777844055706,
            /* [ 542 ]: +2.02= */  0.9783083062324,
            /* [ 543 ]: +2.03= */  0.9788217303573,
            /* [ 544 ]: +2.04= */  0.9793248371339,
            /* [ 545 ]: +2.05= */  0.9798177845943,
            /* [ 546 ]: +2.06= */  0.9803007295906,
            /* [ 547 ]: +2.07= */  0.9807738277725,
            /* [ 548 ]: +2.08= */  0.9812372335651,
            /* [ 549 ]: +2.09= */  0.9816911001483,
            /* [ 550 ]: +2.10= */  0.9821355794372,
            /* [ 551 ]: +2.11= */  0.9825708220623,
            /* [ 552 ]: +2.12= */  0.9829969773524,
            /* [ 553 ]: +2.13= */  0.9834141933164,
            /* [ 554 ]: +2.14= */  0.9838226166278,
            /* [ 555 ]: +2.15= */  0.9842223926089,
            /* [ 556 ]: +2.16= */  0.9846136652161,
            /* [ 557 ]: +2.17= */  0.9849965770263,
            /* [ 558 ]: +2.18= */  0.9853712692240,
            /* [ 559 ]: +2.19= */  0.9857378815893,
            /* [ 560 ]: +2.20= */  0.9860965524865,
            /* [ 561 ]: +2.21= */  0.9864474188536,
            /* [ 562 ]: +2.22= */  0.9867906161927,
            /* [ 563 ]: +2.23= */  0.9871262785614,
            /* [ 564 ]: +2.24= */  0.9874545385641,
            /* [ 565 ]: +2.25= */  0.9877755273450,
            /* [ 566 ]: +2.26= */  0.9880893745815,
            /* [ 567 ]: +2.27= */  0.9883962084781,
            /* [ 568 ]: +2.28= */  0.9886961557614,
            /* [ 569 ]: +2.29= */  0.9889893416756,
            /* [ 570 ]: +2.30= */  0.9892758899783,
            /* [ 571 ]: +2.31= */  0.9895559229380,
            /* [ 572 ]: +2.32= */  0.9898295613313,
            /* [ 573 ]: +2.33= */  0.9900969244408,
            /* [ 574 ]: +2.34= */  0.9903581300546,
            /* [ 575 ]: +2.35= */  0.9906132944652,
            /* [ 576 ]: +2.36= */  0.9908625324694,
            /* [ 577 ]: +2.37= */  0.9911059573697,
            /* [ 578 ]: +2.38= */  0.9913436809745,
            /* [ 579 ]: +2.39= */  0.9915758136007,
            /* [ 580 ]: +2.40= */  0.9918024640754,
            /* [ 581 ]: +2.41= */  0.9920237397393,
            /* [ 582 ]: +2.42= */  0.9922397464494,
            /* [ 583 ]: +2.43= */  0.9924505885837,
            /* [ 584 ]: +2.44= */  0.9926563690447,
            /* [ 585 ]: +2.45= */  0.9928571892647,
            /* [ 586 ]: +2.46= */  0.9930531492114,
            /* [ 587 ]: +2.47= */  0.9932443473929,
            /* [ 588 ]: +2.48= */  0.9934308808645,
            /* [ 589 ]: +2.49= */  0.9936128452351,
            /* [ 590 ]: +2.50= */  0.9937903346742,
            /* [ 591 ]: +2.51= */  0.9939634419196,
            /* [ 592 ]: +2.52= */  0.9941322582847,
            /* [ 593 ]: +2.53= */  0.9942968736670,
            /* [ 594 ]: +2.54= */  0.9944573765569,
            /* [ 595 ]: +2.55= */  0.9946138540459,
            /* [ 596 ]: +2.56= */  0.9947663918364,
            /* [ 597 ]: +2.57= */  0.9949150742510,
            /* [ 598 ]: +2.58= */  0.9950599842422,
            /* [ 599 ]: +2.59= */  0.9952012034029,
            /* [ 600 ]: +2.60= */  0.9953388119763,
            /* [ 601 ]: +2.61= */  0.9954728888670,
            /* [ 602 ]: +2.62= */  0.9956035116519,
            /* [ 603 ]: +2.63= */  0.9957307565909,
            /* [ 604 ]: +2.64= */  0.9958546986390,
            /* [ 605 ]: +2.65= */  0.9959754114572,
            /* [ 606 ]: +2.66= */  0.9960929674251,
            /* [ 607 ]: +2.67= */  0.9962074376523,
            /* [ 608 ]: +2.68= */  0.9963188919908,
            /* [ 609 ]: +2.69= */  0.9964273990476,
            /* [ 610 ]: +2.70= */  0.9965330261970,
            /* [ 611 ]: +2.71= */  0.9966358395933,
            /* [ 612 ]: +2.72= */  0.9967359041841,
            /* [ 613 ]: +2.73= */  0.9968332837226,
            /* [ 614 ]: +2.74= */  0.9969280407813,
            /* [ 615 ]: +2.75= */  0.9970202367649,
            /* [ 616 ]: +2.76= */  0.9971099319238,
            /* [ 617 ]: +2.77= */  0.9971971853672,
            /* [ 618 ]: +2.78= */  0.9972820550773,
            /* [ 619 ]: +2.79= */  0.9973645979221,
            /* [ 620 ]: +2.80= */  0.9974448696696,
            /* [ 621 ]: +2.81= */  0.9975229250012,
            /* [ 622 ]: +2.82= */  0.9975988175258,
            /* [ 623 ]: +2.83= */  0.9976725997933,
            /* [ 624 ]: +2.84= */  0.9977443233085,
            /* [ 625 ]: +2.85= */  0.9978140385451,
            /* [ 626 ]: +2.86= */  0.9978817949596,
            /* [ 627 ]: +2.87= */  0.9979476410051,
            /* [ 628 ]: +2.88= */  0.9980116241451,
            /* [ 629 ]: +2.89= */  0.9980737908678,
            /* [ 630 ]: +2.90= */  0.9981341866996,
            /* [ 631 ]: +2.91= */  0.9981928562192,
            /* [ 632 ]: +2.92= */  0.9982498430713,
            /* [ 633 ]: +2.93= */  0.9983051899807,
            /* [ 634 ]: +2.94= */  0.9983589387658,
            /* [ 635 ]: +2.95= */  0.9984111303526,
            /* [ 636 ]: +2.96= */  0.9984618047883,
            /* [ 637 ]: +2.97= */  0.9985110012548,
            /* [ 638 ]: +2.98= */  0.9985587580827,
            /* [ 639 ]: +2.99= */  0.9986051127645,
            /* [ 640 ]: +3.00= */  0.9986501019684,
            /* [ 641 ]: +3.01= */  0.9986937615512,
            /* [ 642 ]: +3.02= */  0.9987361265723,
            /* [ 643 ]: +3.03= */  0.9987772313064,
            /* [ 644 ]: +3.04= */  0.9988171092569,
            /* [ 645 ]: +3.05= */  0.9988557931690,
            /* [ 646 ]: +3.06= */  0.9988933150426,
            /* [ 647 ]: +3.07= */  0.9989297061453,
            /* [ 648 ]: +3.08= */  0.9989649970252,
            /* [ 649 ]: +3.09= */  0.9989992175234,
            /* [ 650 ]: +3.10= */  0.9990323967868,
            /* [ 651 ]: +3.11= */  0.9990645632805,
            /* [ 652 ]: +3.12= */  0.9990957448002,
            /* [ 653 ]: +3.13= */  0.9991259684844,
            /* [ 654 ]: +3.14= */  0.9991552608265,
            /* [ 655 ]: +3.15= */  0.9991836476872,
            /* [ 656 ]: +3.16= */  0.9992111543056,
            /* [ 657 ]: +3.17= */  0.9992378053119,
            /* [ 658 ]: +3.18= */  0.9992636247384,
            /* [ 659 ]: +3.19= */  0.9992886360314,
            /* [ 660 ]: +3.20= */  0.9993128620621,
            /* [ 661 ]: +3.21= */  0.9993363251386,
            /* [ 662 ]: +3.22= */  0.9993590470163,
            /* [ 663 ]: +3.23= */  0.9993810489096,
            /* [ 664 ]: +3.24= */  0.9994023515021,
            /* [ 665 ]: +3.25= */  0.9994229749576,
            /* [ 666 ]: +3.26= */  0.9994429389310,
            /* [ 667 ]: +3.27= */  0.9994622625782,
            /* [ 668 ]: +3.28= */  0.9994809645668,
            /* [ 669 ]: +3.29= */  0.9994990630862,
            /* [ 670 ]: +3.30= */  0.9995165758576,
            /* [ 671 ]: +3.31= */  0.9995335201439,
            /* [ 672 ]: +3.32= */  0.9995499127594,
            /* [ 673 ]: +3.33= */  0.9995657700796,
            /* [ 674 ]: +3.34= */  0.9995811080506,
            /* [ 675 ]: +3.35= */  0.9995959421981,
            /* [ 676 ]: +3.36= */  0.9996102876374,
            /* [ 677 ]: +3.37= */  0.9996241590816,
            /* [ 678 ]: +3.38= */  0.9996375708510,
            /* [ 679 ]: +3.39= */  0.9996505368817,
            /* [ 680 ]: +3.40= */  0.9996630707343
    };
    // @formatter:on

}