/*******************************************************************************
 * Copyright (c) 2019, 2020 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.security.acme.internal;

import static com.ibm.ws.security.acme.internal.util.AcmeConstants.DEFAULT_ALIAS;
import static com.ibm.ws.security.acme.internal.util.AcmeConstants.DEFAULT_KEY_STORE;
import static com.ibm.ws.security.acme.internal.util.AcmeConstants.KEY_KEYSTORE_SERVICE;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;

import org.bouncycastle.asn1.x509.GeneralName;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.ConfigurationPolicy;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;

import com.ibm.websphere.ras.Tr;
import com.ibm.websphere.ras.TraceComponent;
import com.ibm.websphere.ras.annotation.Sensitive;
import com.ibm.websphere.ras.annotation.Trivial;
import com.ibm.websphere.ssl.SSLConfig;
import com.ibm.ws.container.service.state.ApplicationStateListener;
import com.ibm.ws.ffdc.annotation.FFDCIgnore;
import com.ibm.ws.security.acme.AcmeCaException;
import com.ibm.ws.security.acme.AcmeCertificate;
import com.ibm.ws.security.acme.AcmeProvider;
import com.ibm.ws.ssl.JSSEProviderFactory;
import com.ibm.ws.ssl.KeyStoreService;

/**
 * ACME 2.0 support component service.
 */
@Component(immediate = true, configurationPolicy = ConfigurationPolicy.IGNORE, property = { "service.vendor=IBM" })
public class AcmeProviderImpl implements AcmeProvider {

	private static final TraceComponent tc = Tr.register(AcmeProviderImpl.class);

	/** KeyStoreService to retrieve configured KeyStores from. */
	private final static AtomicReference<KeyStoreService> keyStoreServiceRef = new AtomicReference<KeyStoreService>();

	/**
	 * An {@link ApplicationStateListener} used to validate whether the ACME
	 * authorization web application has started.
	 */
	private final static AtomicReference<AcmeApplicationStateListener> applicationStateListenerRef = new AtomicReference<AcmeApplicationStateListener>();

	/** Client used to communicate with the ACME CA server. */
	private static AcmeClient acmeClient;

	/** Configuration for the ACME client. */
	private static AcmeConfig acmeConfig;

	@Override
	public void refreshCertificate() throws AcmeCaException {
		checkAndInstallCertificate(true, null, null, null);
	}

	/**
	 * Check the certificate, and install a new certificate generated by the
	 * ACME CA if the certificate needs to be replaced.
	 * 
	 * <p/>
	 * If <code>keyStore</code> is non-null, then both <code>keyStoreFile</code>
	 * and <code>password</code> should also be non-null. When these are
	 * non-null the method will use the certificate installed under the
	 * "default" alias in the input {@link KeyStore} as the currently installed
	 * certificate.
	 * 
	 * <p/>
	 * If the <code>keyStore</code> is null, the method will look up the
	 * certificate from the SSL configuration.
	 * 
	 * @param forceRefresh
	 *            Force refreshing of the certificate. Skip any checks used to
	 *            determine whether the certificate should be replaced.
	 * @param keyStore
	 *            {@link KeyStore} that contains the certificate under the
	 *            "default" alias.
	 * @param keyStoreFile
	 *            {@link KeyStore} file to update.
	 * @param password
	 *            The password for the {@link KeyStore}.
	 * @throws AcmeCaException
	 *             If there was an issue checking or updating the certificate.
	 */
	@FFDCIgnore({ AcmeCaException.class })
	private void checkAndInstallCertificate(boolean forceRefresh, KeyStore keyStore, File keyStoreFile,
			@Sensitive String password) throws AcmeCaException {
		/*
		 * Wait until the ACME authorization web application is available. At
		 * this point, it always should be, but check just in case.
		 */
		applicationStateListenerRef.get().waitUntilWebAppAvailable();

		/*
		 * Keep a reference to the existing certificate chain that we will
		 * replace so we can revoke it.
		 */
		List<X509Certificate> existingCertChain = null;
		if (keyStore == null) {
			existingCertChain = getConfiguredDefaultCertificateChain();
		} else {
			try {
				existingCertChain = convertToX509CertChain(keyStore.getCertificateChain(DEFAULT_ALIAS));
			} catch (KeyStoreException e) {
				throw new AcmeCaException(
						Tr.formatMessage(tc, "CWPKI2029E", keyStoreFile, DEFAULT_ALIAS, e.getMessage()), e);
			}
		}

		/*
		 * Check whether we need a new certificate.
		 */
		AcmeCertificate acmeCertificate = checkAndRetrieveCertificate(existingCertChain, forceRefresh);

		if (acmeCertificate != null) {
			/*
			 * Convert the certificate chain to an array from a list.
			 */
			Certificate[] chainArr = convertChainToArray(acmeCertificate.getCertificateChain());

			/*
			 * Store the certificate chain for the default alias in the default
			 * keystore.
			 */
			try {
				if (keyStore == null) {
					keyStoreServiceRef.get().setKeyEntryToKeyStore(DEFAULT_KEY_STORE, DEFAULT_ALIAS,
							acmeCertificate.getKeyPair().getPrivate(), chainArr);
				} else {
					keyStore.setKeyEntry(DEFAULT_ALIAS, acmeCertificate.getKeyPair().getPrivate(),
							password.toCharArray(), chainArr);
					keyStore.store(new FileOutputStream(keyStoreFile), password.toCharArray());
				}
			} catch (CertificateException | KeyStoreException | NoSuchAlgorithmException | IOException ex) {
				throw new AcmeCaException(
						Tr.formatMessage(tc, "CWPKI2030E", DEFAULT_ALIAS, DEFAULT_KEY_STORE, ex.getMessage()), ex);
			}

			/*
			 * Revoke the old certificate, which has now been replaced in the
			 * keystore.
			 */
			if (existingCertChain != null) {
				try {
					revoke(existingCertChain);
				} catch (AcmeCaException e) {
					if (TraceComponent.isAnyTracingEnabled() && tc.isWarningEnabled()) {
						Tr.debug(tc, "Failed to revoke the certificate.", existingCertChain);
					}
				}
			}

			/*
			 * Finally, log a message indicate the new certificate has been
			 * installed.
			 */
			Tr.audit(tc, "CWPKI2007I", acmeCertificate.getCertificate().getSerialNumber().toString(16),
					acmeConfig.getDirectoryURI(),
					acmeCertificate.getCertificate().getNotAfter().toInstant().toString());
		} else {
			if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
				Tr.debug(tc, "Previous certificate requested from ACME CA server is still valid.");
			}
		}
	}

	@Override
	public String getHttp01Authorization(String token) throws AcmeCaException {
		return getAcmeClient().getHttp01Authorization(token);
	}

	/**
	 * Revoke a certificate using an existing account on the ACME server. If the
	 * account key pair cannot be found, we will fail.
	 * 
	 * @param certificateChain
	 *            The certificate chain with the leaf certificate to revoke.
	 * @throws AcmeCaException
	 *             If there was an error revoking the certificate.
	 */
	@FFDCIgnore({ AcmeCaException.class })
	public void revoke(List<X509Certificate> certificateChain) throws AcmeCaException {
		X509Certificate certificate = getLeafCertificate(certificateChain);

		try {
			getAcmeClient().revoke(certificate);
		} catch (AcmeCaException e) {
			if (TraceComponent.isAnyTracingEnabled() && tc.isWarningEnabled()) {
				Tr.warning(tc, "CWPKI2038W", certificate.getSerialNumber().toString(16), e.getMessage());
			}
			throw e;
		}
	}

	/**
	 * Convenience method that will retrieve the {@link AcmeClient} instance or
	 * throw an {@link AcmeCaException} if the {@link AcmeClient} is null.
	 * 
	 * @return The {@link AcmeClient} instance to use.
	 * @throws AcmeCaException
	 *             If the {@link AcmeClient} instance is null.
	 */
	@Trivial
	private AcmeClient getAcmeClient() throws AcmeCaException {
		if (acmeClient == null) {
			throw new AcmeCaException("Internal error. ACME client was not initialized.");
		}
		return acmeClient;
	}

	/**
	 * Set the {@link KeyStoreService} instance.
	 * 
	 * @param keyStoreService
	 *            {@link KeyStoreService} instance.
	 */
	@Reference(name = KEY_KEYSTORE_SERVICE, service = KeyStoreService.class, cardinality = ReferenceCardinality.MANDATORY)
	protected void setKeyStoreService(KeyStoreService keyStoreService) {
		keyStoreServiceRef.set(keyStoreService);
	}

	protected void unsetKeyStoreService(KeyStoreService keyStoreService) {
		keyStoreServiceRef.compareAndSet(keyStoreService, null);
	}

	/**
	 * Check the existing certificate and determine whether a new certificate is
	 * required.
	 * 
	 * @param existingCertChain
	 *            the existing certificate chain.
	 * @return true if a new certificate should be requested, false if the
	 *         existing certificate is still valid.
	 * @throws AcmeCaException
	 *             If there was an issue checking the existing certificate.
	 */
	private boolean isCertificateRequired(List<X509Certificate> existingCertChain) throws AcmeCaException {
		/**
		 * Check to see if we need to fetch a new certificate. Reasons may
		 * include:
		 * 
		 * <pre>
		 * 1. Certificate has not been fetched (does not exist)
		 * 2. Certificate is expired or about to expire.
		 * 3. Certificate has been revoked.
		 * 4. Certificate exists, but is for the wrong domain, or a new domain has been added.
		 * 5. TODO More?
		 * </pre>
		 */
		return existingCertChain == null || existingCertChain.isEmpty() || isExpired(existingCertChain)
				|| isRevoked(existingCertChain) || hasWrongDomains(existingCertChain)
				|| hasWrongSubjectRDNs(existingCertChain);
	}

	/**
	 * Check if a new certificate is required and retrieve it if so.
	 * 
	 * @param existingCertChain
	 *            the existing certificate chain.
	 * @param forceRefresh
	 *            Force a refresh of the certificate.
	 * @return The {@link AcmeCertificate} containing the new certificate.
	 * @throws AcmeCaException
	 *             If there was an issue checking or retrieving the certificate.
	 */
	private AcmeCertificate checkAndRetrieveCertificate(List<X509Certificate> existingCertChain, boolean forceRefresh)
			throws AcmeCaException {
		/*
		 * Check if we need to get a new certificate.
		 */
		if (forceRefresh || isCertificateRequired(existingCertChain)) {
			return fetchCertificate();
		}

		return null;
	}

	/**
	 * Convert an array of {@link Certificate} to a {@link List} of
	 * {@link X509Certificate}.
	 * 
	 * @param certChain
	 *            The certificate chain array to convert.
	 * @return The {@link List} of {@link X509Certificate}s.
	 * @throws AcmeCaException
	 *             If any of the certificates were not an instance of
	 *             {@link X509Certificate}.
	 */
	private List<X509Certificate> convertToX509CertChain(Certificate[] certChain) throws AcmeCaException {
		List<X509Certificate> x509Chain = new ArrayList<X509Certificate>();
		if (certChain != null) {
			for (Certificate cert : certChain) {
				if (cert instanceof X509Certificate) {
					x509Chain.add((X509Certificate) cert);
				} else {
					throw new AcmeCaException(Tr.formatMessage(tc, "CWPKI2044E", cert.getType()));
				}
			}
		}
		return x509Chain;
	}

	/**
	 * Determine if the leaf certificate has domains that no longer match the
	 * domains configured for the ACME feature. We will check that the
	 * certificate subjects common name (CN) and that the subject alternative
	 * DNSNames match.
	 * 
	 * @param certificateChain
	 *            The certificate chain to check.
	 * @return True if the leaf certificate's domains do not match those that
	 *         are configured, false otherwise.
	 * @throws AcmeCaException
	 *             If there was an issue checking the leaf certificate's
	 *             domains.
	 */
	private boolean hasWrongDomains(List<X509Certificate> certificateChain) throws AcmeCaException {
		String methodName = "hasWrongDomains(List<X509Certificate>)";
		boolean hasWrongDomains = false;

		X509Certificate certificate = getLeafCertificate(certificateChain);
		if (certificate == null) {
			return false;
		}

		/*
		 * The common name better match one of the domains.
		 */
		try {
			LdapName dn = new LdapName(certificate.getSubjectX500Principal().getName());

			boolean cnMatches = false;
			for (Rdn rdn : dn.getRdns()) {
				if ("cn".equalsIgnoreCase(rdn.getType())) {
					for (String domain : acmeConfig.getDomains()) {
						if (domain.equalsIgnoreCase((String) rdn.getValue())) {
							cnMatches = true;
							break;
						}
					}
					break;
				}
			}

			if (!cnMatches) {
				if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
					Tr.debug(tc, methodName,
							"The certificate subject's common name does not match any of the domains.");
				}
				hasWrongDomains = true;
			}
		} catch (InvalidNameException e) {
			throw new AcmeCaException(
					Tr.formatMessage(tc, "CWPKI2031E", certificate.getSubjectX500Principal().getName(),
							certificate.getSerialNumber().toString(16), e.getMessage()),
					e);
		}

		/*
		 * Check the subject alternative names for all of our domains. We are OK
		 * if it has more, but we will need to request a new certificate if it
		 * doesn't contain all of the domains.
		 */
		if (!hasWrongDomains) {
			try {
				Collection<List<?>> altNames = certificate.getSubjectAlternativeNames();
				Set<String> dnsNames = new HashSet<String>();
				if (altNames != null) {
					for (List<?> altName : altNames) {
						if (altName.size() < 2) {
							continue;
						}
						switch ((Integer) altName.get(0)) {
						case GeneralName.dNSName:
							Object data = altName.get(1);
							if (data instanceof String) {
								dnsNames.add((String) data);
							}
							break;
						default:
						}
					}
				}

				/*
				 * Check the configured domains against those retrieved from the
				 * certificate.
				 */
				if (!dnsNames.containsAll(acmeConfig.getDomains())) {
					if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
						Tr.debug(tc, methodName,
								"The certificate subject alternative names do not contain all of the configured domains.");
					}
					hasWrongDomains = true;
				}

			} catch (CertificateParsingException e) {
				throw new AcmeCaException(
						Tr.formatMessage(tc, "CWPKI2032E", certificate.getSerialNumber().toString(16), e.getMessage()),
						e);
			}
		}
		return hasWrongDomains;
	}

	/**
	 * Check if the leaf certificate's subject RDNs match the configured
	 * subjectDN.
	 * 
	 * <p/>
	 * Note that this isn't the best check. It is possible that the ACME CA
	 * server may not even honor the RDN's (beside the required CN).
	 * 
	 * @param certificateChain
	 *            The certificate chain to check.
	 * @return whether the leaf certificate's subject RDNs match the configured
	 *         subjectDN.
	 * @throws AcmeCaException
	 */
	private boolean hasWrongSubjectRDNs(List<X509Certificate> certificateChain) throws AcmeCaException {
		String methodName = "hasWrongSubjectRDNs(List<X509Certificate>)";
		boolean hasWrongSubjectRDNs = false;

		X509Certificate certificate = getLeafCertificate(certificateChain);
		if (certificate == null) {
			return false;
		}

		List<Rdn> configuredRdns = acmeConfig.getSubjectDN();
		List<Rdn> certRdns;
		try {
			certRdns = new LdapName(certificate.getSubjectX500Principal().getName()).getRdns();

		} catch (InvalidNameException e) {
			throw new AcmeCaException(
					Tr.formatMessage(tc, "CWPKI2031E", certificate.getSubjectX500Principal().getName(),
							certificate.getSerialNumber().toString(16), e.getMessage()),
					e);
		}

		/*
		 * Determine if the RDNs match between the configuration and the
		 * certificate.
		 */
		boolean rdnsMatch = true;
		if (certRdns.size() == 1) {
			/*
			 * If the certificate only has a single RDN, assume that the CA
			 * doesn't honor other RDNs besides CN.
			 */
			rdnsMatch = certRdns.get(0).equals(configuredRdns.get(0));
		} else if (certRdns.size() == configuredRdns.size()) {
			/*
			 * More than 1 RDN for both the configured subjedDN and the cert?
			 * Make sure they all match.
			 */
			for (int idx = 0; idx < certRdns.size(); idx++) {
				if (!certRdns.get(idx).equals(configuredRdns.get(idx))) {
					rdnsMatch = false;
					break;
				}
			}
		} else {
			rdnsMatch = false;
		}

		if (!rdnsMatch) {
			if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
				Tr.debug(tc, methodName, "The certificate subject's RDNs do not match the configuration.");
			}
			hasWrongSubjectRDNs = true;
		}

		return hasWrongSubjectRDNs;
	}

	/**
	 * Is the existing leaf certificate expired or nearly expired?
	 * 
	 * @param certificateChain
	 *            The certificate chain to check.
	 * @return true if the leaf certificate is expired or nearly expiring.
	 */
	private boolean isExpired(List<X509Certificate> certificateChain) {
		X509Certificate certificate = getLeafCertificate(certificateChain);
		if (certificateChain == null) {
			return false;
		}

		/*
		 * Certificates not after date.
		 */
		Date notAfter = certificate.getNotAfter();

		/*
		 * Get current date.
		 */
		Calendar cal = Calendar.getInstance();
		Date now = cal.getTime();

		/*
		 * Get a date where we want to refresh the certificate.
		 */
		cal.setTime(notAfter);
		cal.add(Calendar.DAY_OF_MONTH, -7); // TODO Hard-coded
		Date refreshDate = cal.getTime();

		/*
		 * Consider the certificate expired if the refresh date has elapsed.
		 */
		return now.compareTo(refreshDate) >= 0;
	}

	/**
	 * Has the certificate been revoked?
	 * 
	 * @param certificateChain
	 *            The certificate chain to check.
	 * @return True if the certificate has been revoked, false otherwise.
	 */
	private boolean isRevoked(List<X509Certificate> certificateChain) {

		// try {
		// CertificateFactory certificateFactory =
		// CertificateFactory.getInstance("X.509");
		// CertPath certPath =
		// certificateFactory.generateCertPath(certificateChain);
		// CertPathValidator validator = CertPathValidator.getInstance("PKIX");
		// PKIXParameters params = new PKIXParameters(keystore);
		// params.setRevocationEnabled(true);
		//
		// if (!Boolean.valueOf(Security.getProperty("ocsp.enabled"))) {
		// Tr.warning(tc,
		// "ABCDEFGH: OCSP certificate revocation checking is not enabled.
		// Certificate revocate checking will be limited to CRLs only.");
		// }
		//
		// PKIXCertPathValidatorResult r = (PKIXCertPathValidatorResult)
		// validator.validate(certPath, params);
		// return true;
		//
		// } catch (NoSuchAlgorithmException e) {
		//
		// }

		return false;
	}

	/**
	 * Fetch the certificate from the ACME CA server using the current
	 * configuration.
	 * 
	 * @return The {@link AcmeCertificate}, which contains the certificate chain
	 *         as well as the public and private keys used to sign the CSR
	 *         request.
	 * @throws AcmeCaException
	 */
	private AcmeCertificate fetchCertificate() throws AcmeCaException {
		/*
		 * Request a new certificate.
		 */
		return getAcmeClient().fetchCertificate(false);
	}

	/**
	 * Convert a certificate chain that is in the form of a {@link List} into an
	 * array of {@link Certificate}s.
	 * 
	 * @param chainList
	 *            The {@link List} of certificates.
	 * @return An array of the same certificates.
	 */
	@Trivial
	private static Certificate[] convertChainToArray(List<X509Certificate> chainList) {
		/*
		 * Convert the certificate chain to an array from a list.
		 */
		Certificate[] chainArray = new X509Certificate[chainList.size()];
		int idx = 0;
		for (Certificate x509cert : chainList) {
			chainArray[idx++] = x509cert;
		}
		return chainArray;
	}

	/**
	 * Get the leaf certificate from the certificate chain.
	 * 
	 * @param certificateChain
	 *            The certificate chain.
	 * @return The leaf certificate.
	 */
	private static X509Certificate getLeafCertificate(List<X509Certificate> certificateChain) {
		if (certificateChain != null && !certificateChain.isEmpty()) {
			return certificateChain.get(0);
		}
		return null;
	}

	/**
	 * Get the current certificate for the default alias from the default
	 * keystore.
	 * 
	 * @return The {@link X509Certificate} chain that is stored under the
	 *         default alias in the default keystore or null if it does not
	 *         exist.
	 * @throws AcmeCaException
	 *             if there was an error getting the configured default cert
	 *             chain
	 */
	@FFDCIgnore({ CertificateException.class })
	private List<X509Certificate> getConfiguredDefaultCertificateChain() throws AcmeCaException {
		/*
		 * Get our existing certificate.
		 */
		try {
			Certificate[] certChain = keyStoreServiceRef.get().getCertificateChainFromKeyStore(DEFAULT_KEY_STORE,
					DEFAULT_ALIAS);
			return convertToX509CertChain(certChain);
		} catch (KeyStoreException | CertificateException e) {
			return null;
		}
	}

	@Override
	public File createDefaultSSLCertificate(String filePath, @Sensitive String password, String keyStoreType,
			String keyStoreProvider) throws CertificateException {
		/*
		 * If we make it in here, Liberty is asking us to generate the default
		 * certificate. We need to not only generate the certificate but also
		 * the keystore itself.
		 * 
		 * First wait until the ACME authorization web application is available.
		 */
		try {
			applicationStateListenerRef.get().waitUntilWebAppAvailable();
		} catch (AcmeCaException e) {
			throw new CertificateException(e.getMessage(), e);
		}

		try {

			/*
			 * Get a new certificate.
			 */
			AcmeCertificate acmeCertificate = fetchCertificate();

			/*
			 * Create a new keystore instance.
			 */
			File file = createKeyStore(filePath, acmeCertificate, password, keyStoreType, keyStoreProvider);

			/*
			 * Finally, log a message indicate the new certificate has been
			 * installed and return the file.
			 */
			Tr.audit(tc, "CWPKI2007I", acmeCertificate.getCertificate().getSerialNumber().toString(16),
					acmeConfig.getDirectoryURI(),
					acmeCertificate.getCertificate().getNotAfter().toInstant().toString());
			return file;
		} catch (AcmeCaException ace) {
			createKeyStore(filePath, null, password, keyStoreType, keyStoreProvider);

			throw new CertificateException(ace.getMessage(), ace);
		}
	}

	/**
	 * Create the keystore instance and return a file that points to the
	 * keystore.
	 * 
	 * @param filePath
	 *            The path to the keystore to create.
	 * @param acmeCertificate
	 *            The {@link AcmeCertificate} instance to insert into the
	 *            keystore. If null, the keystore will be empty.
	 * @param password
	 *            The passsword for the keystore.
	 * @param type
	 *            The keystore type.
	 * @param provider
	 *            The keystore provider.
	 * @return The keystore file.
	 * @throws CertificateException
	 *             If there was an issue creating the keystore.
	 */
	private File createKeyStore(String filePath, AcmeCertificate acmeCertificate, @Sensitive String password,
			String type, String provider) throws CertificateException {
		/*
		 * Create a new keystore instance.
		 */
		KeyStore keyStore;
		try {
			keyStore = JSSEProviderFactory.getInstance().getKeyStoreInstance(type, provider);
			keyStore.load(null, password.toCharArray());
			if (acmeCertificate != null) {
				keyStore.setKeyEntry(DEFAULT_ALIAS, acmeCertificate.getKeyPair().getPrivate(), password.toCharArray(),
						convertChainToArray(acmeCertificate.getCertificateChain()));
			}
		} catch (KeyStoreException | NoSuchAlgorithmException | IOException | NoSuchProviderException ee) {
			throw new CertificateException(Tr.formatMessage(tc, "CWPKI2034E", ee.getMessage()), ee);
		}

		/*
		 * Write the store to a file.
		 */
		File file = new File(filePath);
		try {
			if (file.getParentFile() != null && !file.getParentFile().exists()) {
				file.getParentFile().mkdirs();
			}
			FileOutputStream fos = new FileOutputStream(file);
			keyStore.store(fos, password.toCharArray());

		} catch (KeyStoreException | NoSuchAlgorithmException | IOException e) {
			throw new CertificateException(Tr.formatMessage(tc, "CWPKI2035E", file.getName(), e.getMessage()), e);
		}
		return file;
	}

	/*
	 * This method will be called during keystore initialization when the
	 * default keystore exists.
	 */
	@Override
	public void updateDefaultSSLCertificate(KeyStore keyStore, File keyStoreFile, @Sensitive String password)
			throws CertificateException {
		try {
			checkAndInstallCertificate(false, keyStore, keyStoreFile, password);
		} catch (AcmeCaException e) {
			throw new CertificateException(e.getMessage(), e);
		}
	}

	/**
	 * Get the {@link SSLConfig} object that contains the user-specified SSL
	 * configuration.
	 * 
	 * @return The {@link SSLConfig}.
	 */
	public static SSLConfig getSSLConfig() {
		return acmeConfig.getSSLConfig();
	}

	/**
	 * This method will receive the initial configuration from the
	 * {@link AcmeConfigService} and will behave much like the activate method
	 * would on a regular OSGi component.
	 * 
	 * @param acmeConfigService
	 *            The {@link AcmeConfigService} instance.
	 * @param properties
	 *            The initial properties.
	 */
	@Reference(cardinality = ReferenceCardinality.MANDATORY, updated = "updateAcmeConfigService")
	public void setAcmeConfigService(AcmeConfigService acmeConfigService, Map<String, Object> properties) {
		try {
			acmeConfig = new AcmeConfig(properties);
			acmeClient = new AcmeClient(acmeConfig);
		} catch (AcmeCaException e) {
			Tr.error(tc, e.getMessage()); // AcmeCaExceptions are localized.
		}
	}

	/**
	 * Unset the {@link AcmeConfigService} instance.
	 * 
	 * @param acmeConfigService
	 *            the {@link AcmeConfigService} instance to unet.
	 */
	protected void unsetAcmeConfigService(AcmeConfigService acmeConfigService) {
		acmeConfig = null;
		acmeClient = null;
	}

	/**
	 * This method will receive updated configuration from the
	 * {@link AcmeConfigService} and will behave much like the modified method
	 * would on a regular OSGi component.
	 * 
	 * @param acmeConfigService
	 *            The {@link AcmeConfigService} instance.
	 * @param properties
	 *            The updated properties.
	 */
	protected void updateAcmeConfigService(AcmeConfigService acmeConfigService, Map<String, Object> properties) {
		try {
			acmeConfig = new AcmeConfig(properties);
			acmeClient = new AcmeClient(acmeConfig);

			/*
			 * TODO We need to determine which configuration changes will result
			 * in requiring a certificate to be refreshed. Some that might
			 * trigger a refresh: validFor, directoryURI, country, locality,
			 * state, organization, organizationUnit
			 *
			 * We can't necessarily just check the certificate, b/c they don't
			 * always honor them.
			 */
			checkAndInstallCertificate(false, null, null, null);
		} catch (AcmeCaException e) {
			Tr.error(tc, e.getMessage()); // AcmeCaExceptions are localized.
		}
	}

	/**
	 * Set the {@link AcmeApplicationStateListener} reference.
	 * 
	 * @param acmeApplicationStateListener
	 *            the {@link AcmeApplicationStateListener} instance.
	 */
	@Reference(cardinality = ReferenceCardinality.MANDATORY)
	public void setAcmeApplicationStateListener(AcmeApplicationStateListener acmeApplicationStateListener) {
		applicationStateListenerRef.set(acmeApplicationStateListener);
	}
}
