/*******************************************************************************
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.sib.processor.runtime;

import com.ibm.ws.sib.processor.exceptions.SIMPControllableNotFoundException;
import com.ibm.ws.sib.processor.exceptions.SIMPRuntimeOperationFailedException;

/**
 * Interface to view and manipulate a flow of incoming messages that have been
 * sent to us in response to a request (remote get).
 * 
 * This interface used to extend SIMPDeliveryStreamSet 
 */
public interface SIMPRemoteConsumerReceiverControllable extends SIMPDeliveryReceiverControllable
{

  /**
   * Provides an iterator over all of the remote message requests that this ME
   * has sent 
   *
   * @return Iterator  An iterator over the request messages in the stream. The iterator contains a set of 
   *         requests for remote messages in ascending tick order starting with the oldest.
   *         The iterator gives up SIMPRemoteMessageRequestControllable objects
   */
  SIMPIterator getRemoteMessageRequestIterator();
  
  /**
   * If the target has been restored from a backup and does not want to risk 
   *  reprocessing a retransmitted message. There is a possibility that the request 
   *  will not reach the source quickly so we would have to discard messages on the 
   *  old stream making this not very friendly. Invoking this causes the source to 
   *  execute clearMessagesAtSource(indoubtAction).
   *  This is performed on all streams in the stream set.
   *  The request to do this is hardened and will complete after a restart if
   *  necessary.
   * 
   * @param indoubtAction INDOUBT_DELETE causes indoubt messages to be discarded, risking their loss. 
   *                      INDOUBT_LEAVE means no action is taken for indoubt messages, 
   *                       so that they might be resent and duplicated. 
   * @throws SIMPRuntimeOperationFailedException 
   */
  void requestFlushAtSource(IndoubtAction indoubtAction) throws SIMPRuntimeOperationFailedException;
  
  /**
   * @return the number of outstanding requests that have not been
   * completed by the remote ME
   * @author tpm
   */
  int getNumberOfActiveRequests();
  
  /**
   * @return the number of requests that have been completed by the remote ME
   * @author tpm
   */
  long getNumberOfCompletedRequests();
  
  /**
   * @return a long for the number of message requests that have been issued
   * to this particular remote messaging engine
   */
  long getNumberOfRequestsIssued();  
  
  /**
   * Returns the tick of the last message received and acknowleged
   * @return long  The tick of the last message we received, and acknowleged
   */
  long getLastTickReceived()
    throws SIMPRuntimeOperationFailedException, SIMPControllableNotFoundException;   

  /**
   * If the source has been deleted there is no opportunity to complete the flush  
   *  or fill any gaps. This method will discard the target stream without completing 
   *  and ignoring any gaps. If the source still exists no messages will be lost because
   *  they can be retransmitted by the source. However there is a risk that messages 
   *  will be duplicated because an ack generated by the target may not have been received
   *  by the source, causing the source to send the same message again to a new instance 
   *  of the same stream which the target recreates.
   *  
   *  On completion no stream state exists, just as if flush had completed.
   *
   */
  void forceFlushAtTarget()
    throws SIMPRuntimeOperationFailedException, SIMPControllableNotFoundException;

  /**
   * All of the streams in a set are the same type.
   * @deprecated: a no-op for this type of stream
   * @return DeliveryStreamType  The type of stream, source target p2p etc.
   */
  DeliveryStreamType getType(); 
  
  /**
   * Cancel all message requests.
   */
  void cancelAllRequests();    

  /**
   * Determines whether this remote get stream is used for gathering messages from multiple MEs.
   */
  boolean isGathering();
  
  /**
   * Obtains the destination/alias uuid that this stream is retrieving/sending messages
   * from/to. If gathering is on, the field may show an alias which identifies a subset
   * of the destination's MEs from which to gather from.
   */
  String getDestinationUuid();
}
