/*******************************************************************************
 * Copyright (c) 2019,2022 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.config.admin.internal;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Future;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.Filter;
import org.osgi.service.cm.ConfigurationAdmin;

import com.ibm.websphere.ras.Tr;
import com.ibm.websphere.ras.TraceComponent;
import com.ibm.ws.config.admin.ConfigID;
import com.ibm.ws.config.admin.ConfigurationDictionary;
import com.ibm.ws.config.admin.ExtendedConfiguration;
import com.ibm.ws.config.admin.internal.ConfigurationStorageHelper.ConfigStorageConsumer;

import io.openliberty.checkpoint.spi.CheckpointHook;
import io.openliberty.checkpoint.spi.CheckpointPhase;

/**
 * ConfigurationStore manages all active configurations along with persistence. The current
 * implementation uses a filestore and serialization of the configuration dictionaries to files
 * identified by their pid. Persistence details are in the constructor, saveConfiguration, and
 * deleteConfiguration and can be factored out separately if required.
 */
class ConfigurationStore implements Runnable, CheckpointHook {
    private static final TraceComponent tc = Tr.register(ConfigurationStore.class, ConfigAdminConstants.TR_GROUP, ConfigAdminConstants.NLS_PROPS);

    private final ConfigAdminServiceFactory caFactory;

    private final Map<String, ExtendedConfigurationImpl> configurations;

    private final File persistentConfig;

    private final Lock saveMonitor = new ReentrantLock();
    private volatile Future<?> saveTask = null;
    private boolean shutdown = false;
    private final ReentrantReadWriteLock storeMonitor = new ReentrantReadWriteLock();

    /**
     * Simple integer counter type. Used for generating reference PIDs
     * for configuration elements.
     */
    public static class Counter {
        private int value;

        public Counter() {
            this.value = 0;
        }

        public int increment() {
            return value++;
        }
    }

    /**
     * A counter for PID names to avoid collisions.
     *
     * This operation is not thread safe: The caller,
     * {@link #createFactoryConfiguration(String, String)}, ensures
     * safety by including the invocation in a region that is guarded
     * by the configurations write lock.
     *
     * Keys are factory PIDs. Values are counters, which are used to generate
     * reference PIDs for configuration elements.
     */
    private final Map<String, Counter> configCount = new HashMap<>();

    /**
     * Generate a new count for a given factory PID.
     *
     * Counts start at 0. And are generated by incrementing the previous
     * count.
     *
     * This operation is not thread safe: The caller,
     * {@link #createFactoryConfiguration(String, String)}, ensures
     * safety by including the invocation in a region that is guarded
     * by the configurations write lock.
     *
     * Each factory PID has its own counter. That enables the configuration
     * admin to continue its main processing loops, which aggregate the
     * processing of configurations by factory PID, and which iterate across
     * the factory PIDs in an indeterminate order.
     *
     * The processing per factory PID orders the configuration data by sequence
     * number. That ensures that minimally changed configurations obtain the
     * same PIDs for the same configuration elements.
     *
     * See issue 22058 "Unnecessary application expansion on restart".
     * See also changes made to {@link com.ibm.ws.config.xml.internal.ChangeHandler},
     * which orders the processing of configuration data by sequence number.
     *
     * @param factoryPid The factory PID for which to generate a new count.
     *
     * @return The new count for the specified factory PID.
     */
    private long getCount(String factoryPid) {
        Counter counter = configCount.computeIfAbsent(factoryPid, (String useFactoryPid) -> new Counter());
        return counter.increment();
    }

    /**
     * Generate a new PID for a given factory PID.
     *
     * PIDs are generated using a a factory PID specific counter.
     *
     * The generated PID is guaranteed to be unique relative to the
     * current stored configurations. New PIDs are generated sequentially
     * until a unique PID is generated.
     *
     * This operation is not thread safe: The caller,
     * {@link #createFactoryConfiguration(String, String)}, ensures
     * safety by including the invocation in a region that is guarded
     * by the configurations write lock.
     *
     * @param factoryPid The factory PID for which to generate a new PID.
     *
     * @return The new PID.
     */
    private String generatePid(String factoryPid) {
        String pid;
        do {
            pid = factoryPid + "_" + getCount(factoryPid);
        } while (configurations.containsKey(pid));
        return pid;
    }

    public ConfigurationStore(ConfigAdminServiceFactory configAdminServiceFactory, BundleContext bc) {
        this.caFactory = configAdminServiceFactory;
        this.persistentConfig = bc.getDataFile(ConfigAdminConstants.CONFIG_PERSISTENT);
        this.configurations = loadConfigurationDatas(this.persistentConfig);

        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
            Tr.debug(tc, "config store pids are [" + configurations.keySet() + "]");
    }

    private final void readLock() {
        storeMonitor.readLock().lock();
    }

    private final void readUnlock() {
        storeMonitor.readLock().unlock();
    }

    public final void writeLock() {
        if (storeMonitor.getReadHoldCount() > 0) {
            // this is not supported and will cause deadlock if allowed to proceed.
            // fail fast instead of deadlocking
            throw new IllegalMonitorStateException("Requesting upgrade to write lock.");
        }
        storeMonitor.writeLock().lock();
    }

    public final void writeUnlock() {
        storeMonitor.writeLock().unlock();
    }

    public void removeConfiguration(final String pid) {
        writeLock();
        try {
            configurations.remove(pid);
        } finally {
            writeUnlock();
        }
        save();
        if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
            Tr.debug(tc, "removed configuration pid = " + pid + ", remaining pids are [" + configurations.keySet() + "]");
    }

    public ExtendedConfigurationImpl getConfiguration(String pid, String location) {
        readLock();
        ExtendedConfigurationImpl config;
        try {
            config = configurations.get(pid);
        } finally {
            readUnlock();
        }
        if (config != null) {
            return config;
        }
        writeLock();
        try {
            config = configurations.get(pid);
            if (config == null) {
                config = new ExtendedConfigurationImpl(caFactory, location, null, pid, null, null, null);
                configurations.put(pid, config);
            }
        } finally {
            writeUnlock();
        }
        return config;
    }

    /**
     * Create and return a new factory configuration. Store the new configuration.
     *
     * Assign the factory PID and location as specified. Assign a generated PID
     * based on the factory PID and on a counter of configurations created
     * for the factory PID.
     *
     * @param factoryPid The factory PID of the new configuration.
     * @param location   The location of the new configuration.
     *
     * @return The new configuration.
     */
    public ExtendedConfiguration createFactoryConfiguration(String factoryPid, String location) {
        String pid;
        ExtendedConfigurationImpl config;
        writeLock();
        try {
            pid = generatePid(factoryPid);
            config = new ExtendedConfigurationImpl(caFactory, location, factoryPid, pid, null, null, null);
            configurations.put(pid, config);
        } finally {
            writeUnlock();
        }
        return config;
    }

    public ExtendedConfigurationImpl findConfiguration(String pid) {
        readLock();
        try {
            return configurations.get(pid);
        } finally {
            readUnlock();
        }
    }

    /*
     * This method can return deleted configurations
     */
    public ExtendedConfigurationImpl[] getFactoryConfigurations(String factoryPid) {
        readLock();
        try {
            List<ExtendedConfigurationImpl> resultList = new ArrayList<ExtendedConfigurationImpl>();
            for (ExtendedConfigurationImpl config : configurations.values()) {
                //bypassing the check to see if configuration was deleted
                String otherFactoryPid = config.getFactoryPid(false);
                if (otherFactoryPid != null && otherFactoryPid.equals(factoryPid))
                    resultList.add(config);
            }
            return resultList.toArray(new ExtendedConfigurationImpl[resultList.size()]);
        } finally {
            readUnlock();
        }
    }

    public ExtendedConfiguration[] listConfigurations(Filter filter) {
        List<ExtendedConfigurationImpl> resultList = new ArrayList<>();
        readLock();
        try {
            resultList.addAll(configurations.values());
        } finally {
            readUnlock();
        }
        for (Iterator<ExtendedConfigurationImpl> it = resultList.iterator(); it.hasNext();) {
            ExtendedConfigurationImpl config = it.next();
            if (!config.matchesFilter(filter)) {
                it.remove();
            }
        }
        int size = resultList.size();
        return size == 0 ? null : (ExtendedConfigurationImpl[]) resultList.toArray(new ExtendedConfigurationImpl[size]);
    }

    public void unbindConfigurations(Bundle bundle) {
        Collection<ExtendedConfigurationImpl> currentConfigs;
        readLock();
        try {
            currentConfigs = new ArrayList<>(configurations.values());
        } finally {
            readUnlock();
        }
        for (ExtendedConfigurationImpl config : currentConfigs) {
            config.unbind(bundle);
        }
    }

    void saveConfigurationDatas(boolean shutdown) throws IOException {
        Future<?> currentSaveTask;
        saveMonitor.lock();
        try {
            currentSaveTask = saveTask;
            saveTask = null;
            this.shutdown = shutdown;
        } finally {
            saveMonitor.unlock();
        }
        if (currentSaveTask == null && checkpointPhase.restored()) {
            // no need to save if there is no save task and we have restored
            return;
        }
        readLock();
        try {
            ConfigurationStorageHelper.store(persistentConfig, configurations.values());
            if (shutdown && currentSaveTask != null) {
                currentSaveTask.cancel(false);
            }

        } finally {
            readUnlock();
        }
    }

    private Map<String, ExtendedConfigurationImpl> loadConfigurationDatas(File configDatas) {
        if (configDatas.isFile()) {
            ConfigStorageConsumer<String, ExtendedConfigurationImpl> consumer = new ConfigStorageConsumer<String, ExtendedConfigurationImpl>() {
                @Override
                public ExtendedConfigurationImpl consumeConfigData(String location, Set<String> uniqueVars, Set<ConfigID> references, ConfigurationDictionary dict) {
                    String pid = (String) dict.get(Constants.SERVICE_PID);
                    String factoryPid = (String) dict.get(ConfigurationAdmin.SERVICE_FACTORYPID);
                    return new ExtendedConfigurationImpl(caFactory, location, factoryPid, pid, dict, references, uniqueVars);
                }

                @Override
                public String getKey(ExtendedConfigurationImpl configuration) {
                    return configuration.getPid(false);
                }
            };
            try {
                return ConfigurationStorageHelper.load(configDatas, consumer);
            } catch (IOException e) {
                // auto FFDC is fine
                return new HashMap<>();
            }
        }
        return new HashMap<>();
    }

    private boolean addedCheckpointSaveHook = false;
    private final CheckpointPhase checkpointPhase = CheckpointPhase.getPhase();

    void save() {
        if (saveTask != null) {
            return;
        }
        saveMonitor.lock();
        try {
            if (shutdown) {
                return;
            }
            if (!(checkpointPhase.restored())) {
                if (addedCheckpointSaveHook) {
                    return;
                }
                // Before a checkpoint case;
                // for checkpoint add hook to save during prepare
                if (addedCheckpointSaveHook = checkpointPhase.addMultiThreadedHook(this)) {
                    // Successfully added hook; return without background task to save
                    return;
                }
            }

            // Not during checkpoint; do the normal scheduled save in the background
            if (saveTask == null) {
                saveTask = caFactory.updateQueue.addScheduled(this);
            }
        } finally {
            saveMonitor.unlock();
        }
    }

    @Override
    public void prepare() {
        run();
    }

    @Override
    public void run() {
        try {
            saveConfigurationDatas(false);
        } catch (IOException e) {
            // Auto-FFDC is fine here
        }
    }
}
