/*******************************************************************************
 * Copyright (c) 2013, 2023 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.websphere.connectionpool.monitor;

import com.ibm.websphere.monitor.jmx.StatisticsMeter;

/**
 * <p>Management interface for the connection pool statistics MBean. One connection pool MBean instance exists per datasource or connection factory,
 * whether configured explicitly in server configuration or whether it exists implicitly due to a
 * <code>@DataSourceDefinition</code> or <code>@ConnectionFactoryDefinition</code> annotation. The Liberty profile makes this MBean available in its platform MBean server when the
 * monitor-1.0 feature is
 * enabled to allow monitoring of the connection pool.</p>
 *
 * <p>Important: the MBean instance is not available until the connection factory or data source is first used.</p>
 *
 * <p>The object name has the form <code>WebSphere:type=ConnectionPoolStats,name=*</code>
 * where name is either the JNDI name, if defined, or id of the DataSource or ConnectionFactory. The id may be explicitly defined or
 * generated by the server, as exemplified in the DefaultDataSource snippet below.<p>
 *
 * Object name examples:
 * <ul>
 * <li><code>WebSphere:type=ConnectionPoolStats,name=jdbc/db2</code>
 * <br>corresponds to a dataSource instance with a server configuration-defined JNDI name. For example,
 * <code>
 * <br>&#60;dataSource jndiName="jdbc/db2">
 * <br>&nbsp;...
 * <br>&#60;/dataSource>
 * </code>
 * </li>
 * <li><code>WebSphere:type=ConnectionPoolStats,name=dataSource[DefaultDataSource]/connectionManager[default-0]</code>
 * <br>corresponds to a dataSource instance configured as the Java EE default datasource. For example,
 * <code>
 * <br>&#60;dataSource id="DefaultDataSource">
 * <br>&nbsp;...
 * <br>&#60;/dataSource>
 * </code>
 * </li>
 * <li><code>WebSphere:type=ConnectionPoolStats,name=jmsConnectionFactory[cf1]/connectionManager[default-0]</code>
 * <br>corresponds to a connection factory instance explicitly configured in server configuration with an id. For example,
 * <code>
 * <br>&#60;jmsConnectionFactory id="cf1">
 * <br>&nbsp;...
 * <br>&#60;/jmsConnectionFactory>
 * </code>
 * </li>
 *
 * @ibm-api
 */
public interface ConnectionPoolStatsMXBean {

    /**
     * The total number of managed connections created since pool creation.
     */
    public long getCreateCount();

    /**
     * The total number of managed connections destroyed since pool creation.
     */
    public long getDestroyCount();

    /**
     * The number of connections that are in use, including multiple connections shared from a single managed connection.
     */
    public long getConnectionHandleCount();

    /**
     * The total number of managed connections in the free, shared, and unshared pools.
     */
    public long getManagedConnectionCount();

    /**
     * The average waiting time in milliseconds until a connection is granted if a connection is not currently available.
     */
    public double getWaitTime();

    /**
     * Retrieves StatisticMeter object of WaitTime detail, which provides statistical details on the connection wait time.
     *
     * @return wait time details
     */
    public StatisticsMeter getWaitTimeDetails();

    /**
     * The number of managed connections in the free pool.
     */
    public long getFreeConnectionCount();

    /**
     * The average time in milliseconds a connection is in use.
     */
    public double getInUseTime();

    /**
     * Retrieves StatisticMeter object of InUseTime detail, which provides statistical details on the connection in use time.
     *
     * @return in use time details
     */
    public StatisticsMeter getInUseTimeDetails();

    /**
     * The maximum connections allowed in the connection pool
     *
     * @return maximum connection count
     */
    public long getMaxConnectionsLimit();
}
