/*******************************************************************************
 * Copyright (c) 2021, 2024 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.javamail.fat;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;

import com.ibm.websphere.simplicity.LocalFile;
import com.ibm.websphere.simplicity.config.ServerConfiguration;
import com.ibm.websphere.simplicity.log.Log;
import com.ibm.ws.jmx.connector.client.rest.ClientProvider;

import componenttest.custom.junit.runner.FATRunner;
import componenttest.custom.junit.runner.Mode;
import componenttest.custom.junit.runner.Mode.TestMode;
import componenttest.topology.impl.LibertyServer;
import componenttest.topology.impl.LibertyServerFactory;

/**
 * J2EE JavaMailResource MBean tests
 *
 * 194522: This test has had a troubled history with not finding all the expected messages in the logs.
 * Sometimes which messages appear on which transition depend on the order of test execution.
 * Refactoring to minimize log checking while still testing for what is really supposed to happen.
 */

@Mode(TestMode.FULL)
@RunWith(FATRunner.class)
public class JavaMailResourceMBeanTest {

    private final static Class<?> logClass = JavaMailResourceMBeanTest.class;

    private static final String MAIL_MANAGEMENT_TWO_MAIL_SESSIONS_SERVER_XML = "JavaMailServerConfigurations/javaMail_j2eeManagement_twoMailSessions.xml";
    private static final String MAIL_MANAGEMENT_THREE_MAIL_SESSIONS_SERVER_XML = "JavaMailServerConfigurations/javaMail_j2eeManagement_threeMailSessions.xml";
    private static final String MAIL_ONLY_SERVER_XML = "JavaMailServerConfigurations/javaMail_only.xml";
    private static final String MANAGEMENT_ONLY_SERVER_XML = "JavaMailServerConfigurations/j2eeManagement_only.xml";
    private static final String ORIGINAL_SERVER_XML = "JavaMailServerConfigurations/server.xml";

    private static LibertyServer server = LibertyServerFactory.getLibertyServer("com.ibm.ws.javamail.management.j2ee.fat");
    private static ObjectName objectNameForQuery;
    private static JMXConnector jmxConnector;
    private static MBeanServerConnection connection;
    private static ServerConfiguration originalServerConfig;

    final static String CWWKE0701E = "CWWKE0701E.*registerInjectionProcessorProvider method has thrown an exception.*provider already registered for javax.mail.MailSessionDefinition";

    @BeforeClass
    public static void setUp() throws Exception {
        final String methodName = "setUp";
        Log.entering(logClass, methodName);

        // If the key file was generated by another test using the same server then it will be backed up in tmp
        // copy it back to the server...
        LocalFile keyFile = new LocalFile(server.pathToAutoFVTTestFiles + "/tmp/key.p12");
        if (keyFile.exists())
            keyFile.copyToDest(server.getMachine().getFile(server.getServerRoot() + "/resources/security/key.p12"));

        // Set up the trust store
        System.setProperty("javax.net.ssl.trustStore", server.getServerRoot() + "/resources/security/key.p12");
        System.setProperty("javax.net.ssl.trustStorePassword", "Liberty");
        System.setProperty("javax.net.ssl.trustStoreType", "PKCS12");

        Log.info(logClass, methodName, "Starting server=" + server.getServerName());
        server.startServer();

        Log.info(logClass, methodName, "Waiting for 'CWWKT0016I.*IBMJMXConnectorREST'");
        assertNotNull("'CWWKT0016I.*IBMJMXConnectorREST' was not received on server",
                      server.waitForStringInLog("CWWKT0016I.*IBMJMXConnectorREST"));

        Log.info(logClass, methodName, "Waiting for 'CWWKO0219I.*ssl'");
        assertNotNull("'CWWKO0219I.*ssl' was not received on server",
                      server.waitForStringInLog("CWWKO0219I.*ssl"));

        Log.info(logClass, methodName, "Waiting for 'CWWKS0008I'");
        assertNotNull("Security service did not report it was ready",
                      server.waitForStringInLog("CWWKS0008I"));

        // Backup the key file
        server.copyFileToTempDir("resources/security/key.p12", "key.p12");

        Map<String, Object> environment = new HashMap<String, Object>();
        environment.put("jmx.remote.protocol.provider.pkgs", "com.ibm.ws.jmx.connector.client");
        environment.put(JMXConnector.CREDENTIALS, new String[] { "theUser", "thePassword" });
        environment.put(ClientProvider.DISABLE_HOSTNAME_VERIFICATION, true);
        environment.put(ClientProvider.READ_TIMEOUT, 2 * 60 * 1000);
        JMXServiceURL url = new JMXServiceURL("REST", server.getHostname(), server.getHttpDefaultSecurePort(), "/IBMJMXConnectorREST");
        Log.info(logClass, methodName, "Connecting to JMX connector server with URL: " + url);
        jmxConnector = JMXConnectorFactory.connect(url, environment);
        assertNotNull("JMXConnector should not be null", jmxConnector);

        connection = jmxConnector.getMBeanServerConnection();
        assertNotNull("MBeanServerConnection should not be null", connection);

        // Store the original server config
        originalServerConfig = server.getServerConfiguration().clone();

        // Initialize an ObjectName that can be used to query JavaMailResource MBeans
        StringBuilder nameBuilder = new StringBuilder("WebSphere").append(":");
        nameBuilder.append("j2eeType").append("=").append("JavaMailResource").append(",");
        nameBuilder.append("J2EEServer").append("=").append(server.getServerName());
        nameBuilder.append(",*");

        objectNameForQuery = new ObjectName(nameBuilder.toString());

        Log.exiting(logClass, methodName);
    }

    /**
     * After running each test, restore to the original configuration.
     *
     * @throws Exception
     */
    @After
    public void restoreAfterTest() throws Exception {
        final String methodName = "restoreAfterTest";
        Log.entering(logClass, methodName);

        //server.setMarkToEndOfLog();
        changeConfig(ORIGINAL_SERVER_XML, true);
        //Log.info(logClass, methodName, "Waiting for 'CWWKF0008I'");
        //assertNotNull("'CWWKF0008I' was not received on server", server.waitForStringInLogUsingMark("CWWKF0008I"));

        Log.info(getClass(), methodName, "server configuration restored");

    }

    @AfterClass
    public static void tearDown() throws Exception {
        String methodName = "tearDown";
        Log.entering(logClass, methodName);

        if (server != null && server.isStarted()) {
            Log.finer(logClass, methodName, "Server is up, stopping it");

            server.setMarkToEndOfLog(server.getMostRecentTraceFile());
            jmxConnector.close();

            server.stopServer("SRVE0315E", "SRVE8502W", CWWKE0701E);

        }

        Log.exiting(logClass, methodName);
    }

    @Test
    public void testNoMBeanAddedWhenJavaMailOnly() throws Exception {
        String methodName = "testNoMBeanAddedWhenJavaMailOnly";

        //server.setMarkToEndOfLog(); - changeConfig takes care of this.
        changeConfig(MAIL_ONLY_SERVER_XML, true);
        //server.setServerConfigurationFile(MAIL_ONLY_SERVER_XML);

        // let's not do this next check because it's sensitive to test order.
        //Log.info(logClass, methodName, "Waiting for 'CWWKF0012I.*javaMail'");
        // CWWKF0012I: The server installed the following features: [javaMail-1.5]
        // assertNotNull("'CWWKF0012I.*javaMail' was not recieved on server", server.waitForStringInLogUsingMark("CWWKF0012I.*javaMail"));
        // Log.info(logClass, methodName, "Server configuration successfully updated.");

        /*------------------
        // 191546 change from UsingLastOffset to Using Mark, the former is obviously not working
        // as line was present 6 seconds after we started the test.
        // UsingLastOffset calls some deprecated internal functions.
        // server.waitForStringInLogUsingLastOffset
        assertNotNull("IBMJMXConnectorREST app did not report as ready",
                      server.waitForStringInLogUsingMark("CWWKT0016I.*IBMJMXConnectorREST"));
        
        assertNotNull("JMX REST connector did not report as ready",
                      server.waitForStringInLogUsingMark("CWWKX0103I"));
        
        assertNotNull("The security service did not report it was ready.",
                      server.waitForStringInLogUsingMark("CWWKS0008I"));
        ----------------------*/

        Set<ObjectName> allJavaMailResources = connection.queryNames(objectNameForQuery, null);
        assertTrue("There should not be any JavaMailResource MBean registered. allJavaMailResources=" + allJavaMailResources, allJavaMailResources.size() == 0);
    }

    @Test
    public void testNoMBeanAddedWhenJ2EEManagementOnly() throws Exception {
        String methodName = "testNoMBeanAddedWhenJ2EEManagementOnly";

        server.setMarkToEndOfLog();
        changeConfig(MANAGEMENT_ONLY_SERVER_XML, true);

        // Confirm the MBean server did start for the server
        // assertNotNull("MBeanServer connector did not report as ready",
        //               server.waitForStringInLogUsingLastOffset("SRVE0242I:.*MBeanServerConnector.*Initialization successful"));

        Set<ObjectName> allJavaMailResources = connection.queryNames(objectNameForQuery, null);
        assertTrue("There should not be any JavaMailResource MBean registered. allJavaMailResources=" + allJavaMailResources, allJavaMailResources.size() == 0);
    }

    @Test
    public void testMBeansExistsForMailSessions() throws Exception {
        String methodName = "testMBeansExistsForMailSessions";

        server.setMarkToEndOfLog();
        server.setMarkToEndOfLog(server.getMostRecentTraceFile());

        changeConfig(MAIL_MANAGEMENT_TWO_MAIL_SESSIONS_SERVER_XML, true);
        /*
         * server.setServerConfigurationFile(MAIL_MANAGEMENT_TWO_MAIL_SESSIONS_SERVER_XML);
         * Log.info(logClass, methodName, "Waiting for 'CWWKF0008I'");
         * assertNotNull("'CWWKF0008I' was not received on server", server.waitForStringInLogUsingMark("CWWKF0008I"));
         * Log.info(logClass, methodName, "Feature update completed.");
         */

        /*--------------
        assertNotNull("IBMJMXConnectorREST app did not report as ready",
                      server.waitForStringInLogUsingMark("CWWKT0016I.*IBMJMXConnectorREST"));
        
        assertNotNull("JMX REST connector did not report as ready",
                      server.waitForStringInLogUsingMark("CWWKX0103I"));
        
        assertNotNull("The security service did not report it was ready.",
                      server.waitForStringInLogUsingMark("CWWKS0008I"));
        ----------------- */

        // not sure we need this either, it takes a long time to search that trace log, but we'll leave it in.
        assertNotNull("Mail session (IMAP) was not processed", server.waitForStringInTraceUsingMark(".*description=mailSession for testing IMAP protocol"));
        assertNotNull("Mail session (SMTP) was not processed", server.waitForStringInTraceUsingMark(".*description=mailSession for testing SMTP protocol"));

        Set<ObjectName> allJavaMailResources = connection.queryNames(objectNameForQuery, null);
        assertTrue("There should be 2 JavaMailResource MBeans registered. allJavaMailResources=" + allJavaMailResources, allJavaMailResources.size() == 2);
    }

    @Test
    public void testMBeansWhenAddingOrRemovingMailSessions() throws Exception {

        String methodName = "testMBeansWhenAddingOrRemovingMailSessions";

        //Check 1
        Log.info(logClass, methodName, "Updating server.xml with " + MAIL_MANAGEMENT_TWO_MAIL_SESSIONS_SERVER_XML);
        changeConfig(MAIL_MANAGEMENT_TWO_MAIL_SESSIONS_SERVER_XML, true);

        Set<ObjectName> allJavaMailResources = connection.queryNames(objectNameForQuery, null);
        Log.info(logClass, methodName, "allJavaMailResources=" + allJavaMailResources);
        assertTrue("There should be 2 JavaMailResource MBeans registered. allJavaMailResources=" + allJavaMailResources, allJavaMailResources.size() == 2);

        //Check 2
        Log.info(logClass, methodName, "Updating server.xml with " + MAIL_MANAGEMENT_THREE_MAIL_SESSIONS_SERVER_XML);
        changeConfig(MAIL_MANAGEMENT_THREE_MAIL_SESSIONS_SERVER_XML, false);

        assertNotNull("Mail session (POP3) was not processed", server.waitForStringInTraceUsingMark(".*description=mailSession for testing Pop3 protocol"));

        Set<ObjectName> allJavaMailResources2 = connection.queryNames(objectNameForQuery, null);
        Log.info(logClass, methodName, "allJavaMailResources2=" + allJavaMailResources2);
        assertTrue("There should be 3 JavaMailResource MBeans registered. allJavaMailResources2=" + allJavaMailResources2, allJavaMailResources2.size() == 3);

        //Check 3
        Log.info(logClass, methodName, "Updating server.xml with " + MAIL_MANAGEMENT_TWO_MAIL_SESSIONS_SERVER_XML);
        changeConfig(MAIL_MANAGEMENT_TWO_MAIL_SESSIONS_SERVER_XML, false);

        Set<ObjectName> allJavaMailResources3 = connection.queryNames(objectNameForQuery, null);
        Log.info(logClass, methodName, "allJavaMailResources3=" + allJavaMailResources3);
        assertTrue("There should be 2 JavaMailResource MBeans registered. allJavaMailResources3=" + allJavaMailResources3, allJavaMailResources3.size() == 2);
    }

    /**
     * @param newServerXml
     * @param featureChanged
     * @throws Exception
     */
    private void changeConfig(String newServerXml, boolean featureChanged) throws Exception {

        String methodName = "changeConfig";

        server.setMarkToEndOfLog();
        server.setServerConfigurationFile(newServerXml);

        // this was sensitive to test order, just stop doing this.
        if (featureChanged && false) {
            Log.info(logClass, methodName, "Waiting for 'CWWKF0008I' due to feature change");
            assertNotNull("'CWWKF0008I' was not received on server", server.waitForStringInLogUsingMark("CWWKF0008I"));
            Log.info(logClass, methodName, "Feature update completed.");

            assertNotNull("IBMJMXConnectorREST app did not report as ready",
                          server.waitForStringInLogUsingMark("CWWKT0016I.*IBMJMXConnectorREST"));

            assertNotNull("JMX REST connector did not report as ready",
                          server.waitForStringInLogUsingMark("CWWKX0103I"));

            assertNotNull("The security service did not report it was ready.",
                          server.waitForStringInLogUsingMark("CWWKS0008I"));
        } else {
            // CWWKG0017I: The server configuration was successfully updated
            // this is the -one- message that gets emitted consistently for any change.
            Log.info(logClass, methodName, "Waiting for 'CWWKG0017I', server configuration was successfully updated");
            assertNotNull("'CWWKG0017I' was not received on server", server.waitForStringInLogUsingMark("CWWKG0017I"));
            // rather than wait for more messages that depend on test order and sometimes go missing, just
            // give things another 20 seconds to settle down.
            Log.info(logClass, methodName, "found cwwkg0017i, now wait a bit for the dust to settle");
            Thread.sleep(20000);
        }
        Log.info(logClass, methodName, "changeConfig exit");

    }

}
