/*******************************************************************************
 * Copyright (c) 2019, 2021 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.jdbc.fat.sqlserver;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;

import org.junit.runner.RunWith;
import org.junit.runners.Suite;
import org.junit.runners.Suite.SuiteClasses;
import org.testcontainers.containers.MSSQLServerContainer;

import com.ibm.websphere.simplicity.log.Log;

import componenttest.containers.TestContainerSuite;

@RunWith(Suite.class)
@SuiteClasses({
                SQLServerTest.class,
                SQLServerSSLTest.class
})
public class FATSuite extends TestContainerSuite {

    public static final String DB_NAME = "test";
    public static final String TABLE_NAME = "MYTABLE";

    /**
     * Create database and tables needed by test servlet.
     * Use a native JDBC connection from the driver so that there aren't any dependencies on the appserver.
     * The SQLServer container already has knowledge of how to create a native JDBC connection.
     *
     * @throws SQLException
     */
    public static void setupDatabase(MSSQLServerContainer<?> sqlserver, boolean ssl) throws SQLException {
        /*
         * IBM JDK will return TLSv1 when SSLContext.getInstance(TLS) is called.
         * Force driver to use TLSv1.2 for this setup step.
         * See documentation here: https://github.com/microsoft/mssql-jdbc/wiki/SSLProtocol
         */
        if (ssl) {
            sqlserver.withUrlParam("SSLProtocol", "TLSv1.2");
        }

        //Setup database and settings
        Log.info(FATSuite.class, "setupDatabase", "Attempting to setup database with name: " + DB_NAME + "."
                                                  + " With connection URL: " + sqlserver.getJdbcUrl());
        try (Connection conn = sqlserver.createConnection(""); Statement stmt = conn.createStatement()) {
            stmt.execute("CREATE DATABASE [" + DB_NAME + "];");
            stmt.execute("EXEC sp_sqljdbc_xa_install");
            stmt.execute("ALTER DATABASE " + DB_NAME + " SET ALLOW_SNAPSHOT_ISOLATION ON");
        }

        //Create test table
        sqlserver.withUrlParam("databaseName", DB_NAME);
        Log.info(FATSuite.class, "setupDatabase", "Attempting to setup database table with name: " + TABLE_NAME + "."
                                                  + " With connection URL: " + sqlserver.getJdbcUrl());
        try (Connection conn = sqlserver.createConnection(""); Statement stmt = conn.createStatement()) {
            // Create tables
            int version = conn.getMetaData().getDatabaseMajorVersion();
            try {
                if (version >= 13) // SQLServer 2016 or higher
                    stmt.execute("DROP TABLE IF EXISTS " + TABLE_NAME);
                else
                    stmt.execute("DROP TABLE " + TABLE_NAME);
            } catch (SQLException x) {
                // probably didn't exist
            }
            stmt.execute("CREATE TABLE " + TABLE_NAME + " (ID SMALLINT NOT NULL PRIMARY KEY, STRVAL NVARCHAR(40))");
        }
    }
}