/*******************************************************************************
 * Copyright (c) 2019 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.jndi.ejb;

import static org.osgi.service.component.annotations.ConfigurationPolicy.IGNORE;

import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.Name;
import javax.naming.OperationNotSupportedException;
import javax.naming.spi.ObjectFactory;

import org.osgi.framework.ServiceReference;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;

import com.ibm.websphere.ras.Tr;
import com.ibm.websphere.ras.TraceComponent;
import com.ibm.ws.container.service.naming.EJBLocalNamingHelper;
import com.ibm.wsspi.kernel.service.utils.ConcurrentServiceReferenceSet;

/**
 * EJBLocalContextFactory is an implementation of {@link ObjectFactory} that is
 * intended to be registered in the SR with the osgi.jndi.url.scheme=ejblocal
 * property. As such it is called for handling JNDI names starting with "ejblocal:".
 *
 */
@Component(configurationPolicy = IGNORE, property = { "service.vendor=ibm", "osgi.jndi.url.scheme=ejblocal" })
public class EJBLocalURLContextFactory implements ObjectFactory {

    static final TraceComponent tc = Tr.register(EJBLocalURLContextFactory.class);

    private final ConcurrentServiceReferenceSet<EJBLocalNamingHelper> helperServices = new ConcurrentServiceReferenceSet<EJBLocalNamingHelper>("helpers");

    @Reference(name = "helpers", cardinality = ReferenceCardinality.MULTIPLE, policy = ReferencePolicy.DYNAMIC, bind = "addHelper", unbind = "removeHelper")
    public void addHelper(ServiceReference<EJBLocalNamingHelper> reference) {
        helperServices.addReference(reference);
    }

    public void removeHelper(ServiceReference<EJBLocalNamingHelper> reference) {
        helperServices.removeReference(reference);
    }

    public void activate(ComponentContext cc) {
        helperServices.activate(cc);
    }

    public void deactivate(ComponentContext cc) {
        helperServices.deactivate(cc);
    }

    /**
     * Creates a EJBLocalURLContext for resolving ejblocal: urls. It should only be
     * called by an OSGi JNDI spec implementation. There is no support for
     * non-null Name and Context parameters for this method in accordance with
     * the OSGi specification for JNDI.
     *
     * <UL>
     * <LI>If the parameter o is null a new {@link EJBLocalURLContext} is returned.
     * <LI>If o is a URL String then the Object returned is the result of
     * looking up the String on a {@link EJBLocalURLContext}.
     * <LI>If o is an array of URL Strings then an {@link OperationNotSupportedException} is thrown as there is no
     * sub-context support in this implementation from which to lookup multiple
     * names.
     * <LI>If o is any other Object an {@link OperationNotSupportedException} is
     * thrown.
     * </UL>
     *
     * @param o
     *            {@inheritDoc}
     * @param n
     *            must be null (OSGi JNDI spec)
     * @param c
     *            must be null (OSGi JNDI spec)
     * @param envmt
     *            {@inheritDoc}
     * @return
     * @throws OperationNotSupportedException
     *             if the Object passed in is not null or a String.
     */
    @Override
    public Object getObjectInstance(Object o, Name n, Context c, Hashtable<?, ?> env) throws Exception {
        final boolean isTraceOn = TraceComponent.isAnyTracingEnabled();
        if (isTraceOn && tc.isEntryEnabled()) {
            Tr.entry(tc, "getObjectInstance:");
        }

        if (isTraceOn && tc.isDebugEnabled()) {
            Tr.debug(tc, "Object: " + o);
            Tr.debug(tc, "Name: " + n);
            Tr.debug(tc, "Context: " + c);
            Tr.debug(tc, "Env: " + env);
        }

        // by OSGi JNDI spec Name and Context should be null
        // if they are not then this code is being called in
        // the wrong way
        if (n != null || c != null) {
            if (isTraceOn && tc.isEntryEnabled()) {
                Tr.exit(tc, "getObjectInstance: null");
            }
            return null;
        }
        // Object is String, String[] or null
        // Hashtable contains any environment properties
        Object context;
        if (o == null) {
            context = new EJBLocalURLContext(env, helperServices);
        } else if (o instanceof String) {
            context = new EJBLocalURLContext(env, helperServices).lookup((String) o);
        } else {
            throw new OperationNotSupportedException();
        }
        if (isTraceOn && tc.isEntryEnabled()) {
            Tr.exit(tc, "getObjectInstance: " + context);
        }
        return context;
    }

}
