#!/bin/sh

#
# A script that uses the RHQ and JON tags passed in to build the source
# artifacts to be uploaded/published. Generates jboss-on-src-$JON_VERSION.zip
# See ftp://ftp.redhat.com/redhat/jon/3.0.0/en.source/ for example.
#
# i)checkout RHQ sources from tag
# ii)copy {RHQ}/rhq/modules/* to BUILD
# iii)Uses JON repo source where this script is hosted.
# iv)copy {JON}/jon/* to BUILD
# v)zip up jon/* & rhq/* from BUILD directory

#include the utility library
source ../../../rhq_bash.lib

#
#RHQ_TAG="RHQ_4_2_0_JON300_GA"
#JON_TAG="JBossON_3_0_0_GA"
#JON_VERSION="3.0.0.GA"

## Collect directory information for source generation
# Stores off the module directory
MODULE_DIR=`pwd`;
# Stores off the JON repo root directory in relative location format.
REPO_PWD=$(pwd)/../../..;
# convert and store off directories to generate the absolute paths.
cd "$REPO_PWD";
GIT_REPO_DIR=`pwd`;
# return to original directory.
cd "$MODULE_DIR"
# Store off absolute path to target folder
TARGET=$(pwd)
TARGET="$TARGET/target"

echo
# Set up location to build out copies of RHQ and JON repositories.
SRC_ROOT="$TARGET/source_artifacts"
RHQ_SRC_ROOT=$SRC_ROOT/RHQ_SRC
JON_SRC_ROOT=$SRC_ROOT/JON_SRC
BUILD="$SRC_ROOT/BUILD"
RHQ=rhq
JON=jon

#========================================================================================
# Description: Initialize. Creates SRC and BUILD directories.
#========================================================================================
initialize()
{
 echo 
 echo "Listing environment variables used in building JON source artifacts:"
 environment_variables=( "RHQ_TAG" "JON_TAG" "JON_VERSION" "SRC_ROOT" "RHQ_SRC_ROOT" "JON_SRC_ROOT" "BUILD" "UPDATE_ID" )
 print_variables "${environment_variables[@]}"
 echo 

 ## Test whether kerberos ticket already cached
 KERBEROS_CHECK="kerberos-check.txt";
 ## Note: careful to pipe error and std out to file, or check won't work. 'klist' acess can throw error.
 klist 2>&1 | cat > "$KERBEROS_CHECK"

 if grep -q "No credentials" "$KERBEROS_CHECK";
 then
    echo "[ERROR] No kerberos credentials found."
    echo -e "\t Execute 'kinit' to authenticate before proceeding. Exiting now..."
    exit 1;
 fi

 # Store off original git details
 GIT_COMMIT=`git log -1 > git-commit.txt ; grep commit git-commit.txt | cut -d' ' -f2 | cut -c 1-7`;
 echo -e "\t [GIT_COMMIT=>[$GIT_COMMIT]]"
 GIT_BRANCH=`git branch | grep "*" | cut -d' ' -f2`
 echo -e "\t [GIT_BRANCH=>[$GIT_BRANCH]]"

 ## clean root artifact directory
 echo "Setting up the directories to build source artifacts in..."

 # Save the path to the script
 #GIT_REPO_DIR=`dirname $0`/../../..;
 echo -e "\t [Git repo directory is [$GIT_REPO_DIR]]";
 echo -e "\t [rm -rf [$SRC_ROOT]]"
 rm -rf $SRC_ROOT

 echo -e "\t [mkdir -p [$SRC_ROOT]]"
 mkdir -p "$SRC_ROOT"

 # Create BUILD root for JON release
 echo -e "\t [mkdir -p $BUILD]"
 mkdir -p $BUILD
}

#========================================================================================
# Description: Validate and parse input arguments
#========================================================================================
parse_and_validate_options()
{
   print_function_information $FUNCNAME

   RHQ_TAG=
   JON_TAG=
   JON_VERSION=
   UPDATE_ID=

   short_options="h"
   long_options="help,rhq-tag:,jon-tag:,jon-version:,update-id:"

   PROGNAME=${0##*/}
   ARGS=$(getopt -s bash --options $short_options --longoptions $long_options --name $PROGNAME -- "$@" )
   eval set -- "$ARGS"

   while true; do
	   case $1 in
         -h|--help)
            usage
            ;;
         --rhq-tag)
            shift
            RHQ_TAG="$1"
            shift
            ;;
         --jon-tag)
            shift
            JON_TAG="$1"
            shift
            ;;
         --jon-version)
            shift
            JON_VERSION="$1"
            shift
            ;;
         --update-id)
            shift
            UPDATE_ID="$1"
            shift
            ;;
         --)
            shift
            break
            ;;
         *)
            usage
            ;;
	   esac
   done

   if [ -z "$RHQ_TAG" ];
   then
      usage "RHQ tag not specified!"
   fi

   if [ -z "$JON_VERSION" ];
   then
      usage "JON version not specified!"
   fi

   if [ -z "$JON_TAG" ];
   then
      usage "JON tag not specified!"
   fi

   print_centered "Script Options"
   script_options=( "RHQ_TAG" "JON_TAG" "JON_VERSION" "UPDATE_ID")
   print_variables "${script_options[@]}"
}

#========================================================================================
# Description: Display usage information then abort the script.
#========================================================================================
usage() 
{
   USAGE=$(
cat << EOF
USAGE:   generate-jon-source-artifact.sh OPTIONS

   --rhq-tag=tag              [REQUIRED]
      The RHQ tag to be used by this script.

   --jon-tag=tag              [REQUIRED]
      The JON tag to be used by this script.

   --jon-version=version      [REQUIRED]
      The verions of JON used to name the source artifacts.

   --update-id=number      [OPTIONAL]
      The cumulative patch version. Ex. 01 for cumulative patch 01
EOF
)

   EXAMPLE="generate-jon-source-artifact.sh --rhq-tag=\"RHQ_4_2_0_JON300_GA\" --jon-tag=\"JBossON_3_0_0_GA\" --jon-version=\"3.0.0.GA\""

   abort "$@" "$USAGE" "$EXAMPLE"
}


#========================================================================================
# Description: Perform RHQ source retrieval from TAG passed in.
#========================================================================================
get_rhq_source_for_build(){

 # Create RHQ source directory for JON release
 echo "Retrieving RHQ Source"
 echo -e "\t [mkdir -p $RHQ_SRC_ROOT]"
 mkdir -p "$RHQ_SRC_ROOT"

 echo
 echo "Pulling down the RHQ source components..."
 echo -e "\t [cd $RHQ_SRC_ROOT]"
 cd $RHQ_SRC_ROOT

 echo -e "\t [git clone git://git.app.eng.bos.redhat.com/rhq.git]"
 git clone git://git.app.eng.bos.redhat.com/rhq.git

 echo -e "\t Change to the git checkout dir"
 echo -e "\t [cd $RHQ]"
 cd "$RHQ"

 # Remove old cruft[target and extra files]
 echo -e "\t [git clean -n]"
 git clean -n

 echo
 echo "Checking out the RHQ tag used for JON build."
 echo -e "\t [git checkout $RHQ_TAG]"
 git checkout $RHQ_TAG

 echo
 echo "Copying RHQ src components to [$BUILD]"
 echo -n " Current directory is: "; pwd
 echo -e "\t [mkdir -p $BUILD/$RHQ/modules]"
 mkdir -p "$BUILD/$RHQ/modules"

 echo -e "\t [cp -r $RHQ_SRC_ROOT/$RHQ/modules $BUILD/$RHQ/modules]"
 cp -r "$RHQ_SRC_ROOT/$RHQ"/modules/* "$BUILD/$RHQ"/modules

 echo -e "\t [cp $RHQ_SRC_ROOT/$RHQ/pom.xml $BUILD/$RHQ/pom.xml]"
 cp "$RHQ_SRC_ROOT/$RHQ"/pom.xml "$BUILD/$RHQ"/pom.xml

 # cd back to JON GIT repo root folder now that we're done retrieving RHQ source.
 echo -e "\t [returning to base directory [$GIT_REPO_DIR]]"
 cd "$GIT_REPO_DIR"
}

#========================================================================================
# Description: Perform JON source retrieval from TAG passed in.
#========================================================================================
get_jon_source_for_build(){

 ## Create JON source directory for JON release
 # create familiar structure that would have been created by a new clone process.
 echo -e "\t [mkdir -p $JON_SRC_ROOT/jon]";
 mkdir -p "$JON_SRC_ROOT/jon"

 # recursively copy all content into $JON_SRC_ROOT/jon
 echo -e "\t [cp -r * $JON_SRC_ROOT/jon]"
 cp -r * "$JON_SRC_ROOT/jon"

 # Also copy over the .git* content so git commands execute only on the embedded
 echo -e "\t [cp -r .git* $JON_SRC_ROOT/jon]"
 cp -r .git* "$JON_SRC_ROOT/jon"

 # Build JON source component for JON release
 echo -e "\t [cd $JON_SRC_ROOT/$JON]"
 cd "$JON_SRC_ROOT/$JON"

 # Remove old cruft[target and extra files]
 echo -e "\t [git clean -d -f]"
 git clean -d -f 

 # Checkout/reset to the tag state.
 echo -e "\t [git reset --hard $JON_TAG]"
 git reset --hard "$JON_TAG"

 echo
 echo "Copying JON src components to $BUILD"
 mkdir -p "$BUILD/$JON/jon"
 echo -e "\t [cp -r $JON_SRC_ROOT/$JON/jon/* $BUILD/$JON/jon]"
 cp -r "$JON_SRC_ROOT/$JON"/jon/* "$BUILD/$JON/jon"

 echo "Applying rhq build time changes."
 echo -e "\t [git apply --directory=\"$BUILD/$RHQ\" \"$JON_SRC_ROOT/$JON/patches/rhq/001.brew-versions.patch\"]";
 git apply --directory="$BUILD/$RHQ" "$JON_SRC_ROOT/$JON/patches/rhq/001.brew-versions.patch" 

 echo "Display generated RHQ productization overlay."
 echo -e "\t [git diff]";
 git diff;

 # cd back to root folder now that we're done with git cmds.
 echo -e "\t [cd $GIT_REPO_DIR]"
 cd "$GIT_REPO_DIR"

}

# zip up all components
#========================================================================================
# Description: Perform JON source retrieval from TAG passed in.
#========================================================================================
create_jon_source_artifact(){ 
 #empty patch name by default.
 PATCHED_SOURCE=""
 if [[ -z "$UPDATE_ID" ]] || [[ "$UPDATE_ID" == "\${src.jon.update.id.value}" ]] ;
 then
      :; ## no op
 else
    # Otherwise update PATCHED_SOURCE appropriately
    PATCHED_SOURCE="-update-$UPDATE_ID";
 fi

 cd $BUILD
 ARTIFACT_VERSION="$JON_VERSION$PATCHED_SOURCE"
 SOURCE_ARTIFACT="jboss-on-src-$ARTIFACT_VERSION"
 mkdir "$SOURCE_ARTIFACT"
 cp -r jon $SOURCE_ARTIFACT
 cp -r rhq $SOURCE_ARTIFACT
 
 # brew artifact creation to maven approach. Put in /target folder.
 # spinder: 10-24-14: used maven-assembly-plugin instead.
 #zip "$SOURCE_ARTIFACT.zip" -r $SOURCE_ARTIFACT 

 # return to root folder
 cd "$GIT_REPO_DIR"

 # revert to original git content
 echo -e "\t Resetting git content to '$GIT_BRANCH' with commit '$GIT_COMMIT'";
 echo -e "\t git checkout $GIT_BRANCH"
 git checkout "$GIT_BRANCH"
 echo -e "\t git reset $GIT_COMMIT";
 git reset  "$GIT_COMMIT"

 echo
}

############ MAIN SCRIPT ############

parse_and_validate_options $@

initialize

get_rhq_source_for_build

get_jon_source_for_build

create_jon_source_artifact

## See BZ 1091480 for more details. This needs to be a brew build.
#Replaced by brew build. Uncomment locally and run after maven build complete when not built on brew.
#echo -e "\t Calculating checksum for the generated source artifact [$BUILD/jboss-on-src-$ARTIFACT_VERSION.zip] :"
#openssl dgst -md5 "$GIT_REPO_DIR/jon/dist/jboss-on-src/target/jboss-on-src-$ARTIFACT_VERSION.zip" 


