/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.slack.springboot;

import java.util.HashMap;
import java.util.Map;

import org.apache.camel.CamelContext;
import org.apache.camel.Component;
import org.apache.camel.EndpointInject;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.component.extension.ComponentVerifierExtension;
import org.apache.camel.component.mock.MockEndpoint;
import org.apache.camel.spring.boot.CamelAutoConfiguration;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.assertEquals;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.apache.camel.test.spring.junit5.CamelSpringBootTest;

@DirtiesContext
@CamelSpringBootTest
@SpringBootTest(classes = { CamelAutoConfiguration.class, SlackComponentVerifierExtensionTest.class })
public class SlackComponentVerifierExtensionTest {

    @Autowired
    private CamelContext context;

    @Autowired
    ProducerTemplate template;

    @EndpointInject("mock:authors")
    MockEndpoint mock;

    @Test
    public void testParametersWebhook() throws Exception {
        Component component = context.getComponent("slack");

        ComponentVerifierExtension verifier = component.getExtension(ComponentVerifierExtension.class)
                .orElseThrow(IllegalStateException::new);

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("webhookUrl", "l");

        ComponentVerifierExtension.Result result = verifier.verify(ComponentVerifierExtension.Scope.PARAMETERS,
                parameters);

        assertEquals(ComponentVerifierExtension.Result.Status.OK, result.getStatus());
    }

    @Test
    public void testParametersToken() throws Exception {
        Component component = context.getComponent("slack");

        ComponentVerifierExtension verifier = component.getExtension(ComponentVerifierExtension.class)
                .orElseThrow(IllegalStateException::new);

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("token", "l");

        ComponentVerifierExtension.Result result = verifier.verify(ComponentVerifierExtension.Scope.PARAMETERS,
                parameters);

        assertEquals(ComponentVerifierExtension.Result.Status.OK, result.getStatus());
    }

    @Test
    public void testParametersEmpty() throws Exception {
        Component component = context.getComponent("slack");

        ComponentVerifierExtension verifier = component.getExtension(ComponentVerifierExtension.class)
                .orElseThrow(IllegalStateException::new);

        Map<String, Object> parameters = new HashMap<>();

        ComponentVerifierExtension.Result result = verifier.verify(ComponentVerifierExtension.Scope.PARAMETERS,
                parameters);

        assertEquals(ComponentVerifierExtension.Result.Status.ERROR, result.getStatus());
    }

    @Test
    public void testParametersWebhookUrlTokenBoth() throws Exception {
        Component component = context.getComponent("slack");

        ComponentVerifierExtension verifier = component.getExtension(ComponentVerifierExtension.class)
                .orElseThrow(IllegalStateException::new);

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("token", "l");
        parameters.put("webhookUrl", "l");

        ComponentVerifierExtension.Result result = verifier.verify(ComponentVerifierExtension.Scope.PARAMETERS,
                parameters);

        assertEquals(ComponentVerifierExtension.Result.Status.ERROR, result.getStatus());
    }

    @Test
    public void testConnectivityWebhook() throws Exception {
        Component component = context.getComponent("slack");
        ComponentVerifierExtension verifier = component.getExtension(ComponentVerifierExtension.class)
                .orElseThrow(IllegalStateException::new);

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("webhookUrl", "l");

        ComponentVerifierExtension.Result result = verifier.verify(ComponentVerifierExtension.Scope.CONNECTIVITY,
                parameters);

        assertEquals(ComponentVerifierExtension.Result.Status.ERROR, result.getStatus());
    }

    @Test
    public void testConnectivityToken() throws Exception {
        Component component = context.getComponent("slack");
        ComponentVerifierExtension verifier = component.getExtension(ComponentVerifierExtension.class)
                .orElseThrow(IllegalStateException::new);

        Map<String, Object> parameters = new HashMap<>();
        parameters.put("token", "l");

        ComponentVerifierExtension.Result result = verifier.verify(ComponentVerifierExtension.Scope.CONNECTIVITY,
                parameters);

        assertEquals(ComponentVerifierExtension.Result.Status.ERROR, result.getStatus());
    }
}
