/*
 * Copyright (c) 2019 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at:
 *
 *     https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
package org.eclipse.jkube.kit.build.api.helper;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.core.type.TypeReference;
import org.eclipse.jkube.kit.common.JKubeFileInterpolator;
import org.eclipse.jkube.kit.common.util.Serialization;
import org.eclipse.jkube.kit.config.image.build.BuildConfiguration;

import org.apache.commons.lang3.StringUtils;

import static org.eclipse.jkube.kit.common.util.EnvUtil.getEnv;
import static org.eclipse.jkube.kit.common.util.EnvUtil.getUserHome;

/**
 * Utility class for dealing with dockerfiles
 * @author roland
 */
public class DockerFileUtil {
    private static final String ARG_PATTERN_REGEX = "\\$([\\w|\\-\\.]+)|\\$\\{([\\w\\-|\\.]+)\\}";

    private static final Pattern argPattern = Pattern.compile(ARG_PATTERN_REGEX);

    private DockerFileUtil() {}

    /**
     * Extract the base images from a dockerfile. All lines containing a <code>FROM</code> is
     * taken.
     *
     * @param dockerFile file from where to extract the base image.
     * @param properties properties to interpolate in the provided dockerFile.
     * @param filter string representing filter parameters for properties in dockerfile
     * @param argsFromBuildConfig Docker Build args received from Build Configuration
     * @return LinkedList of base images name or empty collection if none is found.
     * @throws IOException if there's a problem while performing IO operations.
     */
    public static List<String> extractBaseImages(File dockerFile, Properties properties, String filter, Map<String, String> argsFromBuildConfig) throws IOException {
        List<String[]> fromLines = extractLines(dockerFile, "FROM", properties, resolveDockerfileFilter(filter));
        Map<String, String> args = extractArgs(dockerFile, properties, filter, argsFromBuildConfig);
        Set<String> result = new LinkedHashSet<>();
        Set<String> fromAlias = new HashSet<>();
        for (String[] fromLine :  fromLines) {
            if (fromLine.length == 2) { // FROM image:tag use case
                result.add(resolveImageTagFromArgs(fromLine[1], args));
            } else if (fromLine.length == 4) { // FROM image:tag AS alias use case
                if (!fromAlias.contains(fromLine[1])) {
                    result.add(resolveImageTagFromArgs(fromLine[1], args));
                }
                fromAlias.add(resolveImageTagFromArgs(fromLine[3], args));
            }
        }
        return new ArrayList<>(result);
    }

    /**
     * Extract Args from dockerfile. All lines containing ARG is taken.
     *
     * @param dockerfile Docker File
     * @param properties properties
     * @param filter interpolation filter
     * @param argsFromBuildConfig Docker build args from Build Configuration
     * @return HashMap of arguments or empty collection if none is found
     */
    public static Map<String, String> extractArgs(File dockerfile, Properties properties, String filter, Map<String, String> argsFromBuildConfig) throws IOException {
        return extractArgsFromLines(extractLines(dockerfile, "ARG", properties, filter), argsFromBuildConfig);
    }

    /**
     * Extract all lines containing the given keyword
     *
     * @param dockerFile dockerfile to examine.
     * @param keyword keyword to extract the lines for.
     * @param properties properties to interpolate in the provided dockerFile.
     * @param filter string representing filter parameters for properties in dockerfile
     * @return list of matched lines or an empty list.
     */
    public static List<String[]> extractLines(File dockerFile, String keyword, Properties properties, String filter) throws IOException {
        List<String[]> ret = new ArrayList<>();
        try (BufferedReader reader = new BufferedReader(new FileReader(dockerFile))) {
            String line;
            while ((line = reader.readLine()) != null) {
                String lineInterpolated = JKubeFileInterpolator.interpolate(line, properties, filter);
                String[] lineParts = lineInterpolated.split("\\s+");
                if (lineParts.length > 0 && lineParts[0].equalsIgnoreCase(keyword)) {
                    ret.add(lineParts);
                }
            }
        }
        return ret;
    }

    /**
     * Interpolate a docker file with the given properties and filter
     *
     * @param dockerFile docker file to interpolate.
     * @param properties properties to interpolate in the provided dockerFile.
     * @param filter filter for parsing properties from Dockerfile
     * @return The interpolated contents of the file.
     * @throws IOException if there's a problem while performing IO operations.
     */
    public static String interpolate(File dockerFile, Properties properties, String filter) throws IOException {
      return JKubeFileInterpolator.interpolate(dockerFile, properties, filter);
    }

    /**
     * Helper method for extractArgs(exposed for test)
     *
     * @param argLines list of string arrays containing lines with words
     * @param argsFromBuildConfig Docker build args from Build Configuration
     * @return map of parsed arguments
     */
    protected static Map<String, String> extractArgsFromLines(List<String[]> argLines, Map<String, String> argsFromBuildConfig) {
        Map<String, String> result = new HashMap<>();
        for (String[] argLine : argLines) {
            if (argLine.length > 1) {
                updateMapWithArgValue(result, argsFromBuildConfig, argLine[1]);
            }
        }
        return result;
    }

    static String resolveImageTagFromArgs(String imageTagString, Map<String, String> args) {
        String resolvedImageString = imageTagString;
        Set<String> foundArgs = findAllArgs(imageTagString);
        for (String foundArg : foundArgs) {
            if (args.containsKey(foundArg)) {
                resolvedImageString = resolvedImageString.replaceFirst(String.format("\\$\\{*%s\\}*", foundArg),
                    args.get(foundArg));
            }
        }
        return resolvedImageString;
    }

    static Set<String> findAllArgs(String imageTagString) {
        Matcher m = argPattern.matcher(imageTagString);
        Set<String> args = new HashSet<>();
        while(m.find()){
            if(m.group(1)!=null){
                args.add(m.group(1));
            }else if(m.group(2)!=null){
                args.add(m.group(2));
            }
        }
        return args;
    }

    public static Map<String, Object> readDockerConfig() {
        final String dockerConfigDirectory = getEnv("DOCKER_CONFIG");
        final File dockerConfig;
        if (StringUtils.isNotBlank(dockerConfigDirectory)) {
            dockerConfig = new File(dockerConfigDirectory,"config.json");
        } else {
            dockerConfig = new File(getUserHome(),".docker/config.json");
        }
        if (!dockerConfig.exists() || !dockerConfig.isFile()) {
            return Collections.emptyMap();
        }
        try {
            return Serialization.unmarshal(dockerConfig, new TypeReference<Map<String, Object>>() {});
        } catch (IOException e) {
            throw new IllegalStateException("Cannot read docker config from " + dockerConfig.getAbsolutePath(), e);
        }
    }

    private static void updateMapWithArgValue(Map<String, String> result, Map<String, String> args, String argString) {
        if (argString.contains("=") || argString.contains(":")) {
            String[] argStringParts = argString.split("[=:]");
            String argStringKey = argStringParts[0];
            String argStringValue = determineFinalArgValue(argString, argStringParts, args);
            if (argStringValue.startsWith("\"") || argStringValue.startsWith("'")) {
                // Replaces surrounding quotes
                argStringValue = argStringValue.replaceAll("(^[\"'])|([\"']$)", "");
            } else {
                validateArgValue(argStringValue);
            }
            result.put(argStringKey, argStringValue);
        } else {
            validateArgValue(argString);
            result.putAll(fetchArgsFromBuildConfiguration(argString, args));
        }
    }

    private static String determineFinalArgValue(String argString, String[] argStringParts, Map<String, String> args) {
        String argStringValue = argString.substring(argStringParts[0].length() + 1);
        if(args == null || args.get(argStringParts[0]) == null){
            return argStringValue;
        }
        return args.getOrDefault(argStringParts[0], argStringValue);
    }

    private static Map<String, String> fetchArgsFromBuildConfiguration(String argString, Map<String, String> args) {
        Map<String, String> argFromBuildConfig = new HashMap<>();
        if (args != null) {
            argFromBuildConfig.put(argString, args.getOrDefault(argString, ""));
        }
        return argFromBuildConfig;
    }

    private static void validateArgValue(String argStringParam) {
        String[] argStringParts = argStringParam.split("\\s+");
        if (argStringParts.length > 1) {
            throw new IllegalArgumentException("Dockerfile parse error: ARG requires exactly one argument. Provided : " + argStringParam);
        }
    }

    static String resolveDockerfileFilter(String filter) {
        return filter != null ? filter : BuildConfiguration.DEFAULT_FILTER;
    }
}
