/*
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 *
 * http://izpack.org/
 * http://izpack.codehaus.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.izforge.izpack.panels;

import com.izforge.izpack.gui.*;
import com.izforge.izpack.gui.patternfly.PatternflyButtonUI;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.InstallData;
import com.izforge.izpack.installer.InstallerFrame;
import com.izforge.izpack.installer.IzPanel;
import com.izforge.izpack.util.AbstractUIHandler;
import com.izforge.izpack.util.Debug;
import com.izforge.izpack.util.Log;
import com.izforge.izpack.util.VariableSubstitutor;

import javax.accessibility.AccessibleContext;
import javax.swing.*;
import javax.swing.filechooser.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;

/**
 * The finish panel class.
 *
 * @author Julien Ponge
 */
public class FinishPanel extends IzPanel implements ActionListener {

    private static final long serialVersionUID = 3257282535107998009L;

    // Localized string IDs
    private static final String FILE_EXISTS = "FinishPanel.file.already.exists";

    private static final String FINISH = "FinishPanel.";
    private static final String LINK = ".link";
    private static final String LABEL = ".label";
    private static final String SHOW = ".show";
    private static final String AUTO_INFORMATION = FINISH + "auto.information";
    private JButton autoButton;

    // Retrieval method for Localized strings
    private static String getTranslation(final InstallData idata, final String text) {
        return idata.langpack.getString(text);
    }

    /**
     * The variables substitutor.
     */
    protected VariableSubstitutor vs;

    /**
     * The constructor.
     *
     * @param parent The parent.
     * @param idata  The installation data.
     */
    public FinishPanel(InstallerFrame parent, InstallData idata) {
        super(parent, idata, new GridBagLayout());

        vs = new VariableSubstitutor(idata.getVariables());
    }

    /**
     * Indicates wether the panel has been validated or not.
     *
     * @return true if the panel has been validated.
     */
    public boolean isValidated() {
        return true;
    }

    /**
     * Called when the panel becomes active.
     * <p/>
     * 1) Declare that you want to show links on your finish panel
     * <variable name="FinishPanel.links.show" value="true"/>
     * 2) Declare link ids with comma seperated list
     * <variable name="FinishPanel.links.list" value="admin,gov,bpel"/>
     * 3) Choose a condition for your link to be checked against
     * <variable name="FinishPanel.gov.show" value="sramp.install" />
     * 4) Add link label and link text to eng.xml
     * <str id="FinishPanel.gov.label" txt="Governance Console:"/>
     * <str id="FinishPanel.gov.link" txt="http://localhost:8080/gwt-console-server"/>
     */
    public void panelActivate() {
        parent.addDoneButton();
        buildUI();
        Log.getInstance().informUser();
    }

    private void buildUI() {
        if (idata.installSuccess) {
            buildSuccessfulUI();
        } else {
            buildFailureUI();
        }
    }

    private void buildFailureUI() {
        addTitle(getFailureTitle());
        addContentPanel(getFailureContentPanel());
    }

    private JPanel getFailureContentPanel() {
        JPanel failurePanel = PanelFactory.createContentPanel(true);
        GridBagConstraints gbc = GridBagConstraintsFactory.getBasicConstraints();
        failurePanel.add(getLogPathMessage(), gbc);
        return failurePanel;
    }

    private JPanel getLogPathMessage() {
        JPanel logPathPanel = new JPanel(new GridBagLayout());
        String logFilePath = idata.getInstallPath() + File.separator + idata.getVariable("installation.logfile");
        File normalizedLogFilePath = new File(logFilePath);
        //TODO: localize
        logPathPanel.add(LabelFactory.create(parent.langpack.getString("See Installation Log at "
                        + normalizedLogFilePath.getAbsolutePath()),
                LEFT));
        return logPathPanel;
    }

    private void buildSuccessfulUI() {
        addTitle(getSuccessTitle());
        addContentPanel(getSuccessContentPanel());
    }

    private void addContentPanel(JPanel content) {
        add(content, GridBagConstraintsFactory.getContentPanelConstraints());
    }

    private JPanel getSuccessContentPanel() {
        JPanel successPanel = PanelFactory.createContentPanel(true);
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridy = 0;
        if (uninstallJarExists()) {
            gbc.weightx = 1;
            gbc.fill = GridBagConstraints.BOTH;
            successPanel.add(getUninstallerMessage(), gbc);
            gbc.gridy++;
        }
        if (showLinks()) {
            successPanel.add(getLinkPanel(), gbc);
            gbc.gridy++;
        }
        if (showAutoInformation()) {
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.weightx = 1.0;
            successPanel.add(createAutoInformationPanel(), gbc);
            gbc.gridy++;
        }
        gbc.weightx = gbc.weighty = gbc.fill = 0;
        successPanel.add(getAutoButton(), gbc);
        return successPanel;
    }



    private boolean showAutoInformation() {
        return idata.langpack.getString(AUTO_INFORMATION) != null;
    }

    private JPanel getAutoButton() {
        JPanel autoButtonPanel = new JPanel(new GridBagLayout());
        autoButton = ButtonFactory.createButton(parent.langpack.getString("FinishPanel.auto"));
        autoButton.setToolTipText(parent.langpack.getString("FinishPanel.auto.tip"));
        autoButton.addActionListener(this);
        autoButton.setActionCommand("Create auto.xml");
        AccessibleContext ac = autoButton.getAccessibleContext();
        ac.setAccessibleDescription("This JButton will create the auto.xml descriptor for future installations when pressed");
        parent.setFocus(autoButton);
        autoButtonPanel.add(autoButton, GridBagConstraintsFactory.getBasicConstraints());
        return autoButtonPanel;
    }

    private boolean uninstallJarExists() {
        return idata.uninstallOutJar != null;
    }

    private boolean showLinks() {
        boolean showLinks = true;
        String linkFlag = idata.getVariable("FinishPanel.links.show");
        if (linkFlag == null || !linkFlag.equals("true"))
            showLinks = false;
        return showLinks;
    }

    private void addTitle(JLabel title) {
        add(title, GridBagConstraintsFactory.getTitleConstraints(GridBagConstraints.NORTHWEST));
    }

    public String[] getLinkIds() {
        String[] linkIds = null;
        String availableLinks = idata.getVariable("FinishPanel.links.list"); //Get list of link ids, ids seperated with commas
        if (availableLinks != null) {
            linkIds = availableLinks.split(",");         //Get individual link ids
        }
        return linkIds;
    }

    private JPanel getLinkPanel() {
        JPanel linkPanel = new JPanel(new GridBagLayout());
        String[] linkIds = getLinkIds();
        if (showLinks() && linkIds != null) {
            linkPanel.add(getLinkTitle(), GridBagConstraintsFactory.getTitleConstraints(GridBagConstraints.NORTHWEST));
            boolean show = checkLinkConditions(linkIds);
            if (show) {
                GridBagConstraints gbc = new GridBagConstraints();
                gbc.gridy = 1;
                linkPanel.add(getLinksPanel(linkIds), gbc);
            }
        }
        return linkPanel;
    }

    private JPanel getLinksPanel(String[] linkIds) {
        JPanel linksPanel = new JPanel(new GridBagLayout());
        GridBagConstraints gbc = GridBagConstraintsFactory.getBasicConstraints();
        for (String linkId : linkIds) {
            String condId = FINISH + linkId + SHOW;
            boolean linkCond = idata.getRules().isConditionTrue(idata.getVariable(condId));
            if (linkCond) {
                gbc.gridx = 0;
                String storedValue = idata.langpack.getString(FINISH + linkId + LABEL);
                JLabel label = LabelFactory.create(storedValue);
                linksPanel.add(label, gbc);

                JLabel link = new JLabel(idata.langpack.getString(FINISH + linkId + LINK));
                gbc.gridx = 1;
                linksPanel.add(link, gbc);

                gbc.gridx = 2;
                gbc.weightx = 1;
                gbc.weighty = 1;
                linksPanel.add(Box.createGlue(), gbc);
                gbc.weightx = 0;
                gbc.weighty = 0;
            }
            gbc.gridy++;
        }
        return linksPanel;
    }

    private boolean checkLinkConditions(String[] linkIds) {
        for (String linkId : linkIds) {
            String condId = FINISH + linkId + SHOW;
            boolean linkCond = idata.getRules().isConditionTrue(idata.getVariable(condId));
            if (linkCond) {
                return true;
            }
        }
        return false;
    }

    public JLabel getLinkTitle() {
        JLabel linkTitle = LabelFactory.create(idata.langpack.getString("FinishPanel.link.header"), LEFT);
        linkTitle.setFont(FontResources.getOpenSansLight().deriveFont(16.0f));
        return linkTitle;
    }

    private JLabel getSuccessTitle() {
        JLabel successTitle = LabelFactory.createTitleLabel(idata.langpack.getString("FinishPanel.success"), false);
        return successTitle;
    }

    private JLabel getFailureTitle() {
        JLabel failureTitle = LabelFactory.createTitleLabel(idata.langpack.getString("FinishPanel.fail"), false);
        failureTitle.setForeground(Color.RED);
        return failureTitle;
    }

    private JPanel getUninstallerMessage() {
        JPanel uninstallerPanel = new JPanel(new GridBagLayout());
        GridBagConstraints gbc = GridBagConstraintsFactory.getBasicConstraints();
        String path = translatePath(idata.info.getUninstallerPath());
        JLabel uninstallerInfo = LabelFactory.create(parent.langpack.getString("FinishPanel.uninst.info"), LEFT);
        uninstallerPanel.add(uninstallerInfo, gbc);
        gbc.gridy++;

        JTextArea pathLabel = new JTextArea();
        pathLabel.setFocusable(false);
        pathLabel.setEditable(false);
        pathLabel.setBackground(null);
        pathLabel.setLineWrap(true);
        pathLabel.setWrapStyleWord(true);
        pathLabel.setText(path);

        gbc.weightx = gbc.weighty = 1;
        gbc.fill = GridBagConstraints.BOTH;
        uninstallerPanel.add(pathLabel, gbc);
        return uninstallerPanel;
    }

    private JPanel createAutoInformationPanel() {
        JPanel autoPanel = new JPanel(new GridBagLayout());
        GridBagConstraints gbc = GridBagConstraintsFactory.getBasicConstraints();
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1;
        String labelText = getTranslation(idata, AUTO_INFORMATION);
        JLabel autoInformationLabel = LabelFactory.create(String.format("<html><div style=\"width:500px\">%s</div></html>", labelText));
        autoPanel.add(autoInformationLabel, gbc);
        return autoPanel;
    }

    /**
     * Actions-handling method.
     *
     * @param e The event.
     */
    public void actionPerformed(ActionEvent e) {
        // Prepares the file chooser
        JFileChooser fc = new JFileChooser();
        loadFileChooserLang();
        fc.setCurrentDirectory(new File(idata.getInstallPath()));
        fc.setMultiSelectionEnabled(false);
        fc.setDialogTitle("Save");
        javax.swing.filechooser.FileFilter xmlfilter = new FileNameExtensionFilter("XML File (.xml)", "xml");
        fc.setFileFilter(xmlfilter);
        /**
         * removing this for our build
         */
        //fc.addChoosableFileFilter(new AutomatedInstallScriptFilter());
        // fc.setCurrentDirectory(new File("."));

        // Shows it
        try {
            if (fc.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
                // We handle the xml data writing
                File file = fc.getSelectedFile();

                if (fc.getFileFilter().equals(xmlfilter) && !file.getPath().endsWith(".xml")) {
                    file = new File(file.getPath() + ".xml");
                }
                // If the file exists, prompt for overwrite confirmation
                int res = 0;
                if (file.exists()) {
                    res = askQuestion(parent.langpack.getString("installer.warning"), getTranslation(idata, FILE_EXISTS), AbstractUIHandler.CHOICES_YES_NO, AbstractUIHandler.ANSWER_YES);
                }
                if (!file.exists() || (file.exists() && res == AbstractUIHandler.ANSWER_YES)) {
                    FileOutputStream out = new FileOutputStream(file);
                    BufferedOutputStream outBuff = new BufferedOutputStream(out, 5120);
                    parent.writeXMLTree(idata.xmlData, outBuff);
                    outBuff.flush();
                    outBuff.close();

                    file.setWritable(false, false);
                    file.setReadable(false, false);
                    file.setWritable(true, true);
                    file.setReadable(true, true);

                    autoButton.setEnabled(false);
                    String buttonSuccessText = parent.langpack.getString("FinishPanel.auto.generate.success");
                    if (buttonSuccessText != null && !buttonSuccessText.equals("FinishPanel.auto.generate.success")) {
                        autoButton.setText(buttonSuccessText);
                    }
                    outputVariableFile(file);
                }
            }
        } catch (Exception err) {
            Debug.trace(err);
            emitError(parent.langpack.getString("installer.error"), err.toString());
        }
    }

    protected static void outputVariableFile(File file) {
        // if autoPromptVars isn't empty, we need to write out an empty file containing the variable names for the user's reference
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        if (!idata.autoPromptVars.isEmpty()) {
            String propsFilePath = file.getPath() + ".variables";
            File propsFile = null;
            do {
                propsFile = new File(propsFilePath);
                if (propsFile.exists()) {
                    propsFilePath += "1";
                }
            } while (propsFile.exists());

            PrintWriter propsOut = null;
            try {
                propsOut = new PrintWriter(new FileWriter(propsFile));
            } catch (IOException e) {
                e.printStackTrace();
            }

            for (String variable : idata.autoPromptVars) {
                propsOut.println(variable + "=");
            }
            propsOut.close();
        }

    }

    /**
     * Translates a relative path to a local system path.
     *
     * @param destination The path to translate.
     * @return The translated path.
     */
    protected String translatePath(String destination) {
        // Parse for variables
        destination = vs.substitute(destination, null);

        // Convert the file separator characters
        return destination.replace('/', File.separatorChar);
    }

}