/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.ws;

import javax.wsdl.*;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.xml.namespace.QName;
import java.net.URL;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * WSDL helper class
 */
public final class WSDLParser
{

  private Definition wsdlDefinition;
  private int dfsDepth = 0;
  
  public WSDLParser(Definition wsdlDefinition)
  {
    this.wsdlDefinition = wsdlDefinition;
  }

  public void reset()
  {
    dfsDepth = 0;
  }

  public Operation getRPCOperation(QName service, String port, QName payloadName)
  {
    reset();
    Operation result = _getRPCOperation(this.wsdlDefinition, service, port, payloadName);
    if (result == null)
      throw new IllegalArgumentException("Unable to find operation for element "+payloadName);
    return result;
  }

  @SuppressWarnings("unchecked")
  private Operation _getRPCOperation(Definition wsdl, QName serviceName, String portName, QName payloadName)
  {
    if(++dfsDepth>50) // easier then retaining references
      throw new IllegalStateException("Recursive loop detected. DFS depth reached limit");

    // namespace / java.util.List of imports
    Collection<List<Import>> imports = wsdl.getImports().values();
    for(List<Import> importNS : imports)
    {
      for(Import wsdlImport : importNS)
      {
        Operation result = _getRPCOperation(wsdlImport.getDefinition(), serviceName, portName, payloadName);
        if(result!=null)
          return result;
      }
    }

    // resolve the port & operation
    Service service = wsdl.getService(serviceName);
    if (service == null)
      return null; // service might be defined elsewhere

    Port port = service.getPort(portName);
    if (port == null)
      throw new IllegalArgumentException("service "+serviceName+" contains no such port "+portName);

    Binding binding = port.getBinding();
    PortType portType = binding.getPortType();
    List<Operation> operations = portType.getOperations();
    for(Operation op : operations)
    {
        if(op.getName().equals(payloadName.getLocalPart()))
          return op;
    }

    return null;
  }

  public Operation getDocLitOperation(QName service, String port, QName payloadName)
  {
    reset();
    Operation result = _getDocLitOperation(this.wsdlDefinition, service, port, payloadName);
    if (result == null)
      throw new IllegalArgumentException("Unable to find operation for element "+payloadName);
    return result;
  }

  @SuppressWarnings("unchecked")
  private Operation _getDocLitOperation(Definition wsdl, QName serviceName, String portName, QName payloadName)
  {
    if(++dfsDepth>50) // easier then retaining references
      throw new IllegalStateException("Recursive loop detected. DFS depth reached limit");

    // namespace / java.util.List of imports
    Collection<List<Import>> imports = wsdl.getImports().values();
    for(List<Import> importNS : imports)
    {
      for(Import wsdlImport : importNS)
      {
        Operation result = _getDocLitOperation(wsdlImport.getDefinition(), serviceName, portName, payloadName);
        if(result!=null)
          return result;
      }
    }

    // resolve the port & operation
    Service service = wsdl.getService(serviceName);
    if (service == null)
      return null; // service might be defined elsewhere

    Port port = service.getPort(portName);
    if (port == null)
      throw new IllegalArgumentException("service "+serviceName+" contains no such port "+portName);

    Binding binding = port.getBinding();
    PortType portType = binding.getPortType();
    List<Operation> operations = portType.getOperations();
    for(Operation op : operations)
    {
      Message message = op.getInput().getMessage();
      Collection<Part> parts = message.getParts().values();
      for(Part part : parts)
      {
        if(part.getElementName().equals(payloadName))
          return op;
      }
    }

    return null;
  }

  public URL getServiceLocationURL(QName serviceQName, String portName)
  {
    reset();
    return _getServiceLocationURL(this.wsdlDefinition, serviceQName, portName);
  }

  public URL _getServiceLocationURL(Definition wsdl, QName serviceQName, String portName)
  {
    URL match = null;
    dfsDepth++;

    if(dfsDepth>50) // easier then retaining references
        throw new IllegalStateException("Recursive loop detected. DFS depth reached limit");

    // namespace / java.util.List of imports
    Map<String, List<Import>> imports = wsdl.getImports();
    for(String ns : imports.keySet())
    {
      List<Import> importNS = imports.get(ns);
      for(Import wsdlImport : importNS)
      {
        URL result = _getServiceLocationURL(wsdlImport.getDefinition(), serviceQName, portName);
        if(result!=null)
        {
          match = result;
          break;
        }
      }

      if(match!=null) break;
    }

    if(match!=null) // DFS results
        return match;

    try
    {
      Service service = wsdl.getService(serviceQName);
      Port port = null;
      SOAPAddress soapAddress = null;

      // --

      if(service!=null)
      {
        port = service.getPort(portName);
        if(port!=null)
        {
          for(Object obj : port.getExtensibilityElements())
          {
            if(obj instanceof SOAPAddress)
            {
              soapAddress = (SOAPAddress)obj;
            }
          }

        }
      }

      // --

      if(soapAddress!=null)
        match = new URL(soapAddress.getLocationURI());
    }
    catch (Exception e)
    {
      throw new RuntimeException("Failed to parse " + wsdl, e);
    }

    return match;
  }
}
