/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.ws;

import javassist.*;
import javassist.bytecode.AnnotationsAttribute;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstPool;
import javassist.bytecode.annotation.Annotation;
import javassist.bytecode.annotation.EnumMemberValue;
import javassist.bytecode.annotation.StringMemberValue;

import org.jboss.soa.dsp.EndpointMetaData;


/**
 * Creates JAX-WS Provider classes using javassist.
 * These provider classes can then be deployed to JBossWS in memory.<p>
 * The javassist generated class basically just carries the meta data,
 * while the actual implementation resides in {@link org.jboss.soa.dsp.ws.BaseWebServiceEndpoint}
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class WebServiceProviderGenerator
{
  private final static String PACKAGE = WebServiceProviderGenerator.class.getPackage().getName()+".generated";

  private CtClass m_provider=null;
  
  public BaseWebServiceEndpoint createProvider(
      EndpointMetaData metaData,
      WSDLReference wsdlRef,
      ClassLoader loader,
      String handlerFilePath,
      Class<? extends WebServiceProviderFactory> providerFactory
  )
      throws Exception
  {
    ClassPool pool = new ClassPool(true);
    pool.appendClassPath(new LoaderClassPath(loader));

    // Imports
    pool.importPackage("java.lang");
    pool.importPackage("javax.xml.ws");
    pool.importPackage("javax.jws");

    CtClass stringType = pool.get("java.lang.String");    
    
    String implClassName = PACKAGE+".WebServiceEndpoint_"+metaData.getEndpointId();
    //CtClass impl = pool.makeClass(implClassName);
    
    // Load an existing class representing the template for a Web Service provider
    // This was necessary, as javassist does not provide a way to set an implemented
    // interface that supports generics. Although the super class (AbstractWebServiceEndpoint)
    // implements this Provider<SOAPMessage> interface, CXF requires the actual web service
    // implementation class to directly define this interface. (RIFTSAW-123)
    CtClass impl = pool.get(org.jboss.soa.dsp.ws.TemplateWebServiceEndpoint.class.getName());
    impl.setName(implClassName);

    // AbstractWebServiceEndpoint.endpointId property
    CtField idField = new CtField(stringType, "endpointId", impl);
    idField.setModifiers(Modifier.PUBLIC);
    impl.addField(idField, "\""+metaData.getEndpointId()+"\"");

    // AbstractWebServiceEndpoint.serviceName property
    CtField serviceField = new CtField(stringType, "serviceName", impl);
    serviceField.setModifiers(Modifier.PUBLIC);
    impl.addField(serviceField, "\""+metaData.getServiceName().toString()+"\"");

     // AbstractWebServiceEndpoint.wsdlLocation property
    CtField wsdlLocationField = new CtField(stringType, "wsdlLocation", impl);
    wsdlLocationField.setModifiers(Modifier.PUBLIC);
    impl.addField(wsdlLocationField, "\""+wsdlRef.getWsdlURL().toExternalForm()+"\"");

     // AbstractWebServiceEndpoint.portName property
    CtField portNameField = new CtField(stringType, "portName", impl);
    portNameField.setModifiers(Modifier.PUBLIC);
    impl.addField(portNameField, "\""+metaData.getPortName()+"\"");

    // Annotations
    ClassFile classFile = impl.getClassFile();
    classFile.setVersionToJava5();
    ConstPool constantPool = classFile.getConstPool();

    /*
    @WebServiceProvider(
      portName="HelloPort",
      serviceName="HelloService",
      targetNamespace="http://helloservice.org/wsdl",
      wsdlLocation="WEB-INF/wsdl/HelloService.wsdl"
    )
    @ServiceMode(value=Service.Mode.MESSAGE)
     */
    AnnotationsAttribute attr = new
        AnnotationsAttribute(constantPool, AnnotationsAttribute.visibleTag);

    // --
    Annotation providerAnnotation = new Annotation(
        "javax.xml.ws.WebServiceProvider", constantPool);

    providerAnnotation.addMemberValue(
        "serviceName",
        new StringMemberValue(metaData.getServiceName().getLocalPart(), constantPool)
    );
    providerAnnotation.addMemberValue(
        "portName",
        new StringMemberValue(metaData.getPortName(), constantPool)
    );
    providerAnnotation.addMemberValue(
        "targetNamespace",
        new StringMemberValue(metaData.getServiceName().getNamespaceURI(), constantPool)
    );
    providerAnnotation.addMemberValue(
        "wsdlLocation",
        new StringMemberValue(wsdlRef.getWsdlURL().toExternalForm(), constantPool)
    );
    attr.addAnnotation(providerAnnotation);

    // --
    Annotation annotation2 = new Annotation("javax.xml.ws.ServiceMode",
        constantPool);
    EnumMemberValue enumValue = new EnumMemberValue(constantPool);
    enumValue.setType("javax.xml.ws.Service$Mode");
    enumValue.setValue("MESSAGE");
    annotation2.addMemberValue("value", enumValue);

    attr.addAnnotation(annotation2);

    // Specify the web service providerdetails
    Annotation detailsAnnotation = new Annotation(
            org.jboss.soa.dsp.ws.WebServiceDetails.class.getName(), constantPool);

    detailsAnnotation.addMemberValue("factory",
            new StringMemberValue(providerFactory.getName(), constantPool));

    attr.addAnnotation(detailsAnnotation);


    classFile.addAttribute(attr);
    
    
    // Check if handler chain should be established
    if (handlerFilePath != null) {
	    Annotation handlerChain = new Annotation("javax.jws.HandlerChain", constantPool);
	    	    
	    handlerChain.addMemberValue("file",
	    			new StringMemberValue(handlerFilePath, constantPool));
	    
	    attr.addAnnotation(handlerChain);
    }
    

    createStringGetter(impl, stringType, "endpointId", "getEndpointId");
    createStringGetter(impl, stringType, "serviceName", "getServiceName");
    createStringGetter(impl, stringType, "wsdlLocation", "getWsdlLocation");
    createStringGetter(impl, stringType, "portName", "getPortName");

    postProcess(classFile, impl, attr);
    
    // ------------

    // freeze
    //impl.stopPruning(false);
    impl.toClass(loader);
    JavaUtils.clearBlacklists(loader);

    // test it
    Class clazz = loader.loadClass(implClassName);
    
    BaseWebServiceEndpoint obj = (BaseWebServiceEndpoint)clazz.newInstance();

    m_provider = impl;

    return obj;
  }
  
  /**
   * This method performs optional post processing of the generated
   * class.
   * 
   * @param classFile The class file
   * @param impl The class
   * @param attr Annotations attribute
   */
  protected void postProcess(ClassFile classFile, CtClass impl, AnnotationsAttribute attr) {
  }
  
  public void writeClass(java.io.File war) throws Exception {
	    
		String dir=war.getAbsolutePath()+java.io.File.separatorChar+"WEB-INF"+
					java.io.File.separatorChar+"classes";

		m_provider.writeFile(dir);
  }

  private void createStringGetter(CtClass impl, CtClass stringType, String property, String methodName)
      throws Exception
  {
    CtMethod method = new CtMethod(
        stringType, methodName,
        new CtClass[]{},
        impl
    );

    // Method body
    StringBuffer body = new StringBuffer();
    body.append("{");
    body.append("return this."+property+";");
    body.append("}");

    method.setBody(body.toString());
    
    impl.addMethod(method);

  }
}
