/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers;

import java.io.IOException;
import java.util.List;
import java.util.Properties;

import javax.jms.DeliveryMode;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.naming.Context;
import javax.naming.NamingException;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.addressing.eprs.DefaultJmsReplyToEpr;
import org.jboss.internal.soa.esb.couriers.helpers.JmsComposer;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionFailureException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsSession;
import org.jboss.internal.soa.esb.util.MessageFlowContext;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.couriers.CourierTransportException;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierServiceBindException;
import org.jboss.soa.esb.couriers.CourierMarshalUnmarshalException;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.listeners.gateway.DefaultESBPropertiesSetter;
import org.jboss.soa.esb.listeners.gateway.ESBPropertiesSetter;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.notification.jms.DefaultJMSPropertiesSetter;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;
import org.jboss.soa.esb.util.Util;

public class JmsCourier implements PickUpOnlyCourier, DeliverOnlyCourier {
	
    /**
     * package protected constructor - Objects of Courier should only be
     * instantiated by the Factory
     *
     * @param epr
     */
    public JmsCourier(JMSEpr epr) throws CourierException {
        this(epr, false);
    }

    /**
     * package protected constructor - Objects of Courier should only be
     * instantiated by the Factory
     *
     * @param epr
     */
    public JmsCourier(JMSEpr epr, boolean isReceiver) throws CourierException {
        _isReceiver = isReceiver;
        _epr = epr;
        _sleepForRetries = 3000;

        if (!_isReceiver) {
            try {
                _messageProperties = Util.propertiesFromSelector(_epr
                        .getMessageSelector());
            }
            catch (Exception e) {
                throw new CourierException(e);
            }
        }
        
        if (_epr.getAcknowledgeMode() == javax.jms.Session.CLIENT_ACKNOWLEDGE) {
        	throw new CourierException("CLIENT_ACKNOWLEDGE mode is not valid for a non-gateway listener using jms-provider");
        }
    } // ________________________________

    public void cleanup() {
        synchronized(this) {
            try {
                if (_messageProducer != null) {
                    try {
                        _messageProducer.close();
                    } catch (Exception e) {
                        _logger.debug(e.getMessage(), e);
                    } finally {
                        _messageProducer = null;
                    }
                }

                if (_messageConsumer != null) {
                    try {
                       _messageConsumer.close();
                    } catch (JMSException e) {
                        _logger.debug(e.getMessage(), e);
                    } finally {
                        _messageConsumer = null;
                    }
                }
            } finally {
                closeSession() ;
            }
        }
    } // ________________________________

    private synchronized void closeSession() {
        if (jmsSession != null) {
            try {
                getConnectionPool().closeSession(jmsSession);
            } catch (ConnectionException e) {
                _logger.error("Unable to get Connection Pool for closing of JMS Session.", e);
            } finally {
                jmsSession = null;
            }
        }
    }

    private synchronized void releaseSession() {
        if (jmsSession != null) {
            try {
                getConnectionPool().releaseSession(jmsSession);
            } catch (ConnectionException e) {
                _logger.error("Unable to get Connection Pool for releasing of JMS Session.", e);
            } finally {
                jmsSession = null;
            }
        }
    }
    
    public Session getJmsSession() throws CourierException {
    	return getJmsSession(Session.AUTO_ACKNOWLEDGE);
    }

    public synchronized Session getJmsSession(final int acknowledgeMode) throws CourierException {
        if(jmsSession == null) {
            try {
                jmsSession = getConnectionPool().getSession(acknowledgeMode);
            } catch (NamingException e) {
                throw new CourierServiceBindException("Failed to get JMS Session from pool.", e);
            } catch (JMSException e) {
                throw new CourierServiceBindException("Failed to get JMS Session from pool.", e);
            } catch (ConnectionException e) {
                throw new CourierServiceBindException("Failed to get JMS Session from pool.", e);
            }
        }
        return jmsSession;
    }

    /**
     * package the ESB message in a javax.jms.ObjectMessage, and send it
     *
     * @param message Message - the message to deliverAsync
     * @return boolean - the result of the delivery
     * @throws CourierException -
     *                          if problems were encountered
     */
    public boolean deliver(org.jboss.soa.esb.message.Message message) throws CourierException {
        try {
            return internalDeliver(message) ;
        } catch (final JmsConnectionFailureException jcfe) {
            // fall through
        } catch (final IllegalStateException ise) {
            // fall through
        }
        
        cleanup() ;
        
        try {
            return internalDeliver(message) ;
        } catch (final JmsConnectionFailureException jcfe) {
            throw new CourierTransportException("Caught exception during delivery and could not reconnect! ", jcfe);
        } catch (final IllegalStateException ise) {
            throw new CourierTransportException("Caught exception during delivery and could not reconnect! ", ise);
        }
    }
    
    private boolean internalDeliver(org.jboss.soa.esb.message.Message message) throws CourierException, JmsConnectionFailureException, IllegalStateException {
        ObjectMessage msg;

        if (null == message) {
            return false;
        }
        
        synchronized(this) {
            if (_messageProducer == null) {
                try {
                    createMessageProducer();
                } catch (final NamingContextException nce) {
                    throw new CourierServiceBindException("Unexpected exception attempting to access naming context pool", nce) ;
                }
            }
        }

        // Create the JMS message from the serialized ESB message...
        try {
            msg = getJmsSession(_epr.getAcknowledgeMode()).createObjectMessage(Util.serialize(message));
        } catch (JMSException e) {
            throw new CourierMarshalUnmarshalException("Failed to serialize ESB Message.", e);
        } catch (ParserConfigurationException e) {
            throw new CourierMarshalUnmarshalException("Failed to serialize ESB Message.", e);
        } catch (IOException e) {
            throw new CourierMarshalUnmarshalException("Failed to serialize ESB Message.", e);
        }

        // Set the JMS message from the ESB message...
        try {
            setJMSProperties(message, msg);
        } catch (final JmsConnectionFailureException jcfe) {
            throw jcfe ;
        } catch (final IllegalStateException ise) {
            throw ise ;
        } catch (JMSException e) {
            throw new CourierMarshalUnmarshalException("Failed to set JMS Message properties from ESB Message properties.", e);
        }

        return internalDeliver(msg);
    }

    /**
     * Send the JMS message.
     *
     * @param message Message - the message to deliverAsync
     * @return boolean - the result of the delivery
     * @throws CourierException -
     *                          if problems were encountered
     */
    public boolean deliver(javax.jms.Message message) throws CourierException {
        try {
            return internalDeliver(message) ;
        } catch (final JmsConnectionFailureException jcfe) {
            // fall through
        } catch (final IllegalStateException ise) {
            // fall through
        }
        
        cleanup() ;
        
        try {
            return internalDeliver(message) ;
        } catch (final JmsConnectionFailureException jcfe) {
            throw new CourierTransportException("Caught exception during delivery and could not reconnect! ", jcfe);
        } catch (final IllegalStateException ise) {
            throw new CourierTransportException("Caught exception during delivery and could not reconnect! ", ise);
        }
    }
    
    private boolean internalDeliver(javax.jms.Message message) throws CourierException, JmsConnectionFailureException, IllegalStateException {
        if (_isReceiver) {
            throw new CourierException("This is a read-only Courier");
        }

        if (null == message) {
            return false;
        }
        synchronized(this) {
            if (_messageProducer == null) {
                try {
                    createMessageProducer();
                } catch (final NamingContextException nce) {
                    throw new CourierServiceBindException("Unexpected exception attempting to access naming context pool", nce) ;
                }
            }
        }

        synchronized(this) {
            if (_messageProducer != null) {
                try {
                    for (KeyValuePair kvp : _messageProperties) {
                        String key = kvp.getKey();
                        if(message.getStringProperty(key) == null) {
                            message.setStringProperty(key, kvp.getValue());
                        }
                    }
                    final Integer priority = MessageFlowContext.getMessageFlowPriority();
                    _messageProducer.setPriority(priority == null ? javax.jms.Message.DEFAULT_PRIORITY : priority.intValue()) ;
                    if ((priority != null) && _logger.isDebugEnabled())
                    {
                        _logger.debug("Producer initialised with priority " + priority) ;
                    }
                } catch (final JMSException e) {
                    throw new CourierTransportException("Caught exception initialising properties! ",e);
                }
                    
                try {
                    _messageProducer.send(message);
                    
                    return true;
                }
                catch (final JmsConnectionFailureException jcfe) {
                    throw jcfe ;
                }
                catch (final IllegalStateException ise) {
                    throw ise ;
                }
                catch (JMSException e) {
                    if (!jmsConnectRetry(e))
                        throw new CourierTransportException("Caught exception during delivery and could not reconnect! ",e);
                    try {
                        _messageProducer.send(message);
                        
                        return true;
                    } catch (final JMSException e2) {
                        throw new CourierTransportException("Caught exception during delivery having successfully recovered! ",e2);
                    } catch (Exception e2) {
                        throw new CourierException(e2);
                    }
                }
                catch (Exception e) {
                    throw new CourierException(e);
                }
            }
        }
        return false;
    } // ________________________________


    private boolean jmsConnectRetry(Exception exc) {
        _logger.info("JMS error.  Attempting JMS reconnect.", exc);

        synchronized(this) {
            try {
                if ((jmsSession != null) && jmsSession.getTransacted()) {
                    jmsSession.rollback() ;
                    return false ;
                } else {
                    cleanup() ;
                }
            } catch (final JMSException jmse) {
                releaseSession() ;
                return false ;
            }
        }

        final int maxRetry = 5; // TODO Magic number here!!!
        for (int i1 = 0; i1 < maxRetry; i1++) {
            // try to reconnect to the queue
            try {
                synchronized(this) {
                    if (_isReceiver) createMessageConsumer();
                    else
                        createMessageProducer();
                    break;
                }
            }
            catch (Exception e) {
                if (i1 < maxRetry - 1) {
                    try {
                        Thread.sleep(_sleepForRetries);
                    }
                    catch (InterruptedException e1) { // Just return after logging
                        _logger.debug("Unexpected thread interupt exception.", e);
                        break;
                    }
                } else {
                    _logger.debug("Failed to reconnect to JMS", e);
                    
                    return false;
                }
            }
        }
        
        return true;
    } // ________________________________

    private void createMessageProducer() throws CourierException, NamingContextException, JmsConnectionFailureException, IllegalStateException {
        synchronized(this) {
            if (_messageProducer == null) {
                try {
                    final Session session = getJmsSession(_epr.getAcknowledgeMode());
                    Destination destination = null ;
                    final String destinationName = _epr.getDestinationName() ;
                    Context oJndiCtx = NamingContextPool.getNamingContext(_epr.getJndiEnvironment());
                    try {
                        String sType = _epr.getDestinationType();
                        if (JMSEpr.QUEUE_TYPE.equals(sType)) {
                            try {
                                destination = (Destination) oJndiCtx.lookup(destinationName);
                            } catch (NamingException ne) {
                                try {
                                    oJndiCtx = NamingContextPool.replaceNamingContext(oJndiCtx, _epr.getJndiEnvironment());
                                    destination = (Destination) oJndiCtx.lookup(destinationName);
                                } catch (NamingException nex) {
                                    //ActiveMQ
                                    destination = session.createQueue(destinationName);
                                }
                            }
                        } else if (JMSEpr.TOPIC_TYPE.equals(sType)) {
                            try {
                                destination = (Destination) oJndiCtx.lookup(destinationName);
                            }
                            catch (NamingException ne) {
                                destination = session.createTopic(destinationName);
                            }
                        } else {
                            throw new CourierException("Unknown destination type");
                        }
                        _messageProducer = session.createProducer(destination);
                        _messageProducer.setDeliveryMode(_epr.getPersistent()?DeliveryMode.PERSISTENT:DeliveryMode.NON_PERSISTENT);
                        if ( _logger.isDebugEnabled() )
                            _logger.debug("JMSCourier deliveryMode: " + _messageProducer.getDeliveryMode() + ", peristent:" + _epr.getPersistent());
                    } finally {
                        if (oJndiCtx != null) {
                            NamingContextPool.releaseNamingContext(oJndiCtx) ;
                        }
                    }
                }
                catch (final JmsConnectionFailureException jcfe) {
                    throw jcfe ;
                }
                catch (final IllegalStateException ise) {
                    throw ise ;
                }
                catch (JMSException ex) {
                    _logger.debug("Error from JMS system.", ex);

                    throw new CourierException(ex);
                }
            }
        }
    } // ________________________________

    private JmsConnectionPool getConnectionPool() throws ConnectionException {
        synchronized(this) {
            if(jmsConnectionPool == null) {
                String sFactoryClass = _epr.getConnectionFactory();
                Properties properties = _epr.getJndiEnvironment();
                String username = _epr.getJMSSecurityPrincipal();
                String password = _epr.getJMSSecurityCredential();

                if (Util.isNullString(sFactoryClass)) {
                    sFactoryClass = "ConnectionFactory";
                }

                jmsConnectionPool = JmsConnectionPoolContainer.getPool(properties, sFactoryClass, username, password);
            }

            return jmsConnectionPool;
        }
    }
    
    public Message pickup(long millis) throws CourierException, CourierTimeoutException {
        javax.jms.Message jmsMessage = pickupPayload(millis);

        return JmsComposer.compose(jmsMessage, esbPropertiesStrategy) ;
    }

    public javax.jms.Message pickupPayload(long millis) throws CourierException, CourierTimeoutException {
        try {
            return internalPickupPayload(millis) ;
        } catch (final JmsConnectionFailureException jcfe) {
            // fall through
        } catch (final IllegalStateException ise) {
            // fall through
        }
        
        cleanup() ;
        
        try {
            return internalPickupPayload(millis) ;
        } catch (final JmsConnectionFailureException jcfe) {
            throw new CourierTransportException("Caught exception during receive and could not reconnect! ", jcfe);
        } catch (final IllegalStateException ise) {
            throw new CourierTransportException("Caught exception during receive and could not reconnect! ", ise);
        }
    }
    
    private javax.jms.Message internalPickupPayload(long millis) throws CourierException, CourierTimeoutException, JmsConnectionFailureException, IllegalStateException {
        if (!_isReceiver)
            throw new CourierException("This is an outgoing-only Courier");
        if (millis < 1)
            throw new IllegalArgumentException("Timeout millis must be > 0");
            try {
                createMessageConsumer();
            }
        catch (Exception e) {
            try {
                Thread.sleep(1000); // TODO magic number
            }
            catch (InterruptedException eI) {/* OK do nothing */
            }
            throw new CourierException("Unable to create Message Consumer", e);
        }

        javax.jms.Message jmsMessage = null;
        synchronized(this) {
            if (null != _messageConsumer) {
                try {
                    jmsMessage = _messageConsumer.receive(millis);
                }
                catch (final JmsConnectionFailureException jcfe) {
                    throw jcfe ;
                }
                catch (final IllegalStateException ise) {
                    throw ise ;
                }
                catch (JMSException e) {
                    if (!jmsConnectRetry(e))
                        throw new CourierTransportException("Caught exception during receive and could not reconnect! ",e);
                    try {
                        jmsMessage = _messageConsumer.receive(millis);
                    } catch (JMSException e2) {
                        throw new CourierTransportException("Caught exception during delivery having successfully recovered! ",e2);
                    } catch (Exception e2) {
                        throw new CourierTransportException(e2);
                    }
                }
                catch (Exception e) {
                    throw new CourierTransportException(e);
                }
            }
        }
        return jmsMessage;
    } // ________________________________
    
    /**
     * Sets the strategy for handling the setting of properties on an outgoing
     * JMS Message.
     *
     * @param jmsPropertiesStrategy the strategy to use.
     */
    public void setJmsPropertiesStrategy(JMSPropertiesSetter jmsPropertiesStrategy) {
        this.jmsPropertiesStrategy = jmsPropertiesStrategy;
    }

    /**
     * Set the {@link ESBPropertiesSetter} to be used
     *
     * @param esbPropertiesStrategy the strategy to be used
     */
    public void setEsbPropertiesStrategy(ESBPropertiesSetter esbPropertiesStrategy) {
        this.esbPropertiesStrategy = esbPropertiesStrategy;
    }

    /**
     * This method will set appropriate JMSProperties on the outgoing JMSMessage object.
     * </p>
     * Sublclasses can either override this method add a different behaviour, or they can
     * set the strategy by calling {@link #setJmsPropertiesStrategy(JMSPropertiesSetter)}.
     * </p>
     * See {@link org.jboss.soa.esb.notification.jms.JMSPropertiesSetter} for more info.
     */
    protected void setJMSProperties(org.jboss.soa.esb.message.Message fromESBMessage, javax.jms.Message toJMSMessage) throws JMSException {
        jmsPropertiesStrategy.setJMSProperties(fromESBMessage, toJMSMessage);
    }

    /**
     * Delegates to {@link DefaultESBPropertiesSetter#setPropertiesFromJMSMessage(javax.jms.Message,Message)}
     * by default, but this method can be overridden by subclasses that need a different behaviour.
     * </p>
     * It is also possible to set a different strategy by setting {@link #setEsbPropertiesStrategy(ESBPropertiesSetter)}
     */
    protected void setPropertiesFromJMSMessage(javax.jms.Message fromJMS, org.jboss.soa.esb.message.Message toESB) throws JMSException {
        esbPropertiesStrategy.setPropertiesFromJMSMessage(fromJMS, toESB);
    }

    private void createMessageConsumer() throws CourierException, ConfigurationException, MalformedEPRException, NamingContextException, JmsConnectionFailureException, IllegalStateException {
        Context oJndiCtx = null;

        synchronized(this) {
            if (_messageConsumer == null) {
                boolean success = false;
                try {
                    Properties environment = _epr.getJndiEnvironment();
                    final Session session = getJmsSession(_epr.getAcknowledgeMode());
                    Destination destination = null ;
                    final String destinationName = _epr.getDestinationName() ;
                    oJndiCtx = NamingContextPool.getNamingContext(environment);
                    try
                    {
                        String sType = _epr.getDestinationType();
                        if (JMSEpr.QUEUE_TYPE.equals(sType)) {
                            try {
                                destination = (Destination) oJndiCtx.lookup(destinationName);
                            } catch (NamingException ne) {
                                try {
                                    oJndiCtx = NamingContextPool.replaceNamingContext(oJndiCtx, environment);
                                    destination = (Destination) oJndiCtx.lookup(destinationName);
                                } catch (NamingException nex) {
                                    //ActiveMQ
                                    destination = session.createQueue(destinationName);
                                }
                            }
                        } else if (JMSEpr.TOPIC_TYPE.equals(sType)) {
                             try {
                                   destination = (Destination) oJndiCtx.lookup(destinationName);
                             }
                             catch (NamingException ne) {
                                   destination = session.createTopic(destinationName);
                             }
                        } else {
                            throw new CourierException("Unknown destination type");
                        }
                        if (destination == null) {
                            throw new CourierException("Could not locate destination: " + destinationName);
                        }
                        
                        String durableSubscriptionName = _epr.getDurableSubscriptionName();
                        if(durableSubscriptionName != null && destination instanceof Topic) {
                        	_messageConsumer = session.createDurableSubscriber((Topic)destination, durableSubscriptionName, _epr.getMessageSelector(), false);
                        } else {
                            _messageConsumer = session.createConsumer(destination, _epr.getMessageSelector());
                        }
                        
                        success = true;
                    } finally {
                        NamingContextPool.releaseNamingContext(oJndiCtx) ;
                    }
                }
                catch (final JmsConnectionFailureException jcfe) {
                    throw jcfe ;
                }
                catch (final IllegalStateException ise) {
                    throw ise ;
                }
                catch (JMSException ex) {
                    _logger.debug("Error from JMS system.", ex);

                    throw new CourierException(ex);
                }
                finally {
                    if (!success) {
                        closeSession();
                    }
                }
            }
        }
    } // ________________________________

    long _sleepForRetries = 3000; // milliseconds

    protected boolean _isReceiver;

    protected JMSEpr _epr;

    protected Logger _logger = Logger.getLogger(JmsCourier.class);

    protected String _messageSelector;

    protected JmsSession jmsSession;

    protected MessageProducer _messageProducer;

    protected MessageConsumer _messageConsumer;

    protected List<KeyValuePair> _messageProperties;

    protected JmsConnectionPool jmsConnectionPool;
    
    /**
     * Strategy for setting JMSProperties
     */
    private JMSPropertiesSetter jmsPropertiesStrategy = new DefaultJMSPropertiesSetter();
    
    /**
     * Filter pattern for incoming property keys. These are property key names that should not be
     * extracted from the JMS Message object into the ESB Message object.
     */
    private static final String INPUT_PROPERTIES_FILTER = DefaultJmsReplyToEpr.REPLY_UUID_TAG;
    
    /**
     * Strategy for setting JMS Properties on the ESB Message object created
     * by the process method.
     */
    private ESBPropertiesSetter esbPropertiesStrategy = new DefaultESBPropertiesSetter(INPUT_PROPERTIES_FILTER);

}
