/*
 * JBoss, Home of Professional Open Source
 * Copyright 2012, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2012
 */

package org.jboss.soa.esb.actions;

import org.apache.commons.lang.builder.ReflectionToStringBuilder;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.actions.AbstractActionLifecycle;
import org.jboss.soa.esb.actions.annotation.Process;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;

/**
 * Simple action that logs custom text and possibly the message to a logger using the 
 * "<jbossesb-common-string>.<deployment-name>.<service-category>.<service-name>" appender.    This action differs from LogAction
 * in that it allows you to set LogLevelAction on a per service basis, rather
 * than on a per-action basis, and that it allows you to log custom text.
 * Debug level setting will result in the ESB message being output.
 * Trace level setting will result in the output of the message payload
 * 
 * Here is an example of how to use this within your jboss-esb.xml :
 * <p/>
 * <code>
 * &lt;action name="ServiceLoggerAction"
 * 		class="org.jboss.soa.esb.actions.ServiceLoggerAction"&gt;<br/>
 * &nbsp;&nbsp;&nbsp;&lt;property name="text" value="Reached here"/&gt;<br/>
 * &nbsp;&nbsp;&nbsp;&lt;property name="log-payload-location" value="true"/&gt;<br/>
 * &lt;/action&gt;<br/>
 * </code>
 * <p/>
 * 
 * jboss-log4j.xml:
 * When configuring the jboss-log4j.xml, use the pattern :
 * JBOSSESB_COMMON_STRING.DEPLOYMENT_NAME.SERVICE_CATEGORY.SERVICE_NAME
 * 
 * This allows you to log all instances of this action (by just using the JBOSSESB_COMMON_STRING),
 * log all instances within a deployment (JBOSSESB_COMMON_STRING.DEPLOYMENT_NAME), by service category name,
 * and by service name.
 * 
 * <code>
 * &lt;category name="jbossesb.helloworld.esb.FirstServiceESB.SimpleListener"&gt;<br/>
 *     &lt;priority value="TRACE"/&gt;<br/>
 * &lt;/category&gt;<br/>
 * </code>
 * 
 * 
 * Property description:
 * <ul>
 * <li><i>text</i>  Custom text to be output, defaults to action name
 * <li><i>log-payload-location</i> True or False value which specifies
 * whether the payload location should be logged in Trace 
 * </ul>         
 * 
 * 
 * @author <a href="mailto:noconnor@redhat.com">noconnor@redhat.com</a>
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 */
public class ServiceLoggerAction extends AbstractActionLifecycle {
	public static final String TEXT_TAG = "text";
	public static final String LOG_PAYLOAD_LOCATION_TAG = "log-payload-location";
	public static final String COMMON_LOG4J_CATEGORY_STRING = "jbossesb";
	
	protected ConfigTree _config;
	private Logger myLogger;
	private MessagePayloadProxy payloadProxy;

	private boolean logPayloadLocation;
	
	private String currCategoryName;
	private String currServiceName;
	private String currActionName;
	private String currDeploymentName;
	private String text;

	
	public ServiceLoggerAction() {	
		myLogger = Logger.getLogger(this.getClass());
	}
	
	public boolean isLogPayloadLocation() {
		return logPayloadLocation;
	}

	public void setLogPayloadLocation(boolean logPayloadLocation) {
		this.logPayloadLocation = logPayloadLocation;
	}

	public String getText() {
		return text;
	}

	public void setText(String text) {
		this.text = text;
	}

	public Logger getLogger() {
		return myLogger;
	}
	
	public ServiceLoggerAction(ConfigTree configTree) {
		currDeploymentName = configTree.getParent().getParent().getAttribute(ListenerTagNames.DEPLOYMENT_NAME_TAG);
		
		myLogger = Logger.getLogger(
				COMMON_LOG4J_CATEGORY_STRING + "."
				+ currDeploymentName + "."
				+ configTree.getParent().getAttribute(
				ListenerTagNames.SERVICE_CATEGORY_NAME_TAG) + "."
				+ configTree.getParent().getAttribute(ListenerTagNames.SERVICE_NAME_TAG));
		
		currCategoryName = configTree.getParent().getAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG);
		currServiceName = configTree.getParent().getAttribute(ListenerTagNames.SERVICE_NAME_TAG);

		logPayloadLocation = "true".equals(configTree.getAttribute(LOG_PAYLOAD_LOCATION_TAG));
		text = configTree.getAttribute(TEXT_TAG, currCategoryName + "." + currServiceName);

		payloadProxy = new MessagePayloadProxy(configTree);
	}


	@Process
	public Message process(Message message) {
		myLogger.info(text);
		if (message != null) {
			if (myLogger.isDebugEnabled()) {
				myLogger.debug(message.toString());
			}
			if (myLogger.isTraceEnabled()) {
				try {
					if (logPayloadLocation) {
						String payloadLocation = payloadProxy.getGetPayloadLocation();
						myLogger.trace(payloadLocation);						
					} 
					
					Object msgObj = payloadProxy.getPayload(message);
					myLogger.trace(ReflectionToStringBuilder.toString(msgObj));
					
				} catch (MessageDeliverException e) {
					myLogger.trace("Exception thrown during trace prepare"+e.getMessage());
				}
			}
		}
		return message;
	}
}
