/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
package org.jboss.soa.esb.common;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;

import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfig;

/**
 * This MBean wraps the configuration to allow ServiceBindingManager
 * to alter the configuration.
 */
public class JBossESBPropertyService extends ServiceMBeanSupport implements JBossESBPropertyServiceMBean
{
    /**
     * The name of the ESB property file.
     */
    private static final String ESB_PROPERTY_FILE = "jbossesb-properties.xml";
    
    /**
     * The current property file name.
     */
    private String propertyFile ;

    private String propertiesFileDir;
    
    /**
     * Get the property file.
     * @return The name of the property file being used.
     */
    public String getPropertyFile()
    {
        return propertyFile ;
    }
    
    /**
     * Set the property file.
     * @param propertyFile The name of the property file being used.
     */
    public void setPropertyFile(final String propertyFile)
    {
        this.propertyFile = propertyFile ;
    }
    
    /**
     * @deprecated Use {@link Configuration#getBindAddress()}.
     */
    @Deprecated
    public static String getBindAddress()
    {
        return Configuration.getBindAddress() ;
    }
    
    private static String fixRemoteAddress(final String address)
    {
        try
        {
            if (address == null || "0.0.0.0".equals(address))
                return InetAddress.getLocalHost().getHostName();
        }
        catch (UnknownHostException ignored)
        {
        }

        return address;
    }
    
    /**
     * Create the property service.
     */
    @Override
    protected void createService()
        throws Exception
    {
        String esbBindAddress = System.getProperty(Environment.ESB_BIND_ADDRESS) ;
        if (esbBindAddress == null)
        {
            String serverBindAddress = System.getProperty(ServerConfig.SERVER_BIND_ADDRESS) ;
            esbBindAddress = fixRemoteAddress(serverBindAddress) ;
            System.setProperty(Environment.ESB_BIND_ADDRESS, esbBindAddress) ;
        }
        
        if (propertyFile != null)
        {
            final File baseFile = new File(this.propertyFile) ;
            final InputStream xmlPropertyIS ;
            if (!baseFile.isAbsolute()) {
                final URL resourceURL = Thread.currentThread().getContextClassLoader().getResource(propertyFile) ;
                xmlPropertyIS = resourceURL.openStream() ;
            } else {
                xmlPropertyIS = new FileInputStream(baseFile) ;
            }
            
            try {
                if (propertiesFileDir == null) {
                    final String errorMsg = String.format("No property named '%s' was configured in jbossesb.sar/META-INF/jboss-service.xml for %s", "propertiesFileDir", getClass().getName());
                    throw new IllegalArgumentException(errorMsg);
                }
                File dataDir = new File(propertiesFileDir);
                if (!dataDir.exists()) {
                    final String errorMsg = String.format("The directory configured for %s='%s' does not exist.", "warFilesDir", dataDir);
                    throw new FileNotFoundException(errorMsg);
                }
                final File esbPropertyFile = new File(dataDir, ESB_PROPERTY_FILE);
                
                final FileOutputStream esbPropertyOS = new FileOutputStream(esbPropertyFile) ;
                try {
                    XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(xmlPropertyIS),
                        XMLHelper.getXMLStreamWriter(esbPropertyOS)) ;
                } finally {
                    esbPropertyOS.close() ;
                }
                
                System.setProperty(Environment.PROPERTIES_FILE, "abs://" + esbPropertyFile.getAbsolutePath());
            } finally {
                xmlPropertyIS.close() ;
            }
        }
    }
    
    public void setPropertiesFileDir(String directory)
    {
        this.propertiesFileDir = directory;
    }
}
