/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Enumeration;
import java.util.regex.Pattern;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;

/**
 * Default impl of ESBPropertiesSetter.
 * </p>
 * The following JMS Header fields are set on the ESB Message instance:
 * <lu>
 * <li> JMSCorrelationID	-> Call.setRelatesTo	URI = jms:correlationID#YYXX-123456780-GG
 * <li> JMSReplyTo			-> Properties( key = JMSPropertiesSetter.JMS_REPLY_TO )
 * <li> JMSExpiration		-> Properties( key = JMSPropertiesSetter.JMS_EXPIRATION )
 * <li> JMSRedeliverd		-> Properties( key = JMSPropertiesSetter.JMS_REDELIVERED )
 * <li> JMSProperties		-> Properties( key = propertyName )
 * </lu>
 * Note: JMSCorrelationID is set as a fragment on the URI and can be retrieved like this:
 * correlationURI.getFragment();
 *
 * @author <a href="mailto:daniel.bevenius@redpill.se">Daniel Bevenius</a>
 * @since 4.2
 *
 */
public class DefaultESBPropertiesSetter implements ESBPropertiesSetter
{
	private Logger log = Logger.getLogger( DefaultESBPropertiesSetter.class );
    private Pattern propertiesPattern;
	
	/**
	 * No args constructor.
	 */
	public DefaultESBPropertiesSetter()
	{
	}
	
	public DefaultESBPropertiesSetter(final String propertiesFilter)
	{
	    AssertArgument.isNotNull(propertiesFilter, "propertiesFilter");
	    propertiesPattern = Pattern.compile(propertiesFilter);
	}

	public void setPropertiesFromJMSMessage( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		setCorrelationID( fromJMSMessage, toESBMessage );
		setReplyTo( fromJMSMessage, toESBMessage );
		setExpiration( fromJMSMessage, toESBMessage );
		setRedelivered( fromJMSMessage, toESBMessage );
		setProperties( fromJMSMessage, toESBMessage );
	}

	private void setCorrelationID( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		if ( fromJMSMessage.getJMSCorrelationID()!=null)
		{
			try
			{
				final URI correlationURI = new URI( JMSEpr.JMS_PROTOCOL, "correlationID" , fromJMSMessage.getJMSCorrelationID()  );
				toESBMessage.getHeader().getCall().setRelatesTo( correlationURI );
			}
			catch (URISyntaxException e)
			{
				final String errorMessage = "Could not set ESB setRelatesTo to JMSCorrelationID [ " + fromJMSMessage.getJMSCorrelationID() + "]";
				throw new JMSException( errorMessage );
			}
		}
	}

	private void setReplyTo( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message esbMessage ) throws JMSException
	{
		final Destination destination = fromJMSMessage.getJMSReplyTo();
		if ( destination != null )
		{
			esbMessage.getProperties().setProperty( JMSPropertiesSetter.JMS_REPLY_TO, destination );
		}
	}

	private void setExpiration( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		if( fromJMSMessage.getJMSExpiration() != 0 )
		{
			long ttl = fromJMSMessage.getJMSExpiration() - System.currentTimeMillis();
			if ( ttl < 0 )
				ttl = javax.jms.Message.DEFAULT_TIME_TO_LIVE;
			log.debug( "Setting JMS Expiration : " + ttl );
			toESBMessage.getProperties().setProperty( JMSPropertiesSetter.JMS_EXPIRATION, ttl );
		}
	}
	
	private void setRedelivered( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		final boolean redelivered = fromJMSMessage.getJMSRedelivered();
		toESBMessage.getProperties().setProperty( JMSPropertiesSetter.JMS_REDELIVERED, Boolean.valueOf( redelivered ));
	}

	private void setProperties( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		@SuppressWarnings("unchecked")
		Enumeration<String> properties = fromJMSMessage.getPropertyNames();
		Properties esbProperties = toESBMessage.getProperties();
		if (null != properties)
		{
			while (properties.hasMoreElements())
			{
				final String key = properties.nextElement();
				
				if (filter() && propertiesPattern.matcher(key).matches())
				    continue;
				
				final Object value = fromJMSMessage.getObjectProperty(key);
				if (null != value)
					esbProperties.setProperty(key, value);
			}
	     }
	}
	
	private boolean filter()
	{
	    return propertiesPattern != null;
	}
	

}
