/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.lifecycle;

import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.MessageFlowContext;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.schedule.SchedulerJob;
import org.jboss.soa.esb.schedule.SchedulerJobListener;
import org.jboss.soa.esb.schedule.SchedulingException;


/**
 * This class provides threaded support for a managed instance.
 * 
 * @author kevin
 */
public abstract class AbstractScheduledManagedLifecycle extends AbstractManagedLifecycle
{
    /**
     * The job associated with the scheduler.
     */
    private final SchedulerJob job ;
    /**
     * The logger for this class.
     */
    private static final Logger logger = Logger.getLogger(AbstractScheduledManagedLifecycle.class) ;
    /**
     * Message flow priority.
     */
    private final Integer messageFlowPriority ;
    
    /**
     * Construct the threaded managed lifecycle.
     * @param config The configuration associated with this instance.
     * @throws ConfigurationException for configuration errors during initialisation.
     */
    protected AbstractScheduledManagedLifecycle(final ConfigTree config)
        throws ConfigurationException
    {
        super(config) ;
        messageFlowPriority = MessageFlowContext.parseMessageFlowPriority(config) ;
        final Properties properties = extractProperties(config) ;
        final SchedulerJobListener listener = new SchedulerJobListener() {
            public void onSchedule() throws SchedulingException {
                MessageFlowContext.setMessageFlowPriority(messageFlowPriority) ;
                try {                
                    AbstractScheduledManagedLifecycle.this.onSchedule() ;
                } finally {
                        MessageFlowContext.setMessageFlowPriority(null) ;
                }
            }
        } ;
        final String scheduleIdRef = config.getAttribute(ListenerTagNames.SCHEDULE_ID_REF) ;
        if (scheduleIdRef == null)
        {
            final String intervalVal = config.getAttribute(ListenerTagNames.SCHEDULE_FREQUENCY) ;
            final long interval ;
            if (intervalVal != null)
            {
                interval = parseInt(ListenerTagNames.SCHEDULE_FREQUENCY, intervalVal) ;
            }
            else
            {
                interval = 10 ;
            }
            
            job = SchedulerJob.createIntervalSchedulerJob(null, listener, interval*1000, properties) ;
        }
        else
        {
            final Date scheduleStartDate = getDate(config.getAttribute(ListenerTagNames.SCHEDULE_START_DATE)) ;
            final Date scheduleEndDate = getDate(config.getAttribute(ListenerTagNames.SCHEDULE_END_DATE)) ;
            final String simpleFrequency = config.getAttribute(ListenerTagNames.SCHEDULE_SIMPLE_FREQUENCY) ;
            if (simpleFrequency != null)
            {
                final long frequency = parseLong(ListenerTagNames.SCHEDULE_SIMPLE_FREQUENCY, simpleFrequency) ;
                if (frequency <= 0)
                {
                    throw new ConfigurationException("Invalid " + ListenerTagNames.SCHEDULE_SIMPLE_FREQUENCY + " value, must be greate than zero: " + simpleFrequency) ;
                }
                final String simpleExecCount = config.getAttribute(ListenerTagNames.SCHEDULE_SIMPLE_EXEC) ;
                if (simpleExecCount != null)
                {
                    final int execCount = parseInt(ListenerTagNames.SCHEDULE_SIMPLE_EXEC, simpleExecCount) ;
                    if (execCount < 0)
                    {
                        job = SchedulerJob.createIntervalSchedulerJob(scheduleIdRef, listener, frequency, scheduleStartDate, scheduleEndDate, properties) ;
                    }
                    else if (execCount == 0)
                    {
                        logger.warn("<simple-schedule> '" + scheduleIdRef + "' has an execCount of 0 configured.  This schedule will not fire!");
                        job = null ;
                    }
                    else
                    {
                        job = SchedulerJob.createIntervalSchedulerJob(scheduleIdRef, listener, frequency, execCount, scheduleStartDate, scheduleEndDate, properties) ;
                    }
                }
                else
                {
                    job = SchedulerJob.createIntervalSchedulerJob(scheduleIdRef, listener, frequency, scheduleStartDate, scheduleEndDate, properties) ;
                }
            }
            else
            {
                final String cronExpression = config.getAttribute(ListenerTagNames.SCHEDULE_CRON_EXPRESSION) ;
                if (cronExpression == null)
                {
                    throw new ConfigurationException("Missing cron expression") ;
                }
                try
                {
                    job = SchedulerJob.createCronSchedulerJob(scheduleIdRef, listener, cronExpression, scheduleStartDate, scheduleEndDate, properties) ;
                }
                catch (final ParseException pe)
                {
                    throw new ConfigurationException("Invalid cron expression", pe) ;
                }
            }
        }
    }

    /**
     * Handle the initialisation of the managed instance.
     * 
     * @throws ManagedLifecycleException for errors while initialisation.
     */
    @Override
    protected void doInitialise()
        throws ManagedLifecycleException
    {
    }
    
    /**
     * Handle the start of the managed instance.
     * 
     * @throws ManagedLifecycleException for errors while starting.
     */
    @Override
    protected void doStart()
        throws ManagedLifecycleException
    {
        if (job != null)
        {
            try
            {
                job.start() ;
            }
            catch (final SchedulingException se)
            {
                throw new ManagedLifecycleException("Failed to start the scheduling job", se) ;
            }
        }
    }
    
    /**
     * Execute the scheduled event.
     */
    protected abstract void onSchedule()
        throws SchedulingException ;
    
    /**
     * Handle the stop of the managed instance.
     * 
     * @throws ManagedLifecycleException for errors while stopping.
     */
    @Override
    protected void doStop()
        throws ManagedLifecycleException
    {
        if (job != null)
        {
            try
            {
                job.pause() ;
            }
            catch (final SchedulingException se)
            {
                throw new ManagedLifecycleException("Failed to pause the scheduling job", se) ;
            }
        }
    }
    
    /**
     * Handle the destroy of the managed instance.
     * 
     * @throws ManagedLifecycleException for errors while destroying.
     */
    @Override
    protected final void doDestroy()
        throws ManagedLifecycleException
    {
        try
        {
            doScheduledDestroy() ;
        }
        finally
        {
            if (job != null)
            {
                try
                {
                    job.destroy() ;
                }
                catch (final SchedulingException se)
                {
                    throw new ManagedLifecycleException("Failed to destroy the scheduling job", se) ;
                }
            }
        }
    }
    
    /**
     * Handle the scheduled destroy of the managed instance.
     * 
     * @throws ManagedLifecycleException for errors while destroying.
     */
    protected void doScheduledDestroy()
        throws ManagedLifecycleException
    {
    }
    
    /**
     * Parse the value as an integer.
     * @param propertyName The property name.
     * @param value The string representation.
     * @return the integer value.
     * @throws ConfigurationException For failures in parsing the integer.
     */
    private int parseInt(final String propertyName, final String value)
        throws ConfigurationException
    {
        try
        {
            return Integer.parseInt(value) ;
        }
        catch (final NumberFormatException nfe)
        {
            throw new ConfigurationException("Failed to parse " + propertyName + " as integer: " + value) ;
        }
    }
    
    /**
     * Parse the value as a long.
     * @param propertyName The property name.
     * @param value The string representation.
     * @return the long value.
     * @throws ConfigurationException For failures in parsing the long.
     */
    private long parseLong(final String propertyName, final String value)
        throws ConfigurationException
    {
        try
        {
            return Long.parseLong(value) ;
        }
        catch (final NumberFormatException nfe)
        {
            throw new ConfigurationException("Failed to parse " + propertyName + " as long: " + value) ;
        }
    }
    
    /**
     * Extract any scheduler properties from the configuration.
     * @param config The current configuration.
     * @return The scheduler properties or null if none present.
     * @throws ConfigurationException for errors in configuration
     */
    private Properties extractProperties(final ConfigTree config)
        throws ConfigurationException
    {
        final ConfigTree[] children = config.getChildren(ListenerTagNames.SCHEDULE_PROPERTIES) ;
        final int numChildren = (children == null ? 0 : children.length) ;
        if (numChildren == 0)
        {
            return null ;
        }
        else if (numChildren > 1)
        {
            throw new ConfigurationException("Only one " + ListenerTagNames.SCHEDULE_PROPERTIES + " element allowed within the configuration") ;
        }
        
        final List<KeyValuePair> attributeList = children[0].attributesAsList() ;
        if ((attributeList == null) || attributeList.isEmpty())
        {
            return null ;
        }
        
        final Properties properties = new Properties() ;
        for(KeyValuePair pair: attributeList)
        {
            properties.put(pair.getKey(), pair.getValue()) ;
        }
        return properties ;
    }
    
    /**
     * Create a date object from the time value.
     * @param time The time value.
     * @return The date instance.
     * @throws ConfigurationException for parsing errors.
     */
    private Date getDate(final String time)
        throws ConfigurationException
    {
        if (time == null)
        {
            return null ;
        }
        
        final long timeval ;
        try
        {
            timeval = Long.parseLong(time) ;
        }
        catch (final NumberFormatException nfe)
        {
            throw new ConfigurationException("Could not create date from millisecond value: " + time) ;
        }
        return new Date(timeval) ;
    }
}
