/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import java.io.Serializable;
import java.net.URL;
import java.util.concurrent.TimeUnit;

import javax.crypto.SealedObject;
import javax.security.auth.Subject;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.security.PublicCryptoUtil;
import org.jboss.soa.esb.services.security.SecurityConfig;
import org.jboss.soa.esb.services.security.SecurityConfigUtil;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.SecurityService;
import org.jboss.soa.esb.services.security.TestPrincipal;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.principals.User;
import org.jboss.soa.esb.util.ClassUtil;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * This test demonstrates the functionality of the pipeline interceptors.  
 * It attempts to test that all of the interception points are hit and 
 * properties are set by all of the test interceptors that are configured.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a> 
 */
public class InterceptorProcessingUnitTest extends TestCase
{
    private static final String DOMAIN = "SuccessfulLogin" ;
    private static final String DIFF_DOMAIN = "UnsuccessfulLogin" ;
    
	private String jbossEsbProperties;

    @Override
    protected void setUp() throws Exception
    {
        super.setUp();
        MockActionInfo.reset();
        System.setProperty("javax.xml.registry.ConnectionFactoryClass","org.apache.ws.scout.registry.ConnectionFactoryImpl");

        MockRegistry.install();
        LifecycleResourceManager.getSingleton().associateDeployment(getClass().getCanonicalName()) ;

        jbossEsbProperties = System.getProperty(Environment.PROPERTIES_FILE);
        URL resource = ClassUtil.getResource("security-properties.xml", getClass());
        System.setProperty(Environment.PROPERTIES_FILE, "abs://" + resource.getFile());

        URL loginConfig = ClassUtil.getResource("test_jaas.config", getClass());
        System.setProperty("java.security.auth.login.config", loginConfig.getFile());
    }

    @Override
    protected void tearDown() throws Exception
    {
        LifecycleResourceManager.getSingleton().disassociateDeployment(getClass().getCanonicalName()) ;
        if ( jbossEsbProperties != null )
            System.setProperty(Environment.PROPERTIES_FILE, jbossEsbProperties);
        super.tearDown();
    }

    /*
     * Tests to run
     *  - create a pipeline with a failure, make sure failure is called in correct order and subsequent actions are not called.
     *    Check initialise and destroy called.
     *
     *  Do the above for each type of action.
     */
    public void testInterceptors() throws Exception {
        PropertyManager pm = ModulePropertyManager.getPropertyManager(ModulePropertyManager.INTERCEPTOR_MODULE);
        pm.setProperty(Environment.PROCESSING_PIPELINE_END_INTERCEPTORS, "org.jboss.soa.esb.listeners.message.TestEndPipelineInterceptor");
        pm.setProperty(Environment.PROCESSING_PIPELINE_START_INTERCEPTORS, "org.jboss.soa.esb.listeners.message.TestStartPipelineInterceptor");
        pm.setProperty(Environment.PROCESSING_PIPELINE_FAILURE_INTERCEPTORS, "org.jboss.soa.esb.listeners.message.TestFailPipelineInterceptor");

    	final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4) ;

        Message msg = MessageFactory.getInstance().getMessage();
        final boolean result = pipeline.process(msg);
        assertTrue("Pipeline process failure", result) ;

        assertTrue("true".equals(msg.getProperties().getProperty(
        		Environment.PROCESSING_PIPELINE_END_INTERCEPTORS)));
        String failure = (String) msg.getProperties().getProperty(
            		Environment.PROCESSING_PIPELINE_FAILURE_INTERCEPTORS);
        assertEquals(null,failure);
        assertTrue("true".equals(msg.getProperties().getProperty(
        		Environment.PROCESSING_PIPELINE_START_INTERCEPTORS)));
        assertFalse("true".equals(msg.getProperties().getProperty(
        		Environment.PROCESSING_PIPELINE_INSTANTIATION_INTERCEPTORS)));
        
        pipeline.destroy() ;
    }

    public void testFailureInterceptor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
                        ListenerTagNames.MEP_ONE_WAY) ;

        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessorFailure.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4) ;

        Message msg = MessageFactory.getInstance().getMessage();
        
        final boolean result = pipeline.process(msg) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 2, 0) ;

        final boolean secondResult = pipeline.process(msg) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 0, 2) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 2, 0, 2, 0) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 4, 2, 0) ;

        assertTrue("true".equals(msg.getProperties().getProperty(
        		Environment.PROCESSING_PIPELINE_FAILURE_INTERCEPTORS)));
        
        pipeline.destroy() ;
    }

    public static ConfigTree addAction(final ConfigTree configTree, final String actionName) {
        return addAction(configTree, actionName, null, null, null);
    }

    public static ConfigTree addAction(final ConfigTree configTree, final String actionName,
        final String processOverride, final String successOverride,
        final String exceptionOverride)
    {
        final ConfigTree actionChild = new ConfigTree(ListenerTagNames.ACTION_ELEMENT_TAG, configTree) ;
        actionChild.setAttribute(ListenerTagNames.ACTION_CLASS_TAG, actionName) ;
        if (processOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.PROCESS_METHOD_TAG, processOverride) ;
        }
        if (successOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.NORMAL_COMPLETION_METHOD_TAG, successOverride) ;
        }
        if (exceptionOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.EXCEPTION_METHOD_TAG, exceptionOverride) ;
        }
        
        return actionChild;
    }

    private void checkOrder(final Integer[] list, int ... values)
    {
        final int numValues = (values == null ? 0 : values.length) ;
        final int listLength = (list == null ? 0 : list.length) ;

        assertEquals("Unexpected list/values count", numValues, listLength) ;

        for(int count = 0 ; count < numValues ; count++)
        {
            assertEquals("Unexpected call order at count " + count, values[count], list[count].intValue()) ;
        }
    }

    private ConfigTree addSecurityConfig(
            final ConfigTree parent,
            final String runAs,
            final String callerIdentity,
            final String moduleName,
            final String rolesAllowed)
    {
        final ConfigTree securityElement = new ConfigTree(ListenerTagNames.SECURITY_TAG, parent);
        securityElement.setAttribute(ListenerTagNames.RUN_AS_TAG, runAs);
        securityElement.setAttribute(ListenerTagNames.USE_CALLERS_IDENTIDY_TAG, callerIdentity);
        securityElement.setAttribute(ListenerTagNames.MODULE_NAME_TAG, moduleName);
        if ( rolesAllowed != null )
        {
            securityElement.setAttribute(ListenerTagNames.ROLES_ALLOWED, rolesAllowed);
        }
        return securityElement;
    }

    private ConfigTree addSecurityConfig(
            final ConfigTree parent,
            final String runAs,
            final String callerIdentity,
            final String moduleName,
            final String rolesAllowed,
            final long timeout)
    {
        ConfigTree securityElement = this.addSecurityConfig(parent, runAs, callerIdentity, moduleName, rolesAllowed);
        if (timeout != 0l)
        {
            ConfigTree property = new ConfigTree("property", securityElement);
            property.setAttribute("name", Environment.SECURITY_SERVICE_CONTEXT_TIMEOUT);
            property.setAttribute("value", Long.toString(timeout));
        }
        return securityElement;
    }

}
