/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security;

import java.util.Properties;

import javax.crypto.SealedObject;

import static junit.framework.Assert.assertNotNull;
import static junit.framework.Assert.assertNotSame;
import static junit.framework.Assert.assertNull;
import junit.framework.JUnit4TestAdapter;

import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.message.ByReferenceMessage;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

/**
 * Unit tests for security propagation.
 * 
 * @author <a href="mailto:Kevin.Conner@jboss.com">Kevin Conner</a>
 */
public class SecurityPropagationUnitTest
{
    private Properties props ;

    @Before
    public void setup()
    {
        props = System.getProperties() ;
        final Properties newProps = new Properties(props) ;
        newProps.setProperty(Environment.SECURITY_SERVICE_SEAL_ALGORITHM, "TripleDES") ;
        newProps.setProperty(Environment.SECURITY_SERVICE_SEAL_KEYSIZE, "168") ;
        System.setProperties(newProps) ;
        MockCourierFactory.install() ;
        MockRegistry.install() ;
    }

    @After
    public void tearDown()
    {
        MockRegistry.uninstall() ;
        MockCourierFactory.uninstall() ;
        System.setProperties(props) ;
    }

    @Test
    public void securityContextPropagation()
        throws Exception
    {
        final String category = "test" ;
        final String service = "securityContextPropagation" ;
        final MockCourier mockCourier = new PropagationMockCourier() ;
        MockRegistry.register(category, service, mockCourier) ;
        
        final ServiceInvoker si = new ServiceInvoker(category, service) ;
        final Message message = MessageFactory.getInstance().getMessage() ;
        
        assertNull("securityContext", SecurityContext.getSecurityContext()) ;
        assertNull("message context", message.getContext().getContext(SecurityService.CONTEXT)) ;
        
        final SealedObject securityContext = SecurityContext.encryptContext(new SecurityContext()) ;
        SecurityContext.setSecurityContext(securityContext) ;
        
        assertNotNull("securityContext", SecurityContext.getSecurityContext()) ;
        
        si.deliverAsync(message) ;
        
        assertNotSame("messages", message, mockCourier.message) ;
        assertNull("message context", message.getContext().getContext(SecurityService.CONTEXT)) ;
        assertNotNull("delivered message context", mockCourier.message.getContext().getContext(SecurityService.CONTEXT)) ;
        assertNotNull("securityContext", SecurityContext.getSecurityContext()) ;
        
        mockCourier.reset();
        si.deliverAsync(message) ;
        
        assertNotSame("messages", message, mockCourier.message) ;
        assertNull("message context", message.getContext().getContext(SecurityService.CONTEXT)) ;
        assertNotNull("delivered message context", mockCourier.message.getContext().getContext(SecurityService.CONTEXT)) ;
        assertNotNull("securityContext", SecurityContext.getSecurityContext()) ;
    }

    @Test
    public void authenticationRequestPropagation()
        throws Exception
    {
        final String category = "test" ;
        final String service = "authenticationRequestPropagation" ;
        final MockCourier mockCourier = new PropagationMockCourier() ;
        MockRegistry.register(category, service, mockCourier) ;
        
        final ServiceInvoker si = new ServiceInvoker(category, service) ;
        final Message message = MessageFactory.getInstance().getMessage() ;
        
        assertNull("authenticationRequest", AuthenticationRequestImpl.getEncryptedAuthRequest()) ;
        assertNull("message context", message.getContext().getContext(SecurityService.AUTH_REQUEST)) ;
        
        final byte[] authenticationRequest = new byte[0] ;
        AuthenticationRequestImpl.setEncryptedAuthRequest(authenticationRequest) ;
        
        assertNotNull("authenticationRequest", AuthenticationRequestImpl.getEncryptedAuthRequest()) ;
        
        si.deliverAsync(message) ;
        
        assertNotSame("messages", message, mockCourier.message) ;
        assertNull("message context", message.getContext().getContext(SecurityService.AUTH_REQUEST)) ;
        assertNotNull("delivered message context", mockCourier.message.getContext().getContext(SecurityService.AUTH_REQUEST)) ;
        assertNotNull("authenticationRequest", AuthenticationRequestImpl.getEncryptedAuthRequest()) ;
        
        mockCourier.reset();
        si.deliverAsync(message) ;
        
        assertNotSame("messages", message, mockCourier.message) ;
        assertNull("message context", message.getContext().getContext(SecurityService.AUTH_REQUEST)) ;
        assertNotNull("delivered message context", mockCourier.message.getContext().getContext(SecurityService.AUTH_REQUEST)) ;
        assertNotNull("authenticationRequest", AuthenticationRequestImpl.getEncryptedAuthRequest()) ;
    }

    private static final class PropagationMockCourier extends MockCourier
    {
        PropagationMockCourier()
        {
            super(true) ;
        }
        
        @Override
        public boolean deliver(final Message message)
            throws CourierException, MalformedEPRException
        {
            return super.deliver(((ByReferenceMessage)message).reference());
        }
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(SecurityPropagationUnitTest.class);
    }
}
