/*
 * JBoss, Home of Professinal Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm5.actions;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import javax.naming.InitialContext;
import javax.transaction.TransactionManager;
import javax.transaction.UserTransaction;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;

import org.jboss.soa.esb.services.jbpm5.model.ProcessConstants;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.workitem.esb.ESBActionWorkItemHandler;
import org.jboss.soa.esb.workitem.esb.ESBServiceWorkItemHandler;

import org.drools.KnowledgeBase;
import org.drools.KnowledgeBaseFactory;
import org.drools.base.MapGlobalResolver;
import org.drools.builder.KnowledgeBuilder;
import org.drools.builder.KnowledgeBuilderFactory;
import org.drools.builder.ResourceType;
import org.drools.io.ResourceFactory;

import org.drools.persistence.jpa.JPAKnowledgeService;

import org.drools.runtime.Environment;
import org.drools.runtime.EnvironmentName;
import org.drools.runtime.KnowledgeRuntime;
import org.drools.runtime.KnowledgeSessionConfiguration;
import org.drools.runtime.conf.ClockTypeOption;
import org.drools.runtime.process.WorkItemHandler;

import org.drools.runtime.StatefulKnowledgeSession;

/**
 * The Bpm5Processor processes an action on a single JBPM process.
 *  
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.11
 */
public abstract class AbstractBpm5Action extends AbstractActionPipelineProcessor {
    protected ConfigTree configTree;
    protected Logger logger;
    
    /* Defaults are HornetQ default settings to match the BRMS defaults */
    public static final String DEFAULT_HANDLER_CLASS = "org.jbpm.task.service.hornetq.CommandBasedHornetQWSHumanTaskHandler";
    public static final String DEFAULT_HANDLER_HOST = "127.0.0.1";
    public static final String DEFAULT_HANDLER_PORT = "5446";
    
    private UserTransaction emfUserTransaction;
    
    protected String entityFactoryName;
    
    protected String processDefName;

    /* Handler settings */
    protected String handlerClass;
    protected String handlerHost;
    protected String handlerPort;
    
    protected KnowledgeBuilder kbuilder;
    protected Environment kenvironment;
    protected KnowledgeSessionConfiguration config;
    protected KnowledgeBase kbase;
            
	public AbstractBpm5Action(ConfigTree configTree, Logger logger) throws ConfigurationException
	{
		this.configTree = configTree;
		this.logger = logger;

		this.entityFactoryName = configTree.getAttribute(ProcessConstants.ENTITY_FACTORY_NAME, "jbpm5Persistence.jpa");
		
        this.processDefName = configTree.getRequiredAttribute(ProcessConstants.PROCESS_DEFINITION_NAME);

		this.handlerClass = configTree.getAttribute(ProcessConstants.PROCESS_HANDLER_CLASS, DEFAULT_HANDLER_CLASS);
		this.handlerHost = configTree.getAttribute(ProcessConstants.PROCESS_HANDLER_HOST, DEFAULT_HANDLER_HOST);
		this.handlerPort = configTree.getAttribute(ProcessConstants.PROCESS_HANDLER_PORT, DEFAULT_HANDLER_PORT);
	}
	
	@Override
	public void initialise() throws ActionLifecycleException
	{
        kenvironment = getEnvironment(new HashMap<String, Object>());
        Properties properties = new Properties();
        properties.put("drools.processInstanceManagerFactory","org.jbpm.persistence.processinstance.JPAProcessInstanceManagerFactory");
        properties.put("drools.processSignalManagerFactory","org.jbpm.persistence.processinstance.JPASignalManagerFactory");
        config = KnowledgeBaseFactory.newKnowledgeSessionConfiguration(properties);

        KnowledgeBuilder kbuilder = KnowledgeBuilderFactory.newKnowledgeBuilder();
        kbuilder.add(ResourceFactory.newClassPathResource(processDefName), ResourceType.BPMN2);
        kbase = kbuilder.newKnowledgeBase();
	}
	
	public WorkItemHandler getHandler(StatefulKnowledgeSession session) throws ClassNotFoundException,
		NoSuchMethodException, IllegalAccessException, InstantiationException,
		InvocationTargetException {
		Class handlerKlass = ClassUtil.forName(handlerClass, AbstractBpm5Action.class);
		WorkItemHandler wih = (WorkItemHandler) handlerKlass.getDeclaredConstructor(KnowledgeRuntime.class).newInstance(session);
		return wih;
	}
	
	public void connectHandler(WorkItemHandler wih) 
	    throws ClassNotFoundException, InstantiationException, NoSuchMethodException, IllegalAccessException,
	    InvocationTargetException {
		Class handlerKlass = ClassUtil.forName(handlerClass, AbstractBpm5Action.class);
		Method connectionMethod = handlerKlass.getMethod("setConnection", String.class, Integer.TYPE);
		connectionMethod.invoke(wih, handlerHost, Integer.valueOf(handlerPort));
				
		Method connectMethod = handlerKlass.getMethod("connect");
		connectMethod.invoke(wih);
	}
	
	public StatefulKnowledgeSession getSession(Message message) throws ActionProcessingException {
	    StatefulKnowledgeSession ksession ;
	    WorkItemHandler khandler;
	    
		int sessionId = getSessionId(message);
		if (sessionId >= 0) {
			ksession = JPAKnowledgeService.loadStatefulKnowledgeSession( sessionId, kbase, config, kenvironment );		
		} else {		
			ksession = JPAKnowledgeService.newStatefulKnowledgeSession( kbase, config, kenvironment );
		}
		try {
			khandler = getHandler(ksession);
		} catch (Exception e) {
			throw new ActionProcessingException(e);
		}

        // Register ESB WorkItemHandler as well
		ESBServiceWorkItemHandler esbhandler = new ESBServiceWorkItemHandler();
		ESBActionWorkItemHandler actionhandler = new ESBActionWorkItemHandler();
		ksession.getWorkItemManager().registerWorkItemHandler("JBossESB", esbhandler);
		ksession.getWorkItemManager().registerWorkItemHandler("ESBAction", actionhandler);
		ksession.getWorkItemManager().registerWorkItemHandler("Human Task", khandler);

		try {
			connectHandler(khandler);
		} catch (Exception e) {
			throw new ActionProcessingException(e);
		}
		return ksession;
	}

	protected abstract int getSessionId(Message msg) throws ActionProcessingException ;
	
    /** 
     * Return a knowledge session configuration.
     * @param props
     * @return
     */
    public static KnowledgeSessionConfiguration getSessionConfiguration(Properties props) {
        KnowledgeSessionConfiguration ksess = KnowledgeBaseFactory.newKnowledgeSessionConfiguration(props);
        ksess.setOption(ClockTypeOption.get(org.drools.ClockType.REALTIME_CLOCK.getId()));
        return ksess;
    }
    
    /**
     * Returns an environment with an entity manager factory, transaction manager, and user 
     * transaction configured.
     * @param overrides
     * @return
     */
    public Environment getEnvironment(Map<String, Object> overrides) {
    	Environment env = KnowledgeBaseFactory.newEnvironment();
    	
    	emfUserTransaction= getUserTransaction();
    	EntityManagerFactory emf = Persistence.createEntityManagerFactory(entityFactoryName);
    	if ((entityFactoryName != null) && (!"".equals(entityFactoryName))) {
        	logger.info("Setting entityManager");
            env.set(EnvironmentName.ENTITY_MANAGER_FACTORY, emf);  
            env.set(EnvironmentName.GLOBALS, new MapGlobalResolver());
            env.set(EnvironmentName.TRANSACTION_MANAGER, getTransactionManager());
            env.set(EnvironmentName.TRANSACTION, emfUserTransaction);
    	}
        return env;
    }

    /**
     * Gets the user transaction from JNDI.
     * @return
     */
    public static UserTransaction getUserTransaction()   {     
    	UserTransaction ut  = null;
    	try {
    		InitialContext ic = new InitialContext();     
    		ut  = (UserTransaction) ic.lookup("UserTransaction");
    		if (ut == null) throw new IllegalStateException("JNDI lookup of user transaction failed.");     
    	} catch (Exception e) {
    		e.printStackTrace();
    		throw new IllegalStateException("JNDI lookup of user transaction failed.");
    	}
    	return ut; 
    }
    
    /**
     * Gets the transaction manager from JNDI.
     * @return 
     */
    public static TransactionManager getTransactionManager()   {     
    	TransactionManager tm  = null;
    	try {
    		InitialContext ic = new InitialContext();     
    		tm  = (TransactionManager) ic.lookup("java:/TransactionManager");
    		if (tm == null) throw new IllegalStateException("JNDI lookup of transaction manager failed.");     
    	} catch (Exception e) {
    		e.printStackTrace();
    		throw new IllegalStateException("JNDI lookup of transaction manager failed.");
    	}
    	return tm; 
    }
}
