/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc.as6;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.management.ObjectName;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.listeners.war.Servlet;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.vfs.VirtualFile;

/**
 * Metadata for an ESB deployment.
 * <p/>
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @author <a href="mailto:mageshbk@jboss.com">Magesh Kumar B</a>
 */
public class EsbMetaData implements Serializable
{
    /** Serial version unique identifier. */
    private static final long serialVersionUID = 0L;

    /** The name of the .esb archive. */
    private String archiveName;

    /** The adapter for querying the model */
    private ModelAdapter model;

    /** The virtual file. */
    private final VirtualFile file;
    
    /** The deployments dependencies */
    private final Set<ObjectName> dependencies ;
    
    /** The deployment name. The .esb archive name without the .esb suffix */
    private final String deploymentName;
    
    /** The publishers for this deployment */
    private List<ContractReferencePublisher> publishers = new ArrayList<ContractReferencePublisher>();
    
    /** The servlets for this deployment */
    private List<Servlet> servlets = new ArrayList<Servlet>();
    
    /**
     * Sole constructor.
     *
     * @param file              The virtual file. Can point to a file or a directory which will be searched..
     * @param archiveName       The name of the .esb archive from which this deployment comes from.
     * @param dependencies      Dependencies of this deployment. This might be object names or simply bean names.
     */
    public EsbMetaData(final VirtualFile file, final String archiveName, final String deploymentName, final Set<ObjectName> dependencies)
    {
        AssertArgument.isNotNull(file, "file");
        AssertArgument.isNotNullAndNotEmpty(archiveName, "archiveName");
        AssertArgument.isNotNullAndNotEmpty(deploymentName, "deploymentName");
        
        this.file = file;
        this.archiveName = archiveName;
        this.deploymentName = deploymentName;
        this.dependencies = dependencies;
    }

    /**
     * Gets the name of the archive that this metadata came from.
     *
     * @return String   The name of the archive that this metadata came from.
     */
    public final String getArchiveName()
    {
        return archiveName;
    }

    /**
     * Gets the The virtual file. This is a file path
     * @return String The esb xml configuration.
     */
    public VirtualFile getFile()
    {
        return file;
    }
    
    /**
     * Gets a set of dependencies of this deployment.
     * 
     * @return Set<ObjectName> A set of object names that the deployment represented by this instance depends on.
     */
    public Set<ObjectName> getDependencies()
    {
        return Collections.unmodifiableSet(dependencies);
    }
    
    /**
     * The deployment name is the name of the .esb archive without the .esb suffix.
     * 
     * @return String The name of the deployment. This is the archive name without the .esb suffix.
     */
    public String getDeploymentName()
    {
        return deploymentName;
    }
    
    /**
     * The model adapter for the configuration.
     * 
     * @return ModelAdapter The model adapter for the configuration.
     */
    public ModelAdapter getModel()
    { 
        return model;
    }
    
    /**
     * Set the model adapter for the configuration.
     * 
     * @param model The model adapter for the configuration.
     */
    public void setModel(ModelAdapter model)
    { 
        this.model = model;
    }
    
    /**
     * Sets the publishers for this deployment.
     * 
     * @param publishers The publishers for this deployment.
     */
    public void setPublishers(final List<ContractReferencePublisher> publishers)
    {
        this.publishers = publishers;
    }
    
    /**
     * Gets the publishers for this deployment.
     * 
     * @return List<ContractReferencePublishers> List of publishers for this deployment.
     */
    public List<ContractReferencePublisher> getPublishers()
    {
        return Collections.unmodifiableList(publishers);
    }
    
    /**
     * Sets the servlets for this deployment.
     * 
     * @param servlets The servlets for this deployment.
     */
    public void setServlets(final List<Servlet> servlets)
    {
        this.servlets = servlets;
    }
    
    /**
     * Gets the servlets for this deployment.
     * 
     * @return List<Servlet> List of servlets for this deployment.
     */
    public List<Servlet> getServlets()
    {
        return Collections.unmodifiableList(servlets);
    }

    /**
     * @return String   String representation of this metadata.
     */
    @Override
    public final String toString()
    {
        return String.format("EsbMetaData [archiveName='%s', deploymentName='%s', dependencies='%s']", archiveName, deploymentName, dependencies);
    }
}
